/*
** Random numbers for the C64
**
** This code's most simple approach for 8 bit random numbers:
**
** asl
** adc #num1
** eor #num2
**
** This was put together purely by "blind" experimenting, knowing the always
** good nature of eor, and adding a little variation to it. asl spits out a
** bit for adc, then eor messes the result a little up.
**
** By properly selecting #num1 and #num2, sequences spanning all 256 possible
** bytes can be created. There are exactly 40 of them.
**
** As the experiments show only odd numbers for both acd and eor can generate
** these full - span sequences. For an eor value always even adc values exist:
** they form pairs (adc1 = 256 - adc2), but make different (full span)
** sequences. These are already taken in account in this code.
**
** A nice thing in this code is:
** 1. Only 5 bytes
** 2. It does not contain any branches, so can even be used in raster routines
** 3. Can easily be extended to any length
**
** Here is an example for a 3 byte (so potentially giving a 16 million
** elements long sequence) variant:
**
** lda .b0
** asl
** rol .b1
** rol .b2
** adc #add0
** eor #eor0
** sta .b0
** lda .b1
** adc #add1
** eor #eor1
** sta .b1
** lda .b2
** adc #add2
** eor #eor2
** sta .b2
**
** Of course you can use this for 8 bit output as well with greatly increased
** sequence lengths (As you can see, this can be chained to any depth, the
** only constraint is the computing power required to find suitable add and
** eor values). Note that the last two eor's from the example may be omitted
** if you can find an add/eor immediate pair where the eor value is one byte.
**
**
** This short C language code will generate the add/eor pairs for you, running
** in an UNIX like shell you may terminate it with CTRL+C when you got enough
** of it (pairs or running time). The pairs go to a "random.txt" file.
**
** Use the given #define to set how long sequences you need.
** 8  -> one byte
** 16 -> 2 bytes
** 24 -> 3 bytes
** Note that 3 bytes long setting may take --very-- long to find some pairs.
**
**
** Code by Jubatian
*/


#define SEQ_LEN 16




#include <stdio.h>
#include <stdlib.h>


int main(void)
{
 int i,adc,eor,tot=0;
 int a=0;
 FILE* f;
 
 f=fopen("random.txt","w");
 
 for (eor=1; eor<(1<<SEQ_LEN); eor+=2){
  for (adc=1; adc<(1<<SEQ_LEN)>>1; adc+=2){

   a=0;
   i=0;
   do{
    a<<=1;             /* asl */
    a+=adc+(a>>SEQ_LEN);
    a&=(1<<SEQ_LEN)-1; /* adc */
    a^=eor;            /* eor */
    i++;
   }while(a);

   if (i==(1<<SEQ_LEN)){ /* Good, all elements */
    tot+=2;
    fprintf(f,"adc: %d  \t(0x%X),  \teor: %d  \t(0x%X)\n",   adc,   adc, eor, eor);
    fprintf(f,"adc: %d  \t(0x%X),  \teor: %d  \t(0x%X)\n", i-adc, i-adc, eor, eor);
    fflush(f); /* Make sure it is written */
   }

  }
  printf("%d/%d (Current total: %d)\n",eor,1<<SEQ_LEN,tot);
 }
 printf("\n%d total sequences\n",tot);
 
 fclose(f);
 
 return 0;
}
