//-------------------------------------------------------------------------------------
//
// Copyright 2009 Intel Corporation
// All Rights Reserved
//
// Permission is granted to use, copy, distribute and prepare derivative works of this
// software for any purpose and without fee, provided, that the above copyright notice
// and this statement appear in all copies.  Intel makes no representations about the
// suitability of this software for any purpose.  THIS SOFTWARE IS PROVIDED "AS IS."
// INTEL SPECIFICALLY DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, AND ALL LIABILITY,
// INCLUDING CONSEQUENTIAL AND OTHER INDIRECT DAMAGES, FOR THE USE OF THIS SOFTWARE,
// INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PROPRIETARY RIGHTS, AND INCLUDING THE
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  Intel does not
// assume any responsibility for any errors which may appear in this software nor any
// responsibility to update it.
//

#include "StdAfx.h"
#include "Misc/Defines.h"
#include "Gfx/GfxMath.h"
#include "Sono.h"

#define CHUNK_RIFF	0x46464952
#define CHUNK_LIST	0x5453494c
#define CHUNK_INFO	0x4f464e49
#define CHUNK_DLS	0x20534c44 
#define CHUNK_INAM	0x4d414e49 
#define CHUNK_ICOP	0x504f4349 
#define CHUNK_colh	0x686c6f63 
#define CHUNK_lins	0x736e696c
#define CHUNK_ins	0x20736e69 
#define CHUNK_insh	0x68736e69
#define CHUNK_lrgn	0x6e67726c
#define CHUNK_lart	0x7472616c
#define CHUNK_ptbl	0x6c627470
#define CHUNK_wvpl	0x6c707677
#define CHUNK_wave	0x65766177
#define CHUNK_fmt	0x20746d66
#define CHUNK_wsmp	0x706d7377
#define CHUNK_data	0x61746164
#define CHUNK_rgn	0x206e6772
#define CHUNK_rgnh	0x686e6772
#define CHUNK_wlnk	0x6b6e6c77
		
#define ALLOC_ARRAY(_count,_type) ( (_type*)malloc( (_count)*sizeof(_type) ) )

inline DWORD ReadDWORD(const BYTE* p)
{
	return		p[0]
			+	p[1]*0x00000100
			+	p[2]*0x00010000
			+	p[3]*0x01000000;
}

inline WORD ReadWORD(const BYTE* p)
{
	return		p[0]
			+	p[1]*0x00000100;
}

inline ULONG RoundSize(ULONG a) {return (a+1)&0xFFFFFFFE;}

bool CDLSFile::Parse(const BYTE* pData)
{
	bool		bOk;
	const BYTE*	pEnd;
	DWORD		id;
	DWORD		sz;
	
	if (CHUNK_RIFF != *(DWORD*)pData) return false;
	pData+= 4;
	
	sz = ReadDWORD(pData);	pData+=4;
	pEnd = pData+sz;
	
	if (CHUNK_DLS != *(DWORD*)pData) return false;
	pData+= 4;
	
	while (pData<pEnd)
	{
		id = *(DWORD*)pData;	pData+=4;
		sz = ReadDWORD(pData);	pData+=4;
		switch(id)
		{
			case CHUNK_colh: 
			{
				m_InstrumentCount = ReadDWORD(pData);
				m_Instrument = ALLOC_ARRAY(m_InstrumentCount, sDLSInstrument);
				
			}	break;
			
			case CHUNK_ptbl:
			{
				m_SamplesCount = ReadDWORD(pData+4);
				m_Sample = ALLOC_ARRAY(m_SamplesCount, sDLSSample);
				for(int i=0; i<m_SamplesCount; i++)
				{
					m_Sample[i].m_Offset = ReadDWORD(pData+8+4*i);
				}
			}	break;
			
			case CHUNK_LIST:
			{
				switch (*(DWORD*)pData)
				{
					case CHUNK_lins:
					{
						bOk = Parse_InstrumentList(pData+4, pData+sz);
						IF_FAILED_RETURN(bOk);
					}	break;

					case CHUNK_wvpl:
					{
						bOk = Parse_SampleList(pData+4, pData+sz);
						IF_FAILED_RETURN(bOk);
					}	break;
				}
			}	break;
		}
		
		pData+=RoundSize(sz);
	}
	
	return true;
}

bool CDLSFile::Parse_InstrumentList(const BYTE* pData, const BYTE* pEnd)
{
	bool	bOk;
	DWORD	id;
	DWORD	sz;
	ULONG	iInstrument;

	iInstrument = 0;
	while (pData<pEnd)
	{
		id = *(DWORD*)pData;	pData+=4;
		sz = ReadDWORD(pData);	pData+=4;
	
		switch(id)
		{
			case CHUNK_LIST:
			{
				switch (*(DWORD*)pData)
				{
					case CHUNK_ins:
					{
						bOk = m_Instrument[iInstrument++].Parse(pData+4, pData+sz);
						IF_FAILED_RETURN(bOk);
					}	break;
				}
			}
		}
		
		pData+=RoundSize(sz);
	}
	
	return true;
}

bool CDLSFile::Parse_SampleList(const BYTE* pData, const BYTE* pEnd)
{
	bool	bOk;
	DWORD	id;
	DWORD	sz;
	ULONG	iSample;

	iSample = 0;
	while (pData<pEnd)
	{
		id = *(DWORD*)pData;	pData+=4;
		sz = ReadDWORD(pData);	pData+=4;
	
		switch(id)
		{
			case CHUNK_LIST:
			{
				switch (*(DWORD*)pData)
				{
					case CHUNK_wave:
					{
						bOk = m_Sample[iSample++].Parse(pData+4, pData+sz);
						IF_FAILED_RETURN(bOk);
					}	break;
				}
			}
		}
		
		pData+=RoundSize(sz);
	}
	
	return true;
}

void sDLSInstrument::Free()
{
	if (m_Region)
	{
		free(m_Region);
		m_Region = NULL;
	}
}

bool sDLSInstrument::Parse(const BYTE* pData, const BYTE* pEnd)
{
	bool	bOk;
	DWORD	id;
	DWORD	sz;

	m_Region = NULL;

	while (pData<pEnd)
	{
		id = *(DWORD*)pData;	pData+=4;
		sz = ReadDWORD(pData);	pData+=4;
	
		switch(id)
		{
			case CHUNK_insh:
			{
				m_RegionCount	= ReadDWORD(pData);
				m_Region = ALLOC_ARRAY(m_RegionCount, sDLSRegion);
				
				m_Bank			= ReadDWORD(pData+4);
				m_Id			= 1+ReadDWORD(pData+8);
				
				m_bDrum			= (m_Bank&0x80000000)!=0;
				m_Bank &= ~0x80000000;
			}	break;

			case CHUNK_LIST:
			{
				switch (*(DWORD*)pData)
				{
					case CHUNK_lrgn:
					{
						bOk = Parse_RegionList(pData+4, pData+sz);
						IF_FAILED_RETURN(bOk);
					}	break;
					
					case CHUNK_lart:
					{
					}	break;
					
					case CHUNK_INFO:
					{
						bOk = Parse_Info(pData+4, pData+sz);
						IF_FAILED_RETURN(bOk);
					}	break;
				}
			}
		}
		
		pData+=RoundSize(sz);
	}
	
	// TRACE("%s %3d 0x%04x\n", m_Name, m_Id, m_Bank);
	return true;
}

bool sDLSInstrument::Parse_RegionList(const BYTE* pData, const BYTE* pEnd)
{
	bool	bOk;
	DWORD	id;
	DWORD	sz;
	ULONG	iRegion;

	iRegion = 0;
	while (pData<pEnd)
	{
		id = *(DWORD*)pData;	pData+=4;
		sz = ReadDWORD(pData);	pData+=4;
	
		switch(id)
		{
			case CHUNK_LIST:
			{
				switch (*(DWORD*)pData)
				{
					case CHUNK_rgn:
					{
						bOk = m_Region[iRegion++].Parse(pData+4, pData+sz);
						IF_FAILED_RETURN(bOk);
					}	break;
				}
			}
		}
		
		pData+=RoundSize(sz);
	}
	
	return true;
}

bool sDLSInstrument::Parse_Info(const BYTE* pData, const BYTE* pEnd)
{
	DWORD	id;
	DWORD	sz;

	while (pData<pEnd)
	{
		id = *(DWORD*)pData;	pData+=4;
		sz = ReadDWORD(pData);	pData+=4;
	
		switch(id)
		{
			case CHUNK_INAM:
			{
				m_Name = (const char*)pData;
			}	break;

			case CHUNK_ICOP:
			{
				m_Copyright = (const char*)pData;
			}	break;
		}
		
		pData+=RoundSize(sz);
	}
	
	return true;
}

bool sDLSRegion::Parse(const BYTE* pData, const BYTE* pEnd)
{
	DWORD	id;
	DWORD	sz;

	while (pData<pEnd)
	{
		id = *(DWORD*)pData;	pData+=4;
		sz = ReadDWORD(pData);	pData+=4;
	
		switch(id)
		{
			case CHUNK_rgnh:
			{
				m_KeyFrom = (BYTE) ReadWORD(pData   );
				m_KeyTo	  = (BYTE) ReadWORD(pData+ 2);
				/* m_VelLow = ReadWORD(pData+ 4); */ 
				/* m_VelHi  = ReadWORD(pData+ 6); */ 
				/* m_Options= ReadWORD(pData+ 8); */ 
				m_KeyGroup= ReadWORD(pData+10);
			}	break;

			case CHUNK_wsmp:
			{
				id = 0;
			}	break;
			
			case CHUNK_wlnk:
			{
				m_iSample = ReadDWORD(pData+8);
			}	break;
		}
		
		pData+=RoundSize(sz);
	}
	
	return true;
}

bool sDLSSample::Parse(const BYTE* pData, const BYTE* pEnd)
{
	DWORD	id;
	DWORD	sz;

	USHORT	UnityNote;
	LONG	Attenuation;


	m_FormatTag			= 0;
	m_Channels			= 0;
	m_SamplesPerSec		= 0;
	m_AvgBytesPerSec	= 0;
	m_BitsPerSample		= 0;

	UnityNote			= 0;
	m_FineTune			= 0;
	Attenuation			= 0;
	m_Options			= 0;
	m_SampleLoops		= 0;
	
	m_LoopStart			= 0;
	m_LoopLength		= 0;

	m_pData				= NULL;

	while (pData<pEnd)
	{
		id = *(DWORD*)pData;	pData+=4;
		sz = ReadDWORD(pData);	pData+=4;
	
		switch(id)
		{
			case CHUNK_fmt:
			{
				m_FormatTag			= ReadWORD( pData   ); ASSERT(1==m_FormatTag);
				m_Channels			= ReadWORD( pData+ 2);
				m_SamplesPerSec		= ReadDWORD(pData+ 4);
				m_AvgBytesPerSec	= ReadDWORD(pData+ 8);
				m_BitsPerSample		= ReadWORD( pData+14);
			}	break;
			
			case CHUNK_wsmp:
			{
				/* cbSize:ULONG */ 
				UnityNote		= ReadWORD( pData+ 4);
				m_FineTune		= ReadWORD( pData+ 6);
				Attenuation		= ReadDWORD(pData+ 8);
				m_Options		= ReadDWORD(pData+12);
				m_SampleLoops	= ReadDWORD(pData+16);
				
				if (m_SampleLoops)
				{
					/* cbSize:ULONG */ 
					/* LoopType		= ReadDWORD(pData+24); */ 
					m_LoopStart		= ReadDWORD(pData+28);
					m_LoopLength	= ReadDWORD(pData+32);
				}
			}	break;
			
			case CHUNK_data:
			{
				m_pData = (const SHORT*)pData;
				m_DataSize = sz;
			}	break;
		}
		
		pData+=RoundSize(sz);
	}
	
	/* compute a few values */ 
	m_UnityFreq = MIDIKeyToFrequency(UnityNote);
	m_Scale		= Pow(10.0f, -(1-Attenuation)/65536.0f/200.0f) / 32767.0f;
	
	return true;
}
