//-------------------------------------------------------------------------------------
//
// Copyright 2009 Intel Corporation
// All Rights Reserved
//
// Permission is granted to use, copy, distribute and prepare derivative works of this
// software for any purpose and without fee, provided, that the above copyright notice
// and this statement appear in all copies.  Intel makes no representations about the
// suitability of this software for any purpose.  THIS SOFTWARE IS PROVIDED "AS IS."
// INTEL SPECIFICALLY DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, AND ALL LIABILITY,
// INCLUDING CONSEQUENTIAL AND OTHER INDIRECT DAMAGES, FOR THE USE OF THIS SOFTWARE,
// INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PROPRIETARY RIGHTS, AND INCLUDING THE
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  Intel does not
// assume any responsibility for any errors which may appear in this software nor any
// responsibility to update it.
//
#include "StdAfx.h"
#include "Game.h"

#define BLIP_DURATION	1.0f
#define BLIP_PERIOD		4.0f
#define BLIP_MINALPHA	0.1f

bool SetupNulsteinSong(CSono* pSono);

//________________________________________________________________________________
bool CEntityMusic::Update(CFrame* pFrame, float dt)
{
	if (dt>0.0666f)
		dt=0.0666f;
		
	pFrame->m_pSono->Update(dt);
	return true;
}

//________________________________________________________________________________ 
bool CEntityGameCamera::Update(CFrame* pFrame, float dt)
{
	const float R = 1.5f*CUBES_PER_AXIS;
	float a,c,s,r;

	m_t+=dt;

	/* update distance */ 
	if (m_t>=4)
	{
		r = R;
	}
	else
	{
		a = 1.0f-m_t/4; a =  a=1.0f-a*a; a*=4;
		r = max(96.0f-a*24.0f, R);
	}

	/* update rotation */ 
	a = 3.141592654f * m_t/32;	
	SinCos(a, &c,&s);

	m_Camera.m_Position.Set(-r*c, -r*s, 0.0f);	

	/* parent class does the matrixes updating */ 
	return CEntityCamera::Update(pFrame, dt);
}

//________________________________________________________________________________
CEntityDancingSimpleMesh::CEntityDancingSimpleMesh(CEntitySlot* pSlot, int iChannel)
:	CEntitySimpleMesh(pSlot)
{
	m_iChannel		= iChannel;
	m_TrackTicks	= 0;
	m_iTrackItem	= 0;
	m_NotesOn		= 0;
}

bool CEntityDancingSimpleMesh::Update(CFrame* pFrame, float dt)
{
	CSonoChannel*	pChannel = &pFrame->m_pSono->m_Channel[m_iChannel];
	int				i;
	bool			bOk;
	int				NotesStarted = 0;
	int				NotesStopped = 0;
	
	bOk = CEntitySimpleMesh::Update(pFrame, dt);
	if (!bOk) return false;
	
	/* update notes state */ 
	m_TrackTicks += dt * pFrame->m_pSono->m_TicksPerSecond;
	
	for(i=m_iTrackItem; i<pChannel->m_TrackItemCount; i++)
	{
		if (m_TrackTicks < pChannel->m_pTrack[i].dt)
			break;
			
		m_TrackTicks -= pChannel->m_pTrack[i].dt;
		if (pChannel->m_pTrack[i].Velocity)
			 m_NotesOn++;
		else m_NotesOn--;
	}
	m_iTrackItem = i;
	
	/* Change scale based on note count */ 
	float Scl;
	float k; /* blend factor for one frame (assuming 1/60th sec) */ 

	if (pChannel->m_bDrums)
	{
		 Scl = 1.0f - m_NotesOn*0.125f;
		 k   = 0.750f;
	}
	else 
	{	Scl = 1.0f + m_NotesOn*0.050f;
		k   = 0.500f;
	}
	
	k   = Pow(k, dt*60);
	m_Transform.Scl = k      * Scl
					+(1.0f-k)* m_Transform.Scl;
	
	/* Update alpha based on cycle */ 
	m_BlipTime += dt/pFrame->m_pSono->m_Tempo*1000.0f;
	if (m_BlipTime>BLIP_PERIOD)
		m_BlipTime-=BLIP_PERIOD;
		
	if (m_BlipTime<BLIP_DURATION)
		 m_MeshInstance.m_Tint.w = BLIP_MINALPHA + (1.0f-BLIP_MINALPHA)*fabsf(Cos( 3.141592654f * m_BlipTime/BLIP_DURATION) );
	else m_MeshInstance.m_Tint.w = 1.0f;
	return true;
}


//________________________________________________________________________________
CEntityOrbiter::CEntityOrbiter(CEntitySlot* pSlot, CEntitySimpleMesh* pParent)
:	CEntitySimpleMesh(pSlot)
{
	m_bChasing		= false;
	m_NextChase		= 5.0f;
}

bool CEntityOrbiter::PreUpdate(CFrame* pFrame, float dt)
{
	bool				bOk;
	CGame*				pGame;
	CEntitySimpleMesh*	pParent;
	
	/* see if we want to change parent */ 
	m_NextChase -= dt;
	if (m_NextChase<0)
	{
		m_iParent = (m_iParent+1)%CUBES_TOTAL;
		m_NextChase += 5.0f;
		m_bChasing = true;
	}
	
	/* find parent */ 	
	pGame	= GetGame(pFrame);
	pParent = pGame->m_Cube[m_iParent];
	
	/* We'll need our parent's position to lock onto it */ 
	bOk = UpdateDependsOn(pParent);
	IF_FAILED_RETURN(bOk);
	
	return true;
}

bool CEntityOrbiter::Update(CFrame* pFrame, float dt)
{
	bool				bOk;
	CGame*				pGame;
	CEntitySimpleMesh*	pParent;
	CMatrix				ParentToWorld;
	CVector3			Velocity;
	CVector3			Target;
	
	pGame	= GetGame(pFrame);
	pParent = pGame->m_Cube[m_iParent];

	/* update velocity */ 
	Velocity	= (m_Transform.Pos - m_LastPos) * (1.0f/m_LastDt);
	m_LastPos	= m_Transform.Pos;
	m_LastDt	= dt;
	
	/* compute target position */ 
	ParentToWorld.Set(pParent->m_Transform);
	MtxMulPoint(&Target, m_Target, ParentToWorld);
	
	/* update attached/chasing state */ 
	if (m_bChasing)
	{
		bOk = UpdateChase(Target, Velocity, dt);
		IF_FAILED_RETURN(bOk);
	}

	/* if attached, lock position to parent */ 	
	if (!m_bChasing)
	{
		m_Transform.Pos = Target;
	}
	
	/* Alpha */ 
	m_MeshInstance.m_Tint.w = pParent->m_MeshInstance.m_Tint.w;

	return true;
}

bool CEntityOrbiter::UpdateChase(const CVector3 &Target, const CVector3 &Velocity, float dt)
{
	CVector3	I;
	CVector3	V;
	CVector3	F;
	float		d;
	float		k;
	float		inertia;

	inertia = sqrtf(SquareNorm( Velocity*dt ));
	
	/* how much do we need to move? */ 
	V   = Target - m_Transform.Pos;
	d   = sqrtf(SquareNorm(V));
	if (d<=0.1f)
	{	/* snap */ 
		m_bChasing = false;
		return true;
	}
	
	/* Compute pseudo-Force */ 
	k= inertia/d;
	if (k>1.2f)
	{
		F = V* (inertia*0.75f /d);
	}
	else if (k<0.8f)
	{
		F = V* (inertia*1.1f /d);
	}
	else
	{
		F = V* (inertia /d);
	}
	
	/* Move */ 
	m_Transform.Pos += F;
	
	return true;
}



//________________________________________________________________________________
CGame::CGame()
{
}

bool CGame::Init(CFrame* pFrame)
{
	bool				bOk;
	CTransform			T;
	
	/* Load music */ 
	bOk = SetupNulsteinSong(pFrame->m_pSono);
	
	pFrame->m_pSono->Update(4.0f);
	
	m_pMusic = new CEntityMusic( pFrame->AllocEntity() );
	
	/* create geometry */ 
	bOk = m_MeshCube.Create(pFrame->m_pGfx, 0.333f, 1.0f, 15);
	
	/* create instances */ 
	for(int iInstance=0; iInstance<CUBES_TOTAL; iInstance++)
	{
		CEntityDancingSimpleMesh*	pCubeEntity;
		CEntityOrbiter*				pOrbiter;
		
		int iX =  iInstance % CUBES_PER_AXIS;
		int iY = (iInstance/CUBES_PER_AXIS) % CUBES_PER_AXIS;
		int iZ =  iInstance/(CUBES_PER_AXIS*CUBES_PER_AXIS);
		int iChannel = iInstance%7;
		
		pCubeEntity = new CEntityDancingSimpleMesh( pFrame->AllocEntity(), iChannel );
		
		T.Pos.Set(-CUBES_OFFSET+2*iX, -CUBES_OFFSET+2.0f*iY, -CUBES_OFFSET+2.0f*iZ);
		T.Rot.Angle = 3.141592654f / 6.0f;
		T.Rot.Axis.Set(0.0f, 1.0f, 0.0f);
		T.Scl = 1.0f;
		
		pCubeEntity->Init(m_MeshCube, T);
		pCubeEntity->m_RotationSpeed = 2.0f*3.141592654f * (1000.0f / pFrame->m_pSono->m_Tempo);
		if (iX*iX+iY*iY+iZ*iZ)
			 pCubeEntity->m_Transform.Rot.Axis = Normalize( CVector3(-CUBES_OFFSET+2*iX, 2.0f*iY, -CUBES_OFFSET+2.0f*iZ) );
		else pCubeEntity->m_Transform.Rot.Axis.Set(0.0f, 0.0f, 1.0f);
		
		switch (iChannel)
		{	
			case 0:		pCubeEntity->m_MeshInstance.m_Tint.Set( CRGBA(0x5B, 0x59, 0x39) ); break;
			case 1:		pCubeEntity->m_MeshInstance.m_Tint.Set( CRGBA(0xCB, 0xD2, 0xBF ) ); break;
			case 2:		pCubeEntity->m_MeshInstance.m_Tint.Set( CRGBA(0xEC, 0xFF, 0xFF ) ); break;
			case 3:		pCubeEntity->m_MeshInstance.m_Tint.Set( CRGBA(0xCD, 0xF2, 0xF5 ) ); break;
			case 4:		pCubeEntity->m_MeshInstance.m_Tint.Set( CRGBA(0x44, 0x44, 0x44) ); break;
			case 5:		pCubeEntity->m_MeshInstance.m_Tint.Set( CRGBA(0xA6, 0xA8, 0x99) ); break;
			case 6:		pCubeEntity->m_MeshInstance.m_Tint.Set( CRGBA(0xFF, 0xFF, 0xFF) ); break;
			
			default:	pCubeEntity->m_MeshInstance.m_Tint.Set( CVector4(1.0f, 1.0f, 1.0f, 1.0f));
		}
		
		pCubeEntity->m_BlipTime = BLIP_PERIOD * (iInstance%5)/5.0f;
		
		m_Cube[iInstance] = pCubeEntity;
		
		/*  */ 
		pOrbiter = new CEntityOrbiter( pFrame->AllocEntity(), pCubeEntity);
		
		T.Pos.Set(0.0f,0.0f,0.0f);
		T.Rot.Angle = 0.0f;
		T.Rot.Axis.Set(0.0f, 1.0f, 0.0f);
		T.Scl = 0.25f;

		pOrbiter->Init(m_MeshCube, T);
		pOrbiter->m_Target.Set(0.5f, 0.5f, 0.5f);
		pOrbiter->m_iParent = iInstance;
		
		pOrbiter->m_MeshInstance.m_Tint = pCubeEntity->m_MeshInstance.m_Tint;
	}
	
	/* create views */ 
	{
		CEntityGameCamera*	pEntityCamera;
		CCamera*			pCamera;
		
		pEntityCamera = new CEntityGameCamera( pFrame->AllocEntity() );
		
		pCamera = &pEntityCamera->m_Camera;
		pCamera->m_Position.Set(	0.0f, -CUBES_PER_AXIS*0.666f, 2.0f);
		pCamera->m_Target.Set(		0.0f,  0.0f, 0.0f);
		pCamera->m_Near			=	0.001f;
		pCamera->m_Far			= 100.000f;
		
		pFrame->m_View[0].m_pCamera = pCamera;
		pFrame->m_View[0].m_Viewport.m_Near = 0.0f;
		pFrame->m_View[0].m_Viewport.m_Far  = 1.0f;
		pFrame->m_View[0].m_Viewport.m_Rect.left   = 0;
		pFrame->m_View[0].m_Viewport.m_Rect.top    = 0;
		pFrame->m_View[0].m_Viewport.m_Rect.right  = 640;
		pFrame->m_View[0].m_Viewport.m_Rect.bottom = 480;
		
		pFrame->m_ViewCount = 1;
	}
	
	// Light
	pFrame->m_Light[0].Pos.Set(	-1.0f*CUBES_PER_AXIS, 
								-2.0f*CUBES_PER_AXIS, 
								 1.0f*CUBES_PER_AXIS);
	pFrame->m_Light[0].Col.Set( 1.0f,  1.0f, 1.0f);
	
	pFrame->m_LightCount = 1;

	return true;
}