//-------------------------------------------------------------------------------------
//
// Copyright 2009 Intel Corporation
// All Rights Reserved
//
// Permission is granted to use, copy, distribute and prepare derivative works of this
// software for any purpose and without fee, provided, that the above copyright notice
// and this statement appear in all copies.  Intel makes no representations about the
// suitability of this software for any purpose.  THIS SOFTWARE IS PROVIDED "AS IS."
// INTEL SPECIFICALLY DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, AND ALL LIABILITY,
// INCLUDING CONSEQUENTIAL AND OTHER INDIRECT DAMAGES, FOR THE USE OF THIS SOFTWARE,
// INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PROPRIETARY RIGHTS, AND INCLUDING THE
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  Intel does not
// assume any responsibility for any errors which may appear in this software nor any
// responsibility to update it.
//

#include "StdAfx.h"
#include "MeshCube.h"

inline int square(int a) {return a*a;}
static const float kLengthToRadius = 0.8660254f; /* keeps cube corners at same position, whatever spherity */ 
static const float N[] = {	0.0f,0.0f,1.0f, 
							0.0f,1.0f,0.0f, 
							1.0f,0.0f,0.0f, 
							0.0f,0.0f,-1.0f, 
							0.0f,-1.0f,0.0f, 
							-1.0f,0.0f,0.0f, 
							0.0f,0.0f,1.0f, 
							0.0f,1.0f,0.0f	};

//________________________________________________________________________________
CMeshCube::CMeshCube()
{
}

CMeshCube::~CMeshCube()
{
}

//________________________________________________________________________________
bool CMeshCube::Create(CGfx* pGfx, float Spherity /* cube=0.0f, sphere=1.0f */, float Length /* 1.0f */, int SubdivisionCount /* 0 */ )
{

	bool		bOk;	
	CVertexPN*	V;
	WORD*		I;
	int			VertexPerFace;
	int			TrianglePerFace;
	float		kSub;
	
	TrianglePerFace = square(1+SubdivisionCount)*2;
	VertexPerFace	= square(2+SubdivisionCount);

	/* reserve memory */ 
	bOk = Allocate(pGfx, VertexPerFace*6, TrianglePerFace*6, sizeof(CVertexPN));
	IF_FAILED_RETURN(bOk);
	
	/* lock buffers */ 
	bOk = LockVertices((VOID**)&V, 0);
	IF_FAILED_RETURN(bOk);
	
	bOk = LockIndices(&I, 0);
	IF_FAILED_RETURN(bOk);
	
	/* generate cube */ 
	kSub = 1.0f / float(SubdivisionCount+1);
	for(int iFace=0; iFace<6; iFace++)
	{
		float Nx,Ny,Nz;
		float Vx,Vy,Vz;
		float Hx,Hy,Hz;
		
		Nx = N[iFace*3 + 0];
		Ny = N[iFace*3 + 1];
		Nz = N[iFace*3 + 2];
		
		Hx = N[iFace*3 + 3];
		Hy = N[iFace*3 + 4];
		Hz = N[iFace*3 + 5];

		Vx = Ny*Hz + Nz*Hy;
		Vy = Nz*Hx + Nx*Hz;
		Vz = Nx*Hy + Ny*Hx;

		
		/* generate vertices */ 
		for(int iLine=0; iLine<2+SubdivisionCount; iLine++)
		{
			float		kLine = iLine*kSub - 0.5f;
			CVector3	M;
			
			for(int iColumn=0; iColumn<2+SubdivisionCount; iColumn++)
			{
				float kColumn = iColumn*kSub - 0.5f;
				
				M.x  = (Nx*0.5f+ kLine*Vx + kColumn*Hx);
				M.y  = (Ny*0.5f+ kLine*Vy + kColumn*Hy);
				M.z  = (Nz*0.5f+ kLine*Vz + kColumn*Hz);

				if (Spherity==0.0f)
				{	/* Cube */ 
					V->x  = M.x * Length;
					V->y  = M.y * Length;
					V->z  = M.z * Length;

					V->nx = Nx;
					V->ny = Ny;
					V->nz = Nz;
				}
				else if (Spherity==1.0f)
				{	/* sphere */ 
					M = Normalize(M);
					
					V->x  = M.x * Length *kLengthToRadius;
					V->y  = M.y * Length *kLengthToRadius;
					V->z  = M.z * Length *kLengthToRadius;

					V->nx = M.x;
					V->ny = M.y;
					V->nz = M.z;
				}
				else
				{	/* hybrid */ 
					CVector3	M1;

					M1 = Normalize(M);
					
					V->x  = ( M.x*(1.0f-Spherity) + M1.x*Spherity*kLengthToRadius ) * Length;
					V->y  = ( M.y*(1.0f-Spherity) + M1.y*Spherity*kLengthToRadius ) * Length;
					V->z  = ( M.z*(1.0f-Spherity) + M1.z*Spherity*kLengthToRadius ) * Length;

					M.x = ( Nx*(1.0f-Spherity) + M1.x*Spherity );
					M.y = ( Ny*(1.0f-Spherity) + M1.y*Spherity );
					M.z = ( Nz*(1.0f-Spherity) + M1.z*Spherity );
					M = Normalize(M);
					
					V->nx = M.x;
					V->ny = M.y;
					V->nz = M.z;
				}
				
				V++;
			}
		}
		
		/* generate triangles */ 
		for(int iLine=0; iLine<1+SubdivisionCount; iLine++)
		{
			int i0;
			
			i0 = VertexPerFace*iFace + iLine*(SubdivisionCount+2);
			
			for(int iColumn=0; iColumn<1+SubdivisionCount; iColumn++)
			{
				int j0;
				
				j0 = i0+iColumn;

				*I++ = j0 + 0;
				*I++ = j0 + (SubdivisionCount+2);
				*I++ = j0 + 1;
				
				*I++ = j0 + 1; 
				*I++ = j0 + (SubdivisionCount+2); 
				*I++ = j0 + (SubdivisionCount+2)+1; 
			}
		}
	}
	
	/* done */ 
	bOk = UnlockIndices();
	bOk = UnlockVertices();
	
	bOk = pGfx->RegisterMesh(this);
	IF_FAILED_RETURN(bOk);
	
	return true;
}

bool CMeshCube::CreateSphere(CGfx* pGfx, float Radius /* 1.0f */, int SubdivisionCount /* 0 */ )
{
	return Create(pGfx, 1.0f, Radius / kLengthToRadius, SubdivisionCount);
}