//-------------------------------------------------------------------------------------
//
// Copyright 2009 Intel Corporation
// All Rights Reserved
//
// Permission is granted to use, copy, distribute and prepare derivative works of this
// software for any purpose and without fee, provided, that the above copyright notice
// and this statement appear in all copies.  Intel makes no representations about the
// suitability of this software for any purpose.  THIS SOFTWARE IS PROVIDED "AS IS."
// INTEL SPECIFICALLY DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, AND ALL LIABILITY,
// INCLUDING CONSEQUENTIAL AND OTHER INDIRECT DAMAGES, FOR THE USE OF THIS SOFTWARE,
// INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PROPRIETARY RIGHTS, AND INCLUDING THE
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  Intel does not
// assume any responsibility for any errors which may appear in this software nor any
// responsibility to update it.
//

#include "StdAfx.h"
#include "Misc/Defines.h"
#include "Gfx/GfxMath.h"
#include "Sono.h"
#include "ParallelFor.h"

#define NULSTEINSONO_CHANNELS	2
#define NULSTEINSONO_RATE		44100
#define NULSTEINSONO_BITS		16

static const WAVEFORMATEX gWavFormat= 
{
    /* Format		: */ WAVE_FORMAT_PCM,
    /* Channels		: */ NULSTEINSONO_CHANNELS,
    /* Samples/s	: */ NULSTEINSONO_RATE,
    /* bytes/s		: */ NULSTEINSONO_CHANNELS * NULSTEINSONO_RATE * (NULSTEINSONO_BITS/8),
    /* Align		: */ 4,
    /* bits/sample	: */ NULSTEINSONO_BITS,
    /* ExtraSize	: */ 0 
};

float MIDIKeyToFrequency(int iKey) 
{ 
	return 6.875f * Pow(2.0f, (iKey+3)/12.0f); 
}

//________________________________________________________________________________
CSono::CSono()
{
	m_hWaveOut = NULL;
}

bool CSono::Init(int BufferPlayTime)
{
	MMRESULT res;
	
	ASSERT(!m_hWaveOut);

	m_Time			= 0.0f;
	m_Duration		= 0;
	m_LoopStartTick	= 0;
	
	/* allocate memory */ 
	m_BufferSize	= BufferPlayTime*gWavFormat.nAvgBytesPerSec;
	m_pBuffer		= (SHORT*)malloc( m_BufferSize );
	if (!m_pBuffer) return false;
	
	ZeroMemory(m_pBuffer, m_BufferSize);
	m_pWriteBuffer  = m_pBuffer;
	m_pEndOfBuffer	= m_pBuffer+m_BufferSize/sizeof(SHORT);
	
	/* compute keyboard key's frequencies */ 
	for(int i=0; i<127; i++)
	{
		m_KeyFrequency[i] = MIDIKeyToFrequency(i);
	}
	
	/* Load data */ 
	m_DLS.Open();
	InitDrumKeys();
	
	/* setup wave out */ 
	ZeroMemory(&m_WaveHeader, sizeof(m_WaveHeader));
    m_WaveHeader.dwBufferLength	= DWORD( m_BufferSize );
    m_WaveHeader.lpData			= (char*)m_pBuffer;
    m_WaveHeader.dwFlags		= WHDR_BEGINLOOP|WHDR_ENDLOOP;
    m_WaveHeader.dwLoops		= 0xFFFFFFFF;

    res = waveOutOpen( &m_hWaveOut, WAVE_MAPPER, &gWavFormat, 0, 0, 0);
    IF_FAILED_RETURN(res);

    res = waveOutPrepareHeader( m_hWaveOut, &m_WaveHeader, sizeof(WAVEHDR) );
    IF_FAILED_RETURN(res);

    return true;
}

bool CSono::StartPlaying()
{
	MMRESULT res;
    
    res = waveOutWrite( m_hWaveOut, &m_WaveHeader, sizeof(WAVEHDR) );
    IF_FAILED_RETURN(res);
    
    return true;
}

bool CSono::InitDrumKey(int iKey, int iInstrument, int iNote)
{
	m_DrumKey[iKey].m_Instrument	= m_DLS.FindInstrument(iInstrument);
	m_DrumKey[iKey].m_Note			= iNote;
	return true;
}

bool CSono::InitDrumKeys()
{
	// InitDrumKey(35, /* Bass Drum 2 */		118);
	
	InitDrumKey(36, /* Bass Drum 1 */		118,48); /**/ 
	
	// InitDrumKey(37, /* Side Stick */		116);
	
	InitDrumKey(38, /* Snare Drum 1 */		 12); /**/ 
	
	// InitDrumKey(39, /* Hand Clap */			127);
	
	InitDrumKey(40, /* Snare Drum 2 */		 14); /**/ 
	
	// InitDrumKey(41, /* Low Tom 2 */			118);
	// InitDrumKey(42, /* Closed Hi-hat */		128);
	// InitDrumKey(43, /* Low Tom 1 */			128);
	
	InitDrumKey(44, /* Pedal Hi-hat */		116); /**/  
	
	// InitDrumKey(45, /* Mid Tom 2 */			128);
	// InitDrumKey(46, /* Open Hi-hat */		128);
	// InitDrumKey(47, /* Mid Tom 1 */			128);
	// InitDrumKey(48, /* High Tom 2 */		128);
	// InitDrumKey(49, /* Crash Cymbal 1 */	128);
	// InitDrumKey(50, /* High Tom 1 */		128);
	// InitDrumKey(51, /* Ride Cymbal 1 */		128);
	// InitDrumKey(52, /* Chinese Cymbal */	128);
	// InitDrumKey(53, /* Ride Bell */			128);
	// InitDrumKey(54, /* Tambourine */		128);
	// InitDrumKey(55, /* Splash Cymbal */		128);
	// InitDrumKey(56, /* Cowbell */			128);
	// InitDrumKey(57, /* Crash Cymbal 2 */	128);
	// InitDrumKey(58, /* Vibra Slap */		128);
	// InitDrumKey(59, /* Ride Cymbal 2 */		128);
	// InitDrumKey(60, /* High Bongo */		128);
	// InitDrumKey(61, /* Low Bongo */			128);
	// InitDrumKey(62, /* Mute High Conga */	128);
	// InitDrumKey(63, /* Open High Conga */	128);
	// InitDrumKey(64, /* Low Conga */			128);
	// InitDrumKey(65, /* High Timbale */		128);
	// InitDrumKey(66, /* Low Timbale */		128);
	// InitDrumKey(67, /* High Agogo */		128);
	// InitDrumKey(68, /* Low Agogo */			128);
	// InitDrumKey(69, /* Cabasa */			128);
	// InitDrumKey(70, /* Maracas */			128);
	// InitDrumKey(71, /* Short Whistle */		128);
	// InitDrumKey(72, /* Long Whistle */		128);
	// InitDrumKey(73, /* Short Guiro */		128);
	// InitDrumKey(74, /* Long Guiro */		128);
	// InitDrumKey(75, /* Claves */			128);
	// InitDrumKey(76, /* High Wood Block */	128);
	// InitDrumKey(77, /* Low Wood Block */	128);
	// InitDrumKey(78, /* Mute Cuica */		128);
	// InitDrumKey(79, /* Open Cuica */		128);
	// InitDrumKey(80, /* Mute Triangle */		128);
	// InitDrumKey(81, /* Open Triangle */		128);

	return true;
}

void CSono::Close()
{
	m_DLS.Close();
	
	if (!m_hWaveOut) return;
	
	/* stop sound */ 
    waveOutReset( m_hWaveOut );
    waveOutClose( m_hWaveOut );
    m_hWaveOut = NULL;
    
    /* free memory */ 
    free(m_pBuffer); m_pBuffer = NULL;
    m_BufferSize = 0;
}

#ifdef _DEBUG
bool CSono::Save(const char* sPath)
{
	HANDLE	f;
	DWORD	DW;
	WORD	W;
	DWORD	u;
	
	f = CreateFileA(sPath, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
	if (INVALID_HANDLE_VALUE==f) return false;
	
	/* RIFF Chunk */ 
	WriteFile(f, "RIFF", 4, &u,NULL);
	DW = 4+24+8+DWORD(m_BufferSize);
	WriteFile(f, &DW, sizeof(DW), &u,NULL);
	WriteFile(f, "WAVE", 4, &u,NULL);
	
	/* FORMAT chunk */ 
	WriteFile(f, "fmt ", 4, &u,NULL);
	DW = 16;							WriteFile(f, &DW, sizeof(DW), &u,NULL);
	W  = 1;								WriteFile(f, &W, sizeof(W), &u,NULL);
	W  = gWavFormat.nChannels;			WriteFile(f, &W, sizeof(W), &u,NULL);
	DW = gWavFormat.nSamplesPerSec;		WriteFile(f, &DW, sizeof(DW), &u,NULL);
	DW = gWavFormat.nAvgBytesPerSec;	WriteFile(f, &DW, sizeof(DW), &u,NULL);
	W  = gWavFormat.wBitsPerSample*gWavFormat.nChannels/8; WriteFile(f, &W, sizeof(W), &u,NULL);
	W  = gWavFormat.wBitsPerSample;		WriteFile(f, &W, sizeof(W), &u,NULL);
	
	/* DATA chunk */ 
	WriteFile(f, "data", 4, &u,NULL);
	DW = DWORD(m_BufferSize);	WriteFile(f, &DW, sizeof(DW), &u,NULL);
	WriteFile(f, m_pBuffer, DWORD(m_BufferSize), &u,NULL);
	
	CloseHandle(f);
	return true;
}
#endif

bool CSono::SetTempo(int Tempo, int TimeDivision)
{
	m_Tempo = Tempo / 250.0f;
	m_TicksPerSecond = (1000000.0f * TimeDivision) / Tempo;
	m_TicksPerSample = m_TicksPerSecond / NULSTEINSONO_RATE;
	return true;
}


class CSonoUpdater : public IForTask
{
public:
	bool DoRange(CWorkerThread* pThread, const CRange &Range) const
	{
		sSonoSample* pOut;
		
		for(int iChannel=Range.begin; iChannel<Range.end; iChannel++)
		{
			pOut = m_pSono->m_ChannelBuffer[iChannel];

			for(int iSample=0; iSample<m_SampleCount; iSample++)
			{
				m_pSono->m_Channel[iChannel].Update(m_pSono, 1.0f/NULSTEINSONO_RATE, m_pSono->m_TicksPerSample);
				m_pSono->m_Channel[iChannel].ComputeSample(m_pSono, pOut++);
			}
		}
		
		return true;
	}
	
public:
	CSono*		m_pSono;
	int			m_SampleCount;
};

bool CSono::Update(float dt)
{
	CSonoUpdater	SonoUpdater;
	int				SampleCount;

	SonoUpdater.m_pSono	= this;
	
	for(	SampleCount = int( dt*NULSTEINSONO_RATE );
			SampleCount;
			SampleCount -= SonoUpdater.m_SampleCount	)		
	{
		/* update each channel in parallel */ 
		SonoUpdater.m_SampleCount	= min(SampleCount, SONO_MAXSAMPLESPERUPDATE);
		ParallelFor(&SonoUpdater, CRange(0,16));
		
		/* add all channels up */ 
		for(int iSample=0; iSample<SonoUpdater.m_SampleCount; iSample++)
		{
			sSonoSample Out;
			
			Out.L = 0.0f;
			Out.R = 0.0f;

			/* Update time */ 
			m_Time += 1.0f/NULSTEINSONO_RATE;
			
			/* Update all channels */ 
			for(int iChannel=0; iChannel<16; iChannel++)
			{
				Out.Add( m_ChannelBuffer[iChannel][iSample] );
			}
			
			/* output to playback buffer */ 
			Out.Mul( 0.333f );
			
				 if (Out.L> 1.25f) Out.L = 1.25f;
			else if (Out.L<-1.25f) Out.L =-1.25f;
			*m_pWriteBuffer++ = SHORT( Out.L*25000.0f );

				 if (Out.R> 1.25f) Out.R = 1.25f;
			else if (Out.R<-1.25f) Out.R =-1.25f;
			*m_pWriteBuffer++ = SHORT( Out.R*25000.0f );
			
			if (m_pWriteBuffer >= m_pEndOfBuffer)
				m_pWriteBuffer = m_pBuffer;
		}
	}

	return true;
}


long CSono::GetBufferPos() const
{
    MMTIME  mmt;

    mmt.wType = TIME_BYTES;
    waveOutGetPosition( m_hWaveOut, &mmt, sizeof(MMTIME) );

	return mmt.u.cb;
}

long CSono::GetBufferTime() const
{
    MMTIME  mmt;

    mmt.wType = TIME_MS;
    waveOutGetPosition( m_hWaveOut, &mmt, sizeof(MMTIME) );
	
	switch(mmt.wType)
	{
		case TIME_MS	: return mmt.u.ms;
		case TIME_BYTES	: return (mmt.u.cb*10) / (NULSTEINSONO_CHANNELS*NULSTEINSONO_RATE*(NULSTEINSONO_BITS/8)/100);
	}
	
	ASSERT( false);
	return 0;
}

void CSono::SetLoop(int StartTick)
{
	int iChannel;
	
	m_LoopStartTick = StartTick;
	m_Duration = 0;
	for(iChannel=0; iChannel<16; iChannel++)
	{
		m_Duration = max(m_Duration, m_Channel[iChannel].GetTrackTickCount());
	}
	
	for(iChannel=0; iChannel<16; iChannel++)
	{
		m_Channel[iChannel].SetLoop(this, StartTick);		
	}
}

//________________________________________________________________________________
bool CSonoNote::Start(CSono* pSono, SHORT Instrument, BYTE Note, BYTE Velocity)
{
	ASSERT(0==m_State);
	
	m_Instrument	= Instrument;
	m_Note			= Note;
	m_Velocity		= Velocity;
	m_State			= 'A';
	m_Time			= 0.0f;
	
	if (m_Instrument>=0)
	{
		sDLSInstrument*	pInstrument;
		sDLSRegion*		pRegion;
		
		pInstrument = &pSono->m_DLS.m_Instrument[Instrument];
		for(pRegion=pInstrument->m_Region; pRegion->m_KeyFrom>Note || pRegion->m_KeyTo<Note; pRegion++)
		{ ; }
		
		m_Sample = (SHORT)pRegion->m_iSample;
	}
	
	return true;
}

bool CSonoNote::Release()
{
	switch(m_State)
	{
		case 'A': 
			m_State = 'a';	
			return true;
			
		case 'D': 
			m_State = 'd';	
			return true;
			
		case 'S': 
			m_State = 'R'; 
			m_Time	= 0.0f;	
			return true;
		
		/* case 'R': already released ?! */ 
	}
	
	return false;
 }

bool CSonoNote::Update(CSono* pSono, float dt)
{
	m_Time += dt;

	if (m_Instrument<0)
	{
		CSonoInstrument *pInstrument = &pSono->m_Instrument[0];

		switch(m_State)
		{
			case 'A': 
				if (m_Time<pInstrument->m_Attack) break;
				m_State	= 'D';
				m_Time	-= pInstrument->m_Attack;
				/* pass through */ 
			
			case 'D': 
				if (m_Time<pInstrument->m_Decay) break;
				m_State	= 'S';
				m_Time	-= pInstrument->m_Decay;
				/* pass through */ 
			
			case 'S': 
				break;
			
			case 'a':
				if (m_Time<pInstrument->m_Attack) break;
				m_State	= 'd';
				m_Time	-= pInstrument->m_Attack;
				/* pass through */ 
			
			case 'd': 
				if (m_Time<pInstrument->m_Decay) break;
				m_State	= 'R';
				m_Time	-= pInstrument->m_Decay;
				/* pass through */ 

			case 'R':
				if (m_Time<pInstrument->m_Release) break;
				m_State	= 0;
				m_Time	= 0.0f;
		}
	}
	
	return true;
}

bool CSonoNote::ComputeSample(CSono* pSono, sSonoSample* pOut)
{
	float V;
	
	if (m_Instrument>=0)
	{
		V = SampleDLS(pSono);
	}
	else
	{
		CSonoInstrument *pInstrument = &pSono->m_Instrument[1-m_Instrument];
		float S;

		/* sample */ 
		S = SampleWave(pSono);
		
		/* enveloppe */ 
		switch(m_State)
		{
			case 'A': 
			case 'a':
				V = m_Time/pInstrument->m_Attack; 
				break;		
				
			case 'D': 
			case 'd':
				V = m_Time/pInstrument->m_Decay; 
				V = pInstrument->m_Sustain*V + (1.0f-V); 
				break;
						
			case 'S': 
				V = pInstrument->m_Sustain;
				break;
			
			case 'R':
				V = (1.0f-m_Time/pInstrument->m_Release) * pInstrument->m_Sustain;
				break;
				
			default:
				ASSERT(false);
				return false;
		}
		
		/* add it all together */ 
		V = S * V * (m_Velocity/127.0f); 
	}
	
	pOut->L = V;
	pOut->R = V;
	
	return true;
}


float CSonoNote::SampleWave(CSono* pSono)
{
	float a = pSono->m_Time
			* pSono->m_KeyFrequency[m_Note] 
			* (2*3.141592654f);
	
	return Cos(a);
}	

float CSonoNote::SampleDLS(CSono* pSono)
{
	sDLSInstrument*	pInstrument;
	sDLSSample*		pSample;
	float			SamplesPerSec;
	float			k,t,s;
	int				i;
	
	pInstrument = &pSono->m_DLS.m_Instrument[m_Instrument];
	pSample		= &pSono->m_DLS.m_Sample[ m_Sample ];
	
	SamplesPerSec = pSample->m_SamplesPerSec * pSono->m_KeyFrequency[m_Note] / pSample->m_UnityFreq;
	t = m_Time * SamplesPerSec;
	i = int(t);
	k = t-i;
	
	if (!pInstrument->m_bDrum && m_State=='A' && pSample->m_LoopLength)
	{
		if (i >= int(pSample->m_LoopStart+pSample->m_LoopLength))
		{
			int LoopCount;
			int LoopSamples;
			
			LoopCount   = (i-pSample->m_LoopStart) / pSample->m_LoopLength;
			LoopSamples = LoopCount*pSample->m_LoopLength;
			
			i      -= LoopSamples;
			m_Time -= LoopSamples / SamplesPerSec;
		}
	}
	else if (i+1 >= int(pSample->m_DataSize/2) )
	{
		m_State = 0;
		return 0;
	}
	
	s =	(	pSample->m_pData[i  ] * (1.0f-k) 
		+	pSample->m_pData[i+1] *       k  ) 
		* pSample->m_Scale;
	
	return s;
}

//________________________________________________________________________________
bool CSonoChannel::SetTrack(CSono* pSono, const sTrackItem* pTrack)
{
	int n;

	m_pTrack				= pTrack;
	m_LoopCountdown			= 0;
	m_LoopStartItem			= 0;
	m_LoopStartTicks		= 0;
	m_bDrums				= false; /* by default */ 

	n=0;
	while (pTrack->dt != 0xFFFF)
	{
		pTrack++;
		n++;
	}
	m_TrackItemCount	= n;
	
	return ResetTrack();
}

bool CSonoChannel::ResetTrack()
{
	if (!m_pTrack) return false;
	
	m_iTrackItem		= 0;
	m_TrackTicks		= 0;
	
	return true;
}

bool CSonoChannel::SetInstrument(CSono* pSono, int Bank, int Program)
{
	m_Instrument = pSono->m_DLS.FindInstrument(Bank, Program);
	return m_Instrument >= 0;
}


bool CSonoChannel::CheckLoop(CSono* pSono, float Ticks)
{
	if (!m_LoopStartItem) return false;
	
	m_LoopCountdown-=Ticks;
	if (m_LoopCountdown>0) return false;
	
	m_iTrackItem = m_LoopStartItem;
	m_TrackTicks = m_LoopStartTicks - m_LoopCountdown;
	m_LoopCountdown += pSono->m_Duration - pSono->m_LoopStartTick;
	
	return true;
}

bool CSonoChannel::Update(CSono* pSono, float dt, float Ticks)
{
	if (!m_pTrack) return true;
		
	m_TrackTicks += Ticks;
	
	/* consume events */ 
	for(/*m_iTrackItem*/; m_iTrackItem<m_TrackItemCount; m_iTrackItem++)
	{
		if (m_TrackTicks < m_pTrack[m_iTrackItem].dt) break;
		
		m_TrackTicks -= m_pTrack[m_iTrackItem].dt;
		if (m_pTrack[m_iTrackItem].Velocity)
			 UpdateNoteVelocity( pSono, m_pTrack[m_iTrackItem].Note, m_pTrack[m_iTrackItem].Velocity );
		else ReleaseNote( m_pTrack[m_iTrackItem].Note );
	}
	
	CheckLoop(pSono, Ticks);
	
	/* update notes */ 
	for(int i=0; i<SONO_MAXNOTESINCHANNEL; i++)
	{
		if (m_Note[i].m_State==0) continue;
		
		m_Note[i].Update(pSono, dt);
	}
	
	return true;
}

int  CSonoChannel::GetTrackTickCount()
{
	int n=0;
	
	if (!m_pTrack) return 0;
	
	for(int i=0; i<m_TrackItemCount; i++)
	{
		n += m_pTrack[i].dt;
	}
	
	return n;
}

void CSonoChannel::SetLoop(CSono* pSono, int StartTick)
{
	m_LoopCountdown	= float(pSono->m_Duration);
	
	for(int i=0; i<m_TrackItemCount; i++)
	{
		int dt = m_pTrack[i].dt;
		
		if (StartTick<dt) 
		{
			m_LoopStartItem  = i;
			m_LoopStartTicks = float(StartTick);
			return;
		}
		
		StartTick -= dt;
	}
}

bool CSonoChannel::ComputeSample(CSono* pSono, sSonoSample* pOut)
{
	pOut->L = 0.0f;
	pOut->R = 0.0f;
	
	if (m_pTrack) 
	{
		for(int iNote=0; iNote<SONO_MAXNOTESINCHANNEL; iNote++)
		{
			sSonoSample V;
			
			if (m_Note[iNote].m_State==0) continue;
			
			m_Note[iNote].ComputeSample(pSono, &V);
			pOut->Add(V);
		}
	}
	
	return true;
}

bool CSonoChannel::UpdateNoteVelocity(CSono* pSono, BYTE iNote, BYTE Velocity)
{
	int i;
	int iFree;
	
	iFree = -1;
	for(i=0; i<SONO_MAXNOTESINCHANNEL; i++)
	{
		if (m_Note[i].m_State==0)
		{
			if (iFree<0) iFree=i;
			continue;
		}
		
		if (m_Note[i].m_Note == iNote)
		{
			m_Note[i].m_Velocity = Velocity;
			return true;
		}
	}
	
	ASSERT(iFree>=0);
	if (iFree<0) return false;
	
	if (!m_bDrums)
	{
		m_Note[iFree].Start(pSono, m_Instrument, iNote, Velocity);
	}
	else
	{
		CSonoDrumKey* p = &pSono->m_DrumKey[iNote];
		m_Note[iFree].Start(pSono, p->m_Instrument, p->m_Note, Velocity);
		m_Note[iFree].m_State = 'a'; /* ie don't loop */ 
	}
	
	return true;
}

bool CSonoChannel::ReleaseNote(BYTE iNote)
{
	int i;
	
	for(i=0; i<SONO_MAXNOTESINCHANNEL; i++)
	{
		if (m_Note[i].m_State==0) continue;
		
		if (m_Note[i].m_Note==iNote) 
		{
			m_Note[i].Release();
			return true;
		}
	}
	
	return false;
}

