//-------------------------------------------------------------------------------------
//
// Copyright 2009 Intel Corporation
// All Rights Reserved
//
// Permission is granted to use, copy, distribute and prepare derivative works of this
// software for any purpose and without fee, provided, that the above copyright notice
// and this statement appear in all copies.  Intel makes no representations about the
// suitability of this software for any purpose.  THIS SOFTWARE IS PROVIDED "AS IS."
// INTEL SPECIFICALLY DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, AND ALL LIABILITY,
// INCLUDING CONSEQUENTIAL AND OTHER INDIRECT DAMAGES, FOR THE USE OF THIS SOFTWARE,
// INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PROPRIETARY RIGHTS, AND INCLUDING THE
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  Intel does not
// assume any responsibility for any errors which may appear in this software nor any
// responsibility to update it.
//

#ifndef _NULSTEIN_SONO_H_
#define _NULSTEIN_SONO_H_

#include "SonoDLS.h"

#define SONO_MAXCHANNELS			 16
#define SONO_MAXNOTESINCHANNEL		128
#define SONO_MAXINSTRUMENTS			256
#define SONO_MAXSAMPLESPERUPDATE	2048

inline bool _FAILED(MMRESULT mmr)	{return mmr!=MMSYSERR_NOERROR;} 

float MIDIKeyToFrequency(int iKey);

class CSono;

struct sTrackItem
{
	WORD	dt;
	BYTE	Note;
	BYTE	Velocity;
};

struct sSonoSample
{
	float L,R;
	
	void Add(const sSonoSample &S) 
	{
		L+=S.L; 
		R+=S.R;
	}
	
	void Mul(float k)
	{
		L*=k;
		R*=k;
	}
};

class CSonoInstrument
{
public:
	CSonoInstrument()
	{
		m_Attack	= 0.001f;
		m_Decay		= 0.000f;
		m_Sustain	= 1.000f;
		m_Release	= 0.010f;
	}
	
	bool ParseDLS(const BYTE* pData, const BYTE* pEnd);
	bool ParseDLS_RegionList(ULONG iRegion, const BYTE* pData, const BYTE* pEnd);
	bool ParseDLS_Info(const BYTE* pData, const BYTE* pEnd);
	
public:
	float	m_Attack;	/* attack duration	*/ 
	float	m_Decay;	/* decay duration	*/ 
	float	m_Sustain;	/* sustain level	*/ 
	float	m_Release;	/* release duration	*/ 
	
	const char* m_Copyright;
	const char* m_Name;
};

class CSonoNote
{
public:
	CSonoNote() {m_State = 0;}
	
	bool Start(CSono* pSono, SHORT Instrument, BYTE Note, BYTE Velocity);
	bool Release();
	bool Update(CSono* pSono, float dt);
	
	bool ComputeSample(CSono* pSono, sSonoSample* pOut);
	
	float SampleDLS(CSono* pSono);
	float SampleWave(CSono* pSono);

public:
	SHORT	m_Instrument;
	SHORT	m_Sample;
	BYTE	m_Note;
	BYTE	m_Velocity;			
	BYTE	m_State;	/* 'A','D','S','R' ('a','d' if released early) */ 
	float	m_Time;
};

struct CSonoDrumKey
{
	SHORT	m_Instrument;
	BYTE	m_Note;
	BYTE	pad;
};


class CSonoChannel
{
public:
	bool SetTrack(CSono* pSono, const sTrackItem* pTrack);
	bool SetInstrument(CSono* pSono, int Bank, int Program);
	bool SetInstrument(CSono* pSono, int Program) {return SetInstrument(pSono, 0, Program);}
	bool ResetTrack();
	bool Update(CSono* pSono, float dt, float Ticks); 
	bool ComputeSample(CSono* pSono, sSonoSample* pOut);

public:
	bool UpdateNoteVelocity(CSono* pSono, BYTE iNote, BYTE Velocity);
	bool ReleaseNote(BYTE iNote);
	void SetLoop(CSono* pSono, int StartTick);
	int  GetTrackTickCount();
	bool CheckLoop(CSono* pSono, float Ticks);
	
public:
	CSonoNote			m_Note[SONO_MAXNOTESINCHANNEL];
	
	const sTrackItem*	m_pTrack;
	int					m_TrackItemCount;
	int					m_iTrackItem;
	
	float				m_TrackTicks; 
	SHORT				m_Instrument;
	bool				m_bDrums;
	
	int					m_LoopStartItem;
	float				m_LoopStartTicks;
	float				m_LoopCountdown;
};


class CSono
{
public:
	CSono();

	bool Init(int BufferPlayTime); /* in seconds */ 
	bool StartPlaying();
	void Close();
	
	bool InitDrumKeys();
	bool InitDrumKey(int iKey, int iInstrument, int iNote = 60);
	
	bool SetTempo(int Tempo, int TimeDivision);	/* s/quarter-note, ticks per beat */ 
	void SetLoop(int StartTick);
	
	bool Update(float dt); /* seconds */ 
	
	long GetBufferPos() const;
	long GetBufferTime() const; /* in milliseconds */ 
	
	
#ifdef _DEBUG
	bool Save(const char* sPath);
#endif 
	
public:
	sSonoSample				m_ChannelBuffer[SONO_MAXCHANNELS][SONO_MAXSAMPLESPERUPDATE]; /* temp buffer to update channels in parallel */ 
	
	HWAVEOUT				m_hWaveOut;
	WAVEHDR					m_WaveHeader;
	SHORT*					m_pBuffer;
	size_t					m_BufferSize;
	
	SHORT*					m_pWriteBuffer;
	SHORT*					m_pEndOfBuffer;
	
	int						m_Duration; /* in ticks */ 
	int						m_LoopStartTick;
	float					m_Tempo;			/* ms/full-note */ 
	float					m_TicksPerSecond;
	float					m_TicksPerSample;
	float					m_Time; 
	float					m_KeyFrequency[127];
	
	CSonoChannel			m_Channel[SONO_MAXCHANNELS];
	CSonoInstrument			m_Instrument[SONO_MAXINSTRUMENTS];
	
	CDLSFile				m_DLS;
	CSonoDrumKey			m_DrumKey[128];
};

#endif // _NULSTEIN_SONO_H_