;***************************************************************************************************************************
;*
;*   Project	: 'Stars' - A parallax-scrolling starfield example.
;*
;*   Module	: Startup.asm
;*
;*   Function	: The code for the demo.
;*
;*   Comment	: My screen column width is 125.
;*
;*   Tab Stops	: 20 30 70
;*                 |         |                                       |
;*
;*   (Most of it was) Written by Bob Koon (bob@onemanband.com)
;*
;***************************************************************************************************************************

	INCLUDE	"Hardware.inc"
	INCLUDE	"StarDataTest.asm"


;***************************************************************************************************************************
;*	externally referenced symbols
;***************************************************************************************************************************


;***************************************************************************************************************************
;*	equates
;***************************************************************************************************************************

HIGHMEM_VARSIZE	EQU	1	; size (in bytes) of the variable block in high mem (see below)

NUM_STARDEPTH	EQU	13	; number of stars per depth
NUM_STARS	EQU	(NUM_STARDEPTH * 3)	; 3 depths

OAMZ1_ADDR	EQU	BASE_OAMRAM
OAMZ2_ADDR	EQU	(OAMZ1_ADDR + (NUM_STARDEPTH * 4))
OAMZ3_ADDR	EQU	(OAMZ2_ADDR + (NUM_STARDEPTH * 4))

SPRITEDMA_ADDR	EQU	BASE_HIGHRAM + 16	;HIGHMEM_VARSIZE

LCDC_INCREMENT	EQU	32

; various user/state flags
USER_F_VBLANKEND	EQU	%00000010	; set when vblank is done
USER_F_ISCGB	EQU	%00000001	; set when running on a CGB


;***************************************************************************************************************************
;*	variables
;***************************************************************************************************************************

	SECTION	"UserVars",BSS

StarCoordsZ1	DS	(NUM_STARDEPTH * 4)	; y,x,z,attr
StarCoordsZ2	DS	(NUM_STARDEPTH * 4)	; y,x,z,attr
StarCoordsZ3	DS	(NUM_STARDEPTH * 4)	; y,x,z,attr
StarCoordsZ1E	DS	(NUM_STARDEPTH * 4)	; y,x,z,attr
StarCoordsZ2E	DS	(NUM_STARDEPTH * 4)	; y,x,z,attr
StarCoordsZ3E	DS	(NUM_STARDEPTH * 4)	; y,x,z,attr
RandomSeed	DB

	; variables located in high memory
	RSSET	BASE_HIGHRAM
UserFlags	RB	1	; various user/state flags (see list above)
FrameCounter	RW	1
ScanLine	RB	1
ScanSine	RW	1
ScanOffset	RB	1
ScanOffset2	RB	1


;***************************************************************************************************************************
;*	cartridge header
;***************************************************************************************************************************

;	SECTION	"Org $00",HOME[$00]
	; $0000 - Restart $00 address
;RST_00:	jp	<somewhere>

;	SECTION	"Org $08",HOME[$08]
	; $0008 - Restart $08 address
;RST_08:	jp	<somewhere>

;	SECTION	"Org $10",HOME[$10]
	; $0010 - Restart $10 address
;RST_10:	jp	<somewhere>

;	SECTION	"Org $18",HOME[$18]
	; $0018 - Restart $18 address
;RST_18:	jp	<somewhere>

;	SECTION	"Org $20",HOME[$20]
	; $0020 - Restart $20 address
;RST_20:	jp	<somewhere>

;	SECTION	"Org $28",HOME[$28]
	; $0028 - Restart $28 address
;RST_28:	jp	<somewhere>

;	SECTION	"Org $30",HOME[$30]
	; $0030 - Restart $30 address
;RST_30:	jp	<somewhere>

;	SECTION	"Org $38",HOME[$38]
	; $0038 - Restart $38 address
;RST_38:	jp	<somewhere>

	SECTION	"Org $40",HOME[$40]
	; $0040 - V-Blank interrupt start address
	jp	IRQ_VBlank

	SECTION	"Org $48",HOME[$48]
	; $0048 - LCDC Status interrupt start address
	jp	IRQ_LCDC

;	SECTION	"Org $50",HOME[$50]
	; $0050 - Timer Overflow interrupt start address
;	jp	IRQ_Timer

;	SECTION	"Org $58",HOME[$58]
	; $0058 - Serial Transfer Completion interrupt start address
;	jp	IRQ_Serial

;	SECTION	"Org $60",HOME[$60]
	; $0060 - Joypad interrupt start address
;	jp	IRQ_Joypad


	SECTION	"Start",HOME[$100]
	; $0100-$0103 (Game code start)
	nop
	jp	Start

	; $0104-$0133 (Nintendo logo - do _not_ modify the logo data here or the CGB will freeze)
	DB	$CE,$ED,$66,$66,$CC,$0D,$00,$0B,$03,$73,$00,$83,$00,$0C,$00,$0D
	DB	$00,$08,$11,$1F,$88,$89,$00,$0E,$DC,$CC,$6E,$E6,$DD,$DD,$D9,$99
	DB	$BB,$BB,$67,$63,$6E,$0E,$EC,$CC,$DD,$DC,$99,$9F,$BB,$B9,$33,$3E

	; $0134-$013E (Game title - up to 11 upper case ASCII characters; pad with $00)
	DB	"STARS      "
		;0123456789A

	; $013F-$0142 (Product code - 4 ASCII characters)
	DB	"    "
		;0123

	; $0143 (Color GameBoy compatability code)
	DB	$C0	; $C0 - Dedicated Color GameBoy cartridge
			; $80 - Color GameBoy compatible cartridge
			; $00 - Not a Color GameBoy cartridge (all other values)

	; $0144 (High-nibble of license code - normally $00 if $014B != $33)
	DB	$00

	; $0145 (Low-nibble of license code - normally $00 if $014B != $33)
	DB	$00

	; $0146 (GameBoy/Super GameBoy indicator)
	DB	$00	; $00 - GameBoy
			; $03 - Super GameBoy

	; $0147 (Cartridge type - all Color GameBoy cartridges are at least $19)
	DB	$00	; $00 - ROM only
			; $01 - ROM + MBC1
			; $02 - ROM + MBC1 + RAM
			; $03 - ROM + MBC1 + RAM + Battery
			; $05 - ROM + MBC2
			; $06 - ROM + MBC2 + Battery
			; $08 - ROM + RAM
			; $09 - ROM + RAM + Battery
			; $0B - ROM + MMMO1
			; $0C - ROM + MMMO1 + SRAM
			; $0D - ROM + MMMO1 + SRAM + Battery
			; $0F - ROM + MBC3 + Timer + Battery
			; $10 - ROM + MBC3 + Timer + RAM + Battery
			; $11 - ROM + MBC3
			; $12 - ROM + MBC3 + RAM
			; $13 - ROM + MBC3 + RAM + Battery
			; $19 - ROM + MBC5
			; $1A - ROM + MBC5 + RAM
			; $1B - ROM + MBC5 + RAM + Battery
			; $1C - ROM + MBC5 + Rumble
			; $1D - ROM + MBC5 + Rumble + SRAM
			; $1E - ROM + MBC5 + Rumble + SRAM + Battery
			; $1F - Pocket Camera
			; $FD - Bandai TAMA5
			; $FE - Hudson HuC-3
			; $FF - Hudson HuC-1

	; $0148 (ROM size)
	DB	$00	; $00 - 256Kbit =  32Kbyte =   2 banks
			; $01 - 512Kbit =  64Kbyte =   4 banks
			; $02 -   1Mbit = 128Kbyte =   8 banks
			; $03 -   2Mbit = 256Kbyte =  16 banks
			; $04 -   4Mbit = 512Kbyte =  32 banks
			; $05 -   8Mbit =   1Mbyte =  64 banks
			; $06 -  16Mbit =   2Mbyte = 128 banks
			; $52 -   9Mbit = 1.1Mbyte =  72 banks
			; $53 -  10Mbit = 1.2Mbyte =  80 banks
			; $54 -  12Mbit = 1.5Mbyte =  96 banks

	; $0149 (RAM size)
	DB	$00	; $00 - None
			; $01 -  16Kbit =   2Kbyte =  1 bank
			; $02 -  64Kbit =   8Kbyte =  1 bank
			; $03 - 128Kbit =  32Kbyte =  4 banks
			; $04 -   1Mbit = 128Kbyte = 16 banks

	; $014A (Destination code)
	DB	$01	; $00 - Japanese
			; $01 - All others

	; $014B (Licensee code - this _must_ be $33)
	DB	$33	; $33 - Check $0144/$0145 for Licensee code.

	; $014C (Mask ROM version - handled by a post-linking tool)
	DB	$00

	; $014D (Complement check - handled by a post-linking tool)
	DB	$00

	; $014E-$014F (Cartridge checksum - handled by a post-linking tool)
	DW	$00


;***************************************************************************************************************************
;*	let the games...BEGIN!
;***************************************************************************************************************************

Start::
	di

	ld	sp,$E000	; set up a stack

	; if running on a CGB, set the user flag
	and	a
	cp	CPU_GBC
	ld	a,0
	jr	nz,.setuserflags
	ld	a,USER_F_ISCGB
.setuserflags
	ld	[UserFlags],a

	call	TurnOffDisplay

	; clear LCD control registers
	xor	a
	ld	[REG_IE],a
	ld	[REG_IF],a
	ld	[REG_SCX],a
	ld	[REG_SCY],a
	ld	[REG_STAT],a

	; if running on CGB, set the CPU to double speed (the CGB is single speed on boot up)...
	ld	a,[UserFlags]
	and	USER_F_ISCGB
	jr	z,.clearram

	call	ToggleCPUSpeed

	; ...and clear the CGB-specific registers
	xor	a
	ld	[REG_VBK],a	; set VRAM Bank(0)
	ld	[REG_SVBK],a	; set internal RAM Bank(0)
	ld	[REG_RP],a	; clear IR port

	; clear $C000-$DFFF (internal RAM)
.clearram:
	xor	a
	ld	hl,$DFFF
	ld	c,$20
	ld	b,0
.clear1:
	ld	[hl-],a
	dec	b
	jr	nz,.clear1
	dec	c
	jr	nz,.clear1

	; clear $FE00-$FEFF (OAM [sprite attribute] RAM)
	ld	hl,$FEFF
	ld	b,0
.clear2:
	ld	[hl-],a
	dec	b
	jr	nz,.clear2

	; clear $FF80-$FFFF (internal [high] RAM)
	ld	hl,$FFFF
	ld	b,$80 - HIGHMEM_VARSIZE	; also clear the variables (except UserFlags)
.clear3:
	ld	[hl-],a
	dec	b
	jr	nz,.clear3

	; clear the bg maps
	ld	hl,BASE_MAPRAM9800
	call	ClearAllBGAttrMap
	ld	hl,BASE_MAPRAM9C00
	call	ClearAllBGAttrMap

	ld	hl,BASE_MAPRAM9800
	call	ClearAllBGMap
	ld	hl,BASE_MAPRAM9C00
	call	ClearAllBGMap

	; set up the default palettes
	ld	a,$E4
	ld	[REG_BGP],a
	ld	[REG_OBP0],a
	ld	[REG_OBP1],a

	; set up the custom palettes
	ld	a,[UserFlags]
	and	USER_F_ISCGB
	jr	z,.setvram

	ld	hl,BGPaletteData
	call	SetBGPalettes
	ld	hl,OBJPaletteData
	call	SetOBJPalettes

	; copy the graphic tile data to VRAM ($8000:Bank 0)
.setvram:
	ld	bc,TileGraphicData
	ld	hl,BASE_VRAM
	ld	de,TileGraphicDataEnd - TileGraphicData
	call	CopyToVRAM

	; create random star coordinates and copy them to shadow memory
	ld	a,[UserFlags]
	and	USER_F_ISCGB
	jr	z,.printnoncgbmsg

	call	CreateStarCoords
	ld	hl,StarCoordsZ1E
	ld	de,StarCoordsZ1
	call	AssignSpritePos
	ld	hl,StarCoordsZ2E
	ld	de,StarCoordsZ2
	call	AssignSpritePos
	ld	hl,StarCoordsZ3E
	ld	de,StarCoordsZ3
	call	AssignSpritePos

	; print something
	ld	bc,Line0String
	ld	hl,$9800
	ld	de,$0405	; d = y coord, e = x coord
	call	PrintText
	ld	bc,Line1String
	ld	hl,$9800
	ld	de,$0606	; d = y coord, e = x coord
	call	PrintText
	ld	bc,Line2String
	ld	hl,$9800
	ld	de,$0805	; d = y coord, e = x coord
	call	PrintText
	ld	bc,Line3String
	ld	hl,$9800
	ld	de,$0D04	; d = y coord, e = x coord
	call	PrintText

	; initialize the scroll variables
	ld	hl,sinedata
	ld	a,h
	ld	[ScanSine],a
	ld	a,l
	ld	[ScanSine+1],a

	; copy sprite DMA routine to HIRAM
	ld	hl,SpriteDMAFunc
	ld	de,SPRITEDMA_ADDR
	ld	b,SpriteDMAFuncEnd - SpriteDMAFunc
.loadhigh:
	ld	a,[hl+]
	ld	[de],a
	inc	de
	dec	b
	jr	nz,.loadhigh

	; set up the LCDC interrupt
;	ld	a,LCDC_INCREMENT	; set line at which lcdc interrupt occurs
;	ld	[ScanLine],a
;	ld	[REG_LYC],a
;	ld	a,REG_STAT_F_LYC | REG_STAT_F_LYCF
;	ld	[REG_STAT],a
	jp	.initdisplay

	; print a special message for non-CGB users
.printnoncgbmsg:
	ld	bc,NonCGBString0
	ld	hl,$9800
	ld	de,$0701	; d = y coord, e = x coord
	call	PrintText
	ld	bc,NonCGBString1
	ld	hl,$9800
	ld	de,$0803	; d = y coord, e = x coord
	call	PrintText
	ld	bc,NonCGBString2
	ld	hl,$9800
	ld	de,$0903	; d = y coord, e = x coord
	call	PrintText

	; initialize the display
.initdisplay:
	xor	a
	ld	[REG_NR52],a	; turn sound off
	ld	[REG_WY],a	; window ypos = 0
	ld	a,$07
	ld	[REG_WX],a	; window xpos = 8

	; turn the display on
	ld	a,REG_LCDC_F_ON | REG_LCDC_F_BG8000 | REG_LCDC_F_BG9800 | REG_LCDC_F_OBJON | REG_LCDC_F_BGON
	ld	[REG_LCDC],a
	xor	a
	ld	[REG_IF],a

	ld	a,[UserFlags]
	and	USER_F_ISCGB
	jr	z,.setvblankonly
	ld	a,REG_IE_F_LCDC | REG_IE_F_VBLANK
	jp	.setie
.setvblankonly:
	ld	a,REG_IE_F_VBLANK
.setie:
	ld	[REG_IE],a

	ei

.mainloop:
	ld	a,[UserFlags]
	and	USER_F_ISCGB
	jr	z,.donothing

	; do stuff here
	; do stuff here
	; do stuff here

	jp	.waitvblank

	; save the batteries on the old GB
.donothing:
	halt
	nop

	; wait for a V-Blank
.waitvblank:
	ld	a,[UserFlags]
	and	USER_F_VBLANKEND	; clobbers all other bits but bit 1
	jr	z,.waitvblank

	ld	a,[UserFlags]	; see comment above
	xor	USER_F_VBLANKEND
	ld	[UserFlags],a

	jp	.mainloop


;***************************************************************************************************************************
;*	Support routines
;***************************************************************************************************************************

IRQ_VBlank::
	push	af
	push	bc
	push	de
	push	hl

	; do nothing if not a CGB
	ld	a,[UserFlags]
	and	USER_F_ISCGB
	jr	z,.bodgejump	; big-time bodge; the real jump is >128 bytes away

	ld	de,StarCoordsZ1E
	ld	hl,StarCoordsZ1
	call	AssignSpritePos
	ld	de,StarCoordsZ2E
	ld	hl,StarCoordsZ2
	call	AssignSpritePos
	ld	de,StarCoordsZ3E
	ld	hl,StarCoordsZ3
	call	AssignSpritePos

	call	SPRITEDMA_ADDR	; update the OAM RAM

	; move the screen around a little bit to make it interesting
	ld	hl,sinedata
	ld	a,[ScanOffset]
	ld	e,a
	ld	a,0
	ld	d,a
	add	hl,de
	ld	a,h
	ld	[ScanSine],a
	ld	a,l
	ld	[ScanSine + 1],a

	ld	a,[ScanOffset]
	inc	a
	ld	[ScanOffset],a

	ld	a,[ScanSine]
	ld	h,a
	ld	a,[ScanSine + 1]
	ld	l,a

	ld	a, [hl]
	ld	[REG_SCY],a

	ld	hl,hsinedata
	ld	a,[ScanOffset2]
	ld	e,a
	xor	a
	ld	d,a
	add	hl,de
	ld	a,h
	ld	a,l
	ld	a,[hl]
	ld	[REG_SCX],a

	ld	a,[ScanOffset2]
	inc	a
	inc	a
	ld	[ScanOffset2],a

	; store the frame counter and update each z depth accordingly
	ld	a,[FrameCounter]
	inc	a
	ld	[FrameCounter],a
	and	3
	cp	2
	jp	z,.movez3
	and	1
	cp	1
	jp	z,.movez2
	jp	.movez1
.movez3:
	ld	hl,StarCoordsZ3
	ld	de,StarCoordsZ3E
	call	MoveStars
	jp	.movez1
.bodgejump:	jp	.vblankend
.movez2:
	ld	hl,StarCoordsZ2
	ld	de,StarCoordsZ2E
	call	MoveStars
.movez1:
	ld	hl,StarCoordsZ1
	ld	de,StarCoordsZ1E
	call	MoveStars

	; set up the LCDC interrupt
	ld	a,LCDC_INCREMENT	; set line at which lcdc interrupt occurs
	ld	[ScanLine],a
	ld	[REG_LYC],a
	ld	a,REG_STAT_F_LYC | REG_STAT_F_LYCF
	ld	[REG_STAT],a

	; move the sprite positions for the next LCDC interrupt
	ld	hl,StarCoordsZ1
	ld	b,LCDC_INCREMENT
	ld	c,NUM_STARS
	ld	de,4
.adjypos:
	ld	a,[hl]
	add	a,b
	ld	[hl],a
	add	hl,de

	dec	c
	jr	nz,.adjypos

	; set the V-Blank end flag
	ld	a,[UserFlags]
	or	USER_F_VBLANKEND
	ld	[UserFlags],a

.vblankend:
	pop	hl
	pop	de
	pop	bc
	pop	af

	reti


IRQ_LCDC::
	push	af
	push	bc
	push	de
	push	hl

	call	SPRITEDMA_ADDR	; update the OAM RAM

	; set the next LCDC interrupt
	ld	b,LCDC_INCREMENT
	ld	a,[ScanLine]
	add	a,b
	ld	[ScanLine], a
	ld	[REG_LYC],a

	; move the sprite positions for the next LCDC interrupt
	ld	hl,StarCoordsZ1
	ld	b,LCDC_INCREMENT
	ld	c,NUM_STARS
	ld	de,4
.adjypos:
	ld	a,[hl]
	add	a,b
	ld	[hl],a
	add	hl,de

	dec	c
	jr	nz,.adjypos

	pop	hl
	pop	de
	pop	bc
	pop	af

	reti


CreateStarCoords::
	call	Randomize
	ld	c,NUM_STARS
	ld	hl,StarCoordsZ1
.loop:
	; y coord
	call	RandomFrom0To31
	call	RandomFrom0To31
	add	a,16
	ld	[hl+],a
	; x coord
	call	RandomFrom0To159
	add	a,8
	ld	[hl+],a
	inc	hl	; skip depth
	inc	hl	; skip attribute (already 0)
	dec	c
	jr	nz,.loop

	; fix the Z depths
	ld	hl,StarCoordsZ1
	ld	a,1	; depth
	ld	b,3
.zloop0:
	ld	c,NUM_STARDEPTH
.zloop1:
	inc	hl	; skip y pos
	inc	hl	; skip x pos
	ld	[hl+],a	; set depth
	inc	hl	; skip attr
	dec	c
	jr	nz,.zloop1
	inc	a
	dec	b
	jr	nz,.zloop0

	ret


AssignSpritePos::
	ld	b,NUM_STARDEPTH * 4
.loop:
	ld	a,[de]
	inc	de
	ld	[hl+],a

	dec	b
	jr	nz,.loop

	ret


MoveStars::
	ld	c,NUM_STARDEPTH
.loop:
	inc	hl	; skip y pos
	inc	de	; skip y pos
	ld	a,[hl]	; get x pos
	inc	a
	cp	168	; 160 + 8
	jr	nz,.setxpos
	xor	a	; reset x to 0
.setxpos:
	ld	[hl+],a
	ld	[de],a
	inc	de
	inc	hl	; skip the z depth
	inc	de	; skip the z depth
	inc	hl	; skip the attributes
	inc	de	; skip the attributes
	dec	c
	jr	nz,.loop

	ret


SpriteDMAFunc::
	push	af
	ld	a,$C0
	ld	[REG_DMA],a
	; wait 160 ms
	ld	a,40
.oamloop:
	dec	a
	jr	nz,.oamloop
	pop	af

	ret
SpriteDMAFuncEnd::


ToggleCPUSpeed::
	di

	ld	hl,REG_IE
	ld	a,[hl]
	push	af

	; disable interrupts
	xor	a
	ld	[hl],a
	ld	[REG_IF],a

	ld	a,REG_P1_F_5 | REG_P1_F_4
	ld	[REG_P1],a

	ld	a,1
	ld	[REG_KEY1],a

	stop

	pop	af
	ld	[hl],a

	ei

	ret


TurnOffDisplay::
	ld	hl,REG_LCDC
	bit	7,[hl]	; is the LCD already off?
	ret	z	; yes, exit

	; disable the V-Blank interrupt if enabled
	ld	a,[REG_IE]
	push	af
	res	0,a
	ld	[REG_IE],a

	; wait for the next V-Blank
.waitvbl:
	ld	a,[REG_LY]
	cp	145
	jr	nz,.waitvbl

	; turn off the screen
	res	7,[hl]

	; restore the state of the V-Blank interrupt
	pop	af
	ld	[REG_IE],a

	ret


ClearAllBGAttrMap::
	ld	a,1
	ld	[REG_VBK],a	; set bank 1 (for attributes)

	xor	a
	ld	de,(MAP_BLOCKWIDTH - SCREEN_BLOCKWIDTH)
.memc0:
	ld	c,MAP_BLOCKHEIGHT
.memc1:
	ld	b,MAP_BLOCKWIDTH
.memc2:
	ld	[hl+],a

	dec	b
	jr	nz,.memc2

	add	hl,de

	dec	c
	jr	nz,.memc1

	xor	a
	ld	[REG_VBK],a	; reset back to bank 0

	ret


ClearAllBGMap::
	xor	a
	ld	e,SCREEN_BLOCKHEIGHT

.clearloop1:
	ld	d,MAP_BLOCKWIDTH
.clearloop2:
	ld	[hl+],a
	dec	d
	jr	nz,.clearloop2
	dec	e
	jr	nz,.clearloop1

	ret


; *** copy data to vram ***
; Entry: BC = Pointer to BG tile data
;        HL = Pointer to VRAM destination
;        DE = Number of bytes to copy
CopyToVRAM::
.copyloop:
	; wait for OAM RAM to become available for use
	ld	a,[REG_STAT]
	and	REG_STAT_F_OAM
	jr	nz,.copyloop

	ld	a,[bc]
	ld	[hl+],a
	inc	bc
	dec	de
	ld	a,d
	or	e
	jr	nz,.copyloop

	ret


; *** Set background palettes ***
; Entry: HL = Pointer to BG palettes
SetBGPalettes::
	ld	a,$80
	ld	[REG_BCPS],a

	ld	bc,$4069	; b = 64, c = $69
.loop:
	ld	a,[hl+]
	ld	[c],a
	dec	b
	jr	nz,.loop

	ret


; *** Set Object palettes ***
; Entry: HL = Pointer to OBJ (sprite) palettes
SetOBJPalettes::
	ld	a,$80
	ld	[REG_OCPS],a

	ld	bc,$406B	; b = 64, c = $6B
.loop:
	ld	a,[hl+]
	ld	[c],a
	dec	b
	jr	nz,.loop

	ret


; Entry: HL = Pointer to BG map destination
;        BC = Pointer to NULL-terminated string
;         D = Y coordinate (in tiles)
;         E = X coordinate (in tiles)
PrintText::
	; calculate the destination start address
	push	bc
	ld	bc,MAP_BLOCKWIDTH

	ld	a,d
.yloop
	cp	0
	jr	z,.addx
	add	hl,bc	; add a whole row
	dec	a
	jp	.yloop	

.addx
	ld	b,0
	ld	c,e
	add	hl,bc	; add x coordinate
	pop	bc

.textloop:
	ld	a,[bc]
	cp	0
	jr	z,.end	; found the NULL terminator, exit
	ld	[hl+],a
	inc	bc
	jp	.textloop

.end:
	ret


; Increase randomness of random number generator
Randomize::
	push	af
	push	bc

	ld	a,[REG_LY]
	ld	b,a
	ld	a,[REG_DIV]
	ld	c,a
	ld	a,[RandomSeed]
	add	c
	add	c
	add	c
	add	c
	ld	[RandomSeed],a

	pop	bc
	pop	af

	ret

;RandomFrom0To7:
;	call	RandomNumber
;	and	$07
;	ret

;RandomFrom0To15:
;	call	RandomNumber
;	and	$0E	; force even
;	ret

RandomFrom0To31:
	call	RandomNumber
	and	$1F
	ret

;RandomFrom0To35:
;	call	RandomNumber
;	cp	36
;	jr	nc,RandomFrom0To63
;	ret

;RandomFrom0To47:
;	call	RandomNumber
;	cp	48
;	jr	nc,RandomFrom0To63
;	ret

;RandomFrom0To63:
;	call	RandomNumber
;	and	$3F
;	ret

;RandomFrom0To127:
;	call	RandomNumber
;	and	$7F
;	ret

;RandomFrom0To143:
;	call	RandomNumber
;	cp	144
;	jr	nc,RandomFrom0To143
;	ret

RandomFrom0To159:
	call	RandomNumber
	cp	160
	jr	nc,RandomFrom0To159
	ret

RandomNumber:
	push	hl

	ld	hl,RandomSeed
	inc	[hl]
	ld	a,[hl]

	ld	hl,.table
	add	l
	ld	l,a
	jr	nc,.nocarry
	inc	h
.nocarry:
	ld	a,[hl]
	pop	hl

	ret

.table
	DB	$3B,$02,$B7,$6B,$08,$74,$1A,$5D,$21,$99,$95,$66,$D5,$59,$05,$42
	DB	$F8,$03,$0F,$53,$7D,$8F,$57,$FB,$48,$26,$F2,$4A,$3D,$E4,$1D,$D9
	DB	$9D,$DC,$2F,$F5,$92,$5C,$CC,$00,$73,$15,$BF,$B1,$BB,$EB,$9E,$2E
	DB	$32,$FC,$4B,$CD,$A7,$E6,$C2,$10,$11,$80,$52,$B2,$DA,$77,$4F,$EC
	DB	$13,$54,$64,$ED,$94,$8C,$C6,$9A,$19,$9F,$75,$FA,$AA,$8D,$FE,$91
	DB	$01,$23,$07,$C1,$40,$18,$51,$76,$3C,$BD,$2A,$88,$2D,$F1,$8A,$72
	DB	$F6,$98,$35,$97,$68,$93,$B3,$0C,$82,$4E,$CB,$39,$D8,$5F,$C7,$D4
	DB	$CE,$AE,$6D,$A3,$7C,$6A,$B8,$A6,$6F,$5E,$E5,$1B,$F4,$B5,$3A,$14
	DB	$78,$FD,$D0,$7A,$47,$2C,$A8,$1E,$EA,$2B,$9C,$86,$83,$E1,$7B,$71
	DB	$F0,$FF,$D1,$C3,$DB,$0E,$46,$1C,$C9,$16,$61,$55,$AD,$36,$81,$F3
	DB	$DF,$43,$C5,$B4,$AF,$79,$7F,$AC,$F9,$37,$E7,$0A,$22,$D3,$A0,$5A
	DB	$06,$17,$EF,$67,$60,$87,$20,$56,$45,$D7,$6E,$58,$A9,$B0,$62,$BA
	DB	$E3,$0D,$25,$09,$DE,$44,$49,$69,$9B,$65,$B9,$E0,$41,$A4,$6C,$CF
	DB	$A1,$31,$D6,$29,$A2,$3F,$E2,$96,$34,$EE,$DD,$C0,$CA,$63,$33,$5B
	DB	$70,$27,$F7,$1F,$BE,$12,$B6,$50,$BC,$4D,$28,$C8,$84,$30,$A5,$4C
	DB	$AB,$E9,$8E,$E8,$7E,$C4,$89,$8B,$0B,$24,$85,$3E,$38,$04,$D2,$90

sinedata:
db  0,  1,  2,  3,  3,  4,  5,  6
db  7,  8,  8,  9,  10,  10,  11,  11
db  12,  12,  12,  13,  13,  13,  13,  13
db  13,  13,  13,  13,  13,  12,  12,  11
db  11,  10,  9,  8,  8,  7,  6,  5
db  3,  2,  1,  0, -2, -3, -5, -6
db -8, -9, -11, -13, -14, -16, -18, -20
db -21, -23, -25, -27, -29, -30, -32, -34
db -36, -38, -39, -41, -43, -44, -46, -47
db -49, -50, -52, -53, -54, -56, -57, -58
db -59, -60, -60, -61, -62, -62, -63, -63
db -63, -63, -63, -63, -63, -63, -63, -62
db -62, -61, -60, -59, -58, -57, -56, -55
db -54, -52, -51, -49, -47, -45, -44, -42
db -40, -38, -35, -33, -31, -29, -26, -24
db -21, -19, -16, -14, -11, -9, -6, -3
db -1,  2,  5,  7,  10,  13,  15,  18
db  20,  23,  25,  27,  30,  32,  34,  37
db  39,  41,  43,  45,  46,  48,  50,  51
db  53,  54,  55,  57,  58,  59,  60,  60
db  61,  62,  62,  63,  63,  63,  63,  63
db  63,  63,  63,  62,  62,  61,  60,  60
db  59,  58,  57,  56,  55,  53,  52,  51
db  49,  48,  46,  45,  43,  42,  40,  38
db  36,  35,  33,  31,  29,  28,  26,  24
db  22,  20,  19,  17,  15,  13,  12,  10
db  8,  7,  5,  4,  2,  1,  0, -2
db -3, -4, -5, -6, -7, -8, -9, -9
db -10, -11, -11, -12, -12, -12, -13, -13
db -13, -13, -13, -13, -13, -13, -12, -12
db -12, -11, -11, -10, -9, -9, -8, -7
db -7, -6, -5, -4, -3, -3, -2, -1

db  0,  1,  2,  3,  3,  4,  5,  6
db  7,  8,  8,  9,  10,  10,  11,  11
db  12,  12,  12,  13,  13,  13,  13,  13
db  13,  13,  13,  13,  13,  12,  12,  11
db  11,  10,  9,  8,  8,  7,  6,  5
db  3,  2,  1,  0, -2, -3, -5, -6
db -8, -9, -11, -13, -14, -16, -18, -20
db -21, -23, -25, -27, -29, -30, -32, -34
db -36, -38, -39, -41, -43, -44, -46, -47
db -49, -50, -52, -53, -54, -56, -57, -58
db -59, -60, -60, -61, -62, -62, -63, -63
db -63, -63, -63, -63, -63, -63, -63, -62
db -62, -61, -60, -59, -58, -57, -56, -55
db -54, -52, -51, -49, -47, -45, -44, -42
db -40, -38, -35, -33, -31, -29, -26, -24
db -21, -19, -16, -14, -11, -9, -6, -3
db -1,  2,  5,  7,  10,  13,  15,  18
db  20,  23,  25,  27,  30,  32,  34,  37
db  39,  41,  43,  45,  46,  48,  50,  51
db  53,  54,  55,  57,  58,  59,  60,  60
db  61,  62,  62,  63,  63,  63,  63,  63
db  63,  63,  63,  62,  62,  61,  60,  60
db  59,  58,  57,  56,  55,  53,  52,  51
db  49,  48,  46,  45,  43,  42,  40,  38
db  36,  35,  33,  31,  29,  28,  26,  24
db  22,  20,  19,  17,  15,  13,  12,  10
db  8,  7,  5,  4,  2,  1,  0, -2
db -3, -4, -5, -6, -7, -8, -9, -9
db -10, -11, -11, -12, -12, -12, -13, -13
db -13, -13, -13, -13, -13, -13, -12, -12
db -12, -11, -11, -10, -9, -9, -8, -7
db -7, -6, -5, -4, -3, -3, -2, -1

hsinedata:
db  0,  0,  1,  1,  2,  2,  3,  3
db  4,  4,  5,  5,  5,  6,  6,  6
db  6,  7,  7,  7,  7,  7,  7,  7
db  7,  7,  7,  7,  7,  7,  6,  6
db  6,  6,  5,  5,  4,  4,  3,  3
db  2,  1,  1,  0, -1, -2, -3, -3
db -4, -5, -6, -7, -8, -9, -10, -11
db -12, -13, -14, -15, -16, -17, -18, -19
db -20, -21, -22, -23, -24, -25, -26, -26
db -27, -28, -29, -29, -30, -31, -31, -32
db -33, -33, -34, -34, -34, -35, -35, -35
db -35, -35, -35, -35, -35, -35, -35, -35
db -34, -34, -33, -33, -32, -32, -31, -31
db -30, -29, -28, -27, -26, -25, -24, -23
db -22, -21, -20, -18, -17, -16, -15, -13
db -12, -10, -9, -8, -6, -5, -3, -2
db  0,  1,  3,  4,  6,  7,  8,  10
db  11,  13,  14,  15,  17,  18,  19,  20
db  21,  23,  24,  25,  26,  27,  28,  29
db  29,  30,  31,  31,  32,  33,  33,  34
db  34,  34,  35,  35,  35,  35,  35,  35
db  35,  35,  35,  35,  34,  34,  34,  33
db  33,  32,  32,  31,  30,  30,  29,  28
db  27,  27,  26,  25,  24,  23,  22,  21
db  20,  19,  18,  17,  16,  15,  14,  13
db  12,  11,  10,  9,  8,  7,  6,  6
db  5,  4,  3,  2,  1,  1,  0, -1
db -2, -2, -3, -3, -4, -4, -5, -5
db -6, -6, -6, -7, -7, -7, -7, -7
db -7, -7, -7, -7, -7, -7, -7, -7
db -6, -6, -6, -6, -5, -5, -5, -4
db -4, -3, -3, -2, -2, -1, -1,  0
