        format binary as 'rom'

RAM_START                       EQU     16 * 1024                ;first address of RAM
RAM_SIZE                        EQU     1 * 1024                 ;size of ram
RAM_TOP                         EQU     RAM_START + RAM_SIZE - 1 ;last address of 16k RAM
ROM_TOP                         EQU     8 * 1024 - 1             ;last address of 8k ROM
FRAME_BUFFER_WIDTH              EQU     64
FRAME_BUFFER_HEIGHT             EQU     48
FRAME_BUFFER_WIDTH_PIXELS       EQU     FRAME_BUFFER_WIDTH * 8
FRAME_BUFFER_HEIGHT_PIXELS      EQU     FRAME_BUFFER_HEIGHT * 8
SCREEN_WIDTH                    EQU     34
SCREEN_HEIGHT                   EQU     24
SCREEN_WIDTH_PIXELS             EQU     SCREEN_WIDTH * 8
SCREEN_HEIGHT_PIXELS            EQU     SCREEN_HEIGHT * 8
TOTAL_RASTERS                   EQU     312                     ;PAL=312, NTSC=262
VSYNC_RASTERS                   EQU     5
VISIBLE_RASTERS                 EQU     SCREEN_HEIGHT_PIXELS
TOP_MARGIN                      EQU     7 * 8 - 2               ;PAL=7, NTSC=4
BOTTOM_MARGIN                   EQU     TOTAL_RASTERS - VISIBLE_RASTERS - VSYNC_RASTERS - TOP_MARGIN - 2

;-------------------------------------
virtual at RAM_START
        FrameCounter            dw      ?
        FrameBufferPtr          dw      ?
        ScreenReturn            dw      ?
        IRQStackPtr             dw      ?
        IRQReturnAddr           dw      ?
        YOffset                 db      ?       ;nr of rasters of first char line to show (1-8)
        VisibleRasters          db      ?
        FrameBufferWidth        dw      ?
        BottomEmptyRasters      db      ?
        TopEmptyRasters         db      ?
        XPos                    db      ?
        XAdd                    db      ?
        XAddBuffer              dw      ?
        YPos                    db      ?
        YAdd                    db      ?
        YAddBuffer              dw      ?
        label YOffsetAndVisibleRasters word at YOffset
end virtual

assert VisibleRasters = YOffset + 1
assert YOffset = YOffsetAndVisibleRasters

;-------------------------------------
        org     0
ROM_START:
        out     ($fd),a                 ;NMI-generator off
        ld      sp,RAM_TOP              ;set SP to top of RAM

        ld      hl,FRAME_BUFFER
        ld      (FrameBufferPtr),hl
        ld      a,VISIBLE_RASTERS
        ld      (VisibleRasters),a
        ld      a,8
        ld      (YOffset),a
        ld      hl,FRAME_BUFFER_WIDTH
        ld      (FrameBufferWidth),hl
        ld      a,TOP_MARGIN
        ld      (TopEmptyRasters),a
        ld      a,BOTTOM_MARGIN
        ld      (BottomEmptyRasters),a

        ld      a,FontShift0 shr 8      ;character set address
        ld      i,a                     ;i register is used to generate characters
        im      1                       ;Interrupt Mode 1
        call    START_SCREEN            ;start display generation
        jp      MAIN                    ;and jump to move the screen around

;-------------------------------------
;###filler
        repeat  IRQ - $
                db      0
        end repeat

;-------------------------------------
; requires:
; A = timing for INT, used in R, number of visible chars + 1
; HL = pointer to start of FRAME_BUFFER
; DE = nr of chars per row, can be larger than nr of visible lines (32)
; B = nr of visible rasters (char rows * 8)
; C = 8, the char line counter
; should be 58 cycles total per int, with 34 chars/row
        org     $38                     ;IRQ-vector
IRQ:
        dec     b                       ;4 the visible raster counter
        jr      z,IRQ_DONE              ;12/7 end when all rasters are drawn
        dec     c                       ;4 the char line counter
        jr      nz,CONT_ROWS            ;12/7 draw char line 1-7
NEW_ROW:                                ;=22 cycles already
        set     3,c                     ;8 C == 0, so set C = 8
        add     hl,de                   ;11 calculate new framebuffer-pointer
                                        ;=41 cycles already
DRAW_ROW:
        ld      r,a                     ;9 Load R with initial value
        ei                              ;4 Enable Interrupts, also interrupt aknowledge
        jp      (hl)                    ;4 jump to the echo display file
                                        ;=58 total
CONT_ROWS:                              ;=27 cycles already
        nop                             ;4 dummy timing
        jp      DRAW_ROW                ;10 go and do the real work
                                        ;=41 cycles already
IRQ_DONE:
        ld      hl,(IRQStackPtr)
        ld      sp,hl                   ;restore ordinary stack
BOTTOM_BLANK:
        ld      A,(BottomEmptyRasters)
        ld      hl,SCREEN_LOOP
        ld      (ScreenReturn),hl
        ex      af,af                   ;place RasterCounter in a'

        ld      hl,(FrameCounter)
        inc     hl                      ;increment FrameCounter
        ld      (FrameCounter),hl

        out     ($fe),a                 ;enable the NMI generator.

        pop     hl
        pop     de
        pop     bc
        pop     af
        ret                             ; return - end of interrupt.  Return is to
                                        ; user's program - BASIC or machine code.
                                        ; which will be interrupted by every NMI.

;-------------------------------------
;###filler
        repeat  NMI - $
                db      0
        end repeat

;-------------------------------------
        org     $66                     ;NMI-vector
NMI:
        ex      af,af'                  ;4 switch in RasterCounter
        dec     a                       ;4
        jr      z,NMI_CONT              ;12/7 when (RasterCounter == 0)
NMI_RET:
        ex      af,af'                  ;4 switch out RasterCounter
        ret                             ;10 return to User application for a while.
NMI_CONT:
        ex      af,af'                  ;4 restore the main accumulator.
        push    af
        push    bc
        push    de
        push    hl

        halt                            ;1 HALT synchronizes with NMI.
        out     ($fd),a                 ;11 Stop the NMI generator.

        ld      hl,(ScreenReturn)       ;16
        jp      (hl)                    ;4 generate picture, blank lines or vsync

;-------------------------------------
START_SCREEN:
        push    af
        push    bc
        push    de
        push    hl
        ex      af,af'

SCREEN_LOOP:
        ld      b,(207 * VSYNC_RASTERS) / 13
        in      a,($fe)                 ;start VSync
GEN_VSYNC:
        djnz    GEN_VSYNC               ;13/8
        nop                             ;4
        out     ($ff),a                 ;end VSync

        LD      A,(YOffset)
        ld      hl,TOP_BLANK
        ld      (ScreenReturn),hl
        ex      af,af'                  ;place RasterCounter in A'

        out     ($fe),a                 ;enable the NMI generator.

        pop     hl
        pop     de
        pop     bc
        pop     af
        ret                             ; return - end of interrupt.  Return is to
                                        ; user's program - BASIC or machine code.
                                        ; which will be interrupted by every NMI.

;-------------------------------------
TOP_BLANK:
repeat  5
        nop                             ;4 dummy timing
        inc     hl                      ;6 dummy timing
end repeat

        ld      hl,SCREEN               ;10
        ld      (ScreenReturn),hl       ;16

        ld      a,(YOffset)             ;13
        ld      b,a                     ;4
        ld      a,(TopEmptyRasters)     ;13
        sub     b                       ;4
        ld      b,a                     ;4

        in      a,($fe)                 ;11 start VSync to reset LINECNTR in ULA
        out     ($fe),a                 ;11 enable the NMI generator. Also ends VSync

        ld      a,b                     ;remaining empty rasters before screen
        ex      af,af'                  ;place RasterCounter in A'

        pop     hl
        pop     de
        pop     bc
        pop     af
        ret                             ; return - end of interrupt.  Return is to
                                        ; user's program - BASIC or machine code.
                                        ; which will be interrupted by every NMI.

;-------------------------------------
SCREEN:
        nop                             ;4 dummy timing
        nop                             ;4 dummy timing
        nop                             ;4 dummy timing

        ld      hl,BOTTOM_BLANK         ;10 not really necessary
        ld      (IRQReturnAddr),hl      ;16 save return address, not really necessary
        ld      (IRQStackPtr),sp        ;20 save ordinary SP
        ld      hl,$1fff                ;10 set SP to top of ROM
        ld      sp,hl                   ;6 the cascaded IRQ PUSHes will have no effect since they are "written" to ROM
        ld      hl,(FrameBufferPtr)     ;20 set HL to the Display File from D_FILE
        set     7,h                     ;8 set bit 15 to address the echo display
                                        ;above can be removed if FrameBufferPtr has this bit set already
        ld      de,(FrameBufferWidth)   ;20
        ld      bc,(YOffsetAndVisibleRasters);20 b = VisibleRasters, c = YOffset
        ld      a,-SCREEN_WIDTH-1       ;7
        jp      DRAW_ROW                ;10

;-------------------------------------
MAIN:
;       XPos = 0
;       YPos = 0
        ld      a,0
        ld      (XPos),a
        ld      (YPos),a
;       XAdd = 1
;       YAdd = 1
        ld      a,1
        ld      (XAdd),a
        ld      (YAdd),a
;       XAddBuffer = 1
        ld      hl,1
        ld      (XAddBuffer),hl
;       YAddBuffer = FRAME_BUFFER_WIDTH
        ld      hl,FRAME_BUFFER_WIDTH
        ld      (YAddBuffer),hl

SYNC_FRAME:
        call    WAIT_FOR_SYNC
;       XPos += XAdd
        ld      a,(XAdd)
        ld      b,a
        ld      a,(XPos)
        add     a,b
        ld      (XPos),a
;       SetFontPtr(((XPos and 7) * 2) + (FontShift0 shr 8))
        and     7
        sla     a
        add     a,FontShift0 shr 8
        ld      i,a
;       if (XAdd < 0)
        ld      a,(XAdd)
        cp      0
        jp      p,X_ADD_IS_POS
;               if ((XPos and 7) == 7)
                ld      a,(XPos)
                and     7
                cp      7
                jp      nz,X_OFFSET_0_CHECKED
;                       FrameBufferPtr += XAddBuffer
                        jp      FrameBufferModifyX
;       else
X_ADD_IS_POS:
;               if ((XPos and 7) == 0)
                ld      a,(XPos)
                and     7
                jp      nz,X_OFFSET_0_CHECKED
;                       FrameBufferPtr += XAddBuffer
FrameBufferModifyX:
                        ld      hl,(FrameBufferPtr)
                        ld      bc,(XAddBuffer)
                        add     hl,bc
                        ld      (FrameBufferPtr),hl
X_OFFSET_0_CHECKED:
;       if (XPos == 0)
        ld      a,(XPos)
        cp      0
        jp      nz,X_POS_MIN_DONE
;               XAdd = 1
                ld      a,1
                ld      (XAdd),a
;               XAddBuffer = 1
                ld      hl,1
                ld      (XAddBuffer),hl
X_POS_MIN_DONE:
;       if (XPos == (FRAME_BUFFER_WIDTH_PIXELS - SCREEN_WIDTH_PIXELS - 1))
        ld      a,(XPos)
        cp      FRAME_BUFFER_WIDTH_PIXELS - SCREEN_WIDTH_PIXELS - 1 ;-1 to make the loop-pattern longer
        jp      nz,X_POS_MAX_DONE
;               XAdd = -1
                ld      a,-1
                ld      (XAdd),a
;               XAddBuffer = -1
                ld      hl,-1
                ld      (XAddBuffer),hl
X_POS_MAX_DONE:
;        jp     SYNC_FRAME        ;debug
;       YPos += YAdd
        ld      a,(YAdd)
        ld      b,a
        ld      a,(YPos)
        add     a,b
        ld      (YPos),a
;       YOffset = 8 - (YPos and 7)
        ld      a,(YPos)
        and     7
        neg
        add     a,8
        ld      (YOffset),a


;       if (YAdd < 0)
        ld      a,(YAdd)
        cp      0
        jp      p,Y_ADD_IS_POS
;               if (YOffset == 1)
                ld      a,(YOffset)
                cp      1
                jp      nz,Y_OFFSET_0_DONE
;                       FrameBufferPtr += YAddBuffer
                        jp      FRAME_BUFFER_MODIFY_Y
Y_ADD_IS_POS:
;       else
;               if (YOffset == 8)
                ld      a,(YOffset)
                cp      8
                jp      nz,Y_OFFSET_0_DONE
;                       FrameBufferPtr += YAddBuffer
FRAME_BUFFER_MODIFY_Y:
                        ld      hl,(FrameBufferPtr)
                        ld      bc,(YAddBuffer)
                        add     hl,bc
                        ld      (FrameBufferPtr),hl
Y_OFFSET_0_DONE:
;       if (YPos == 0)
        ld     a,(YPos)
        cp     0
        jp     nz,Y_POS_MIN_DONE
;               YAdd = 1
                ld      a,1
                ld      (YAdd),a
;               YAddBuffer = FRAME_BUFFER_WIDTH
                ld      hl,FRAME_BUFFER_WIDTH
                ld      (YAddBuffer),hl
Y_POS_MIN_DONE:
;       if (YPos == (FRAME_BUFFER_HEIGHT_PIXELS - SCREEN_HEIGHT_PIXELS))
        ld      a,(YPos)
        cp      FRAME_BUFFER_HEIGHT_PIXELS - SCREEN_HEIGHT_PIXELS
        jp      nz,Y_POS_MAX_DONE
;               YAdd = -1
                ld      a,-1
                ld      (YAdd),a
;               YAddBuffer = -FRAME_BUFFER_WIDTH
                ld      hl,-FRAME_BUFFER_WIDTH
                ld      (YAddBuffer),hl
Y_POS_MAX_DONE:
        jp      SYNC_FRAME

;-------------------------------------
WAIT_FOR_SYNC:
        ld      hl,FrameCounter
        ld      a,(hl)
WAIT_FOR_SYNC0:
        cp      (hl)
        jr      z,WAIT_FOR_SYNC0
        ret

;-------------------------------------
        include 'font.inc'

;-------------------------------------
FRAME_BUFFER:
        include 'screen.inc'
FRAME_BUFFER_END:

;-------------------------------------
;make a full 8k ROM
        repeat  ROM_TOP - $ + 1
            db      0
        end repeat