#include "config.h"

#ifdef HAVE_SCHED_H
#include <sched.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>

#include "crap_types.h"
#include "crap_lib.h"
#include "crap_video.h"
#include "crap_mod.h"
#include "crap_font.h"
#include "crap_file.h"
#include "crap_png.h"
#include "block_font.h"

#include "credits.h"
#include "interference.h"
#include "ball.h"
#include "rotozoom.h"
#include "voxel.h"
#include "twistbar.h"

int realtime_process ()
{
#ifdef HAVE_SCHED_H
    struct sched_param sp;

    printf("trying to set minimal SCHED_RR scheduling priority\n");
    sp.sched_priority = sched_get_priority_min ( SCHED_RR );
    if ( sched_setscheduler ( 0, SCHED_RR, &sp ) )
    {
        fprintf ( stderr, "could not enable realtime scheduling, continuing.\n" );
        return -1;
    };
#endif
    return 0;
}

int renice_process ()
{
    if ( nice ( -20 ) )
    {
        fprintf ( stderr, "could not renice process by -20, continuing.\n" );
        return -1;
    };
    return 0;
}

int mod_fake_pattern_lengths[] = { 16, 64, 64, 16, 64, 64, 16, 64, 64, 64, 64 };

int
main ( int argc, char* argv[] )
{
  int logo_y_pos = 200;
  int note;
  
  Ball_data* ball_data;
  Rotozoom_data* rotozoom_data;
  Voxel_data* voxel_data;
  Twistbar_data* twistbar_data;
  Interference_data* interference_data;
  Credits_data* credits_data;

  int end_intro = 0;
  
  Image logo_image;
  Image image;
  Palette palette;
  
  Image *root_image;
  Image effect_image;
  
  printf("\n");
  printf("another way to scroll "VERSION"\n");
  printf("Copyright (C) 2000, Tijs van Bakel and Jorik Blaas\n");
  printf("\n");
  printf("This program is free software; you may redistribute it under the terms of\n");
  printf("the GNU General Public License. (See the file COPYING for more details).\n");
  printf("This program has absolutely no warranty.\n");
  printf("\n");
  
  if (argc <= 1 )
    realtime_process ();

  crap_video_init ( 320, 200 );

  load_block_font ();

  crap_mod_init ( crap_file_find ("chip.xm") , 6 );
  crap_mod_fake_pattern_lengths ( 11, mod_fake_pattern_lengths );
  
  root_image =  crap_video_root_image();
  crap_image_fill ( root_image, 0 );
  crap_image_subimage ( root_image, &effect_image, 0, 0, 320, 180 );
  
  crap_png_load ( &image, &palette, "logo.png" );
  crap_image_subimage ( &image, &logo_image, 0, 0, 320, 20 );

  ball_data = create_fx_ball ( &effect_image, 750 );
  rotozoom_data = create_fx_rotozoom ( &effect_image );
  interference_data = create_fx_interference ( &effect_image );
  twistbar_data = create_fx_twistbar ( &effect_image );
  voxel_data = create_fx_voxel ( &effect_image );
  credits_data = create_fx_credits ( &effect_image );

  crap_setpalette ( &palette, 0, 256 );
  crap_mod_start_playing ();

  while (!end_intro)
    {

      note = crap_mod_get_patpos();
      #define BARS 4
      switch ( crap_mod_get_songpos() )
	{
	case 0:
	  break;
	case 1:
	  /* bars appear */
	  fx_twistbar ( &effect_image, twistbar_data,
			TWISTBAR_APPEAR,
			(note * BARS) / 64 + 1,
			TWISTBAR_NO_TEXT );
	  break;
	case 2:
	  /* bars are there, text is added */

	  fx_twistbar ( &effect_image, twistbar_data,
			TWISTBAR_LEAVE,
			range ( BARS - (( note ) * BARS) / 28, 0, BARS) ,
			TWISTBAR_TEXT );
	  break;
	case 3:
	  /* bars leave the scene */
	  fx_twistbar ( &effect_image, twistbar_data,
			TWISTBAR_LEAVE, 0, TWISTBAR_TEXT );
	  break;
	case 4:
	  fx_ball ( &effect_image, ball_data,
		    note > 15, ( note > 20 ) && ( (note & 15) == 3 ) );

	  if ( logo_y_pos > 180 )
	    {
	      logo_y_pos--;
	      crap_image_blit ( root_image, 0, logo_y_pos, &logo_image );
	    }
	  break;
	case 5:
	  fx_ball ( &effect_image, ball_data,
		    1, (note & 15) == 3 );
	  break;
	case 6:
	  fx_rotozoom ( &effect_image, rotozoom_data );
	  break;
	case 7:
	  fx_voxel ( &effect_image, voxel_data,
		     (note & 15) == 11 );
	  break;
	case 8:
	  fx_interference ( &effect_image, interference_data,
			    (note & 15) == 10, ( note & 7 ) == 4,
			    (note > 45) );
	  if ( note > 56 )
	    {
	      if ( logo_y_pos < 200 )
		{
		  logo_y_pos++;
		  crap_image_blit ( root_image, 0, logo_y_pos, &logo_image );
		  crap_image_rect_fill ( root_image, 0, 0, logo_y_pos-1, root_image->width, 1 );
		}
	    }
	  break;
	case 9:
	  crap_mod_set_volume ( 127-note*2 );
	  if ( note == 63 )
	    end_intro = 1;
	  fx_credits ( &effect_image, credits_data );
	  break;
	default:
	  end_intro = 1;
	  break;
	};
      
      crap_mod_update();
      
      crap_video_waitretrace();
      crap_video_blit ( root_image );
      
      if ( (crap_video_keypress()) && ( crap_video_getkey() == CRAP_KEY_ESCAPE ) )
	break;
    }

  crap_video_done ();
  crap_mod_done ();

  printf("\n");
  printf("(defcredits :code '(avoozl smoke)\n");
  printf("            :gfx '(smoke)\n");
  printf("            :guest-music '(floppi))\n");
  printf("\n");
  printf("(defemail (('smoke \"smoke@casema.net\")\n");
  printf("           ('avoozl \"jrk@panic.et.tudelft.nl\")\n");
  printf("           ('floppi \"tero.laihanen@sci.fi\"))\n");
  printf("\n");

  return 0;
}
