/*************************************************************************************************
*
*	Title:	DDraw.cpp
*	Desc:	Provides a DirectDraw Virtual Video driver
*	
*	Note:	
*			
**************************************************************************************************/

#include <Windows.h>
#include <memory.h>
#include <stdio.h>
#include <ddraw.h>
#include "Winmain.h"
#include "VVideo.h"

/*************************************************************************************************/
// Defines
/*************************************************************************************************/

#define MAX_MODES 64

/*************************************************************************************************/
// External Data
/*************************************************************************************************/

/*************************************************************************************************/
// Global Data
/*************************************************************************************************/

/*************************************************************************************************/
// Module Data
/*************************************************************************************************/

static Color32_t	*m_VPage = NULL;

// DDraw surfaces
static LPDIRECTDRAW	m_DDraw = NULL;
static LPDIRECTDRAWSURFACE m_FrontBuffer = NULL;
static LPDIRECTDRAWSURFACE m_BackBuffer = NULL;

// color space masks
static unsigned int	m_RedMask, m_RedRShift, m_RedLShift;
static unsigned int	m_GreenMask, m_GreenRShift, m_GreenLShift;
static unsigned int	m_BlueMask, m_BlueRShift, m_BlueLShift;

// list of avaliable modes
static struct
{
	int Width;
	int Height;
	int Bpp;
} m_ModeList[ MAX_MODES ];
static int m_NumberModes;

/*************************************************************************************************/
// Functions
/*************************************************************************************************/

static HRESULT WINAPI DDraw_DisplayCallback( LPDDSURFACEDESC lpDDSurfaceDesc, LPVOID lpContext );
static int GetLowestBit( unsigned int Value );
static int GetHighestBit( unsigned int Value );

/*************************************************************************************************
*
*	Function:	DDraw_Open()
*
*	Desc:		Creates our Surface
*
*	Notes:		
*
***************************************************************************************************/
int DDraw_Open( void )
{
	int i;
	HRESULT hr;
	int Width, Height, Bpp;
	int ModeFound;
	DDSURFACEDESC ddsd;
	DDSCAPS ddcs;
	DDPIXELFORMAT PixelFormat;

	// init vars
	Width = 320;
	Height = 240;
	Bpp = 16;

	// create DDraw object
	hr = DirectDrawCreate( NULL, &m_DDraw, NULL );
	if (FAILED(hr)) return 1;

	// search though all avaliable modes to find 320x240
	m_NumberModes = 0;
	hr = m_DDraw->EnumDisplayModes( DDEDM_STANDARDVGAMODES, NULL, NULL, DDraw_DisplayCallback);
	if (FAILED(hr)) goto error;
	
	// search for our desired mode
	ModeFound = 0;
	for (i=0; i < m_NumberModes; i++)
	{
		if ( (m_ModeList[i].Width == Width) && (m_ModeList[i].Height == Height) && (m_ModeList[i].Bpp == Bpp) )
		{
			ModeFound = 1;
		}
	}

	// was one found?
	if (!ModeFound) goto error;

	// so the mode exists, set the cooperative level so we can go fullscreen
	hr = m_DDraw->SetCooperativeLevel( g_hWnd, DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN );
	if (FAILED(hr)) goto error;

	// change the display mode
	hr = m_DDraw->SetDisplayMode( Width, Height, Bpp);
	if (FAILED(hr)) goto error;

	// create front and back buffers
	memset( &ddsd, 0, sizeof(ddsd) );
	ddsd.dwSize = sizeof(ddsd);
	ddsd.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
	ddsd.dwBackBufferCount = 1;
	ddsd.ddsCaps.dwCaps = DDSCAPS_FLIP | DDSCAPS_COMPLEX | DDSCAPS_VIDEOMEMORY | DDSCAPS_PRIMARYSURFACE;
	hr = m_DDraw->CreateSurface( &ddsd, &m_FrontBuffer, NULL );
	if (FAILED(hr)) goto error;
	
	// get back buffer
	memset( &ddcs, 0, sizeof(ddcs) );
	ddcs.dwCaps = DDSCAPS_BACKBUFFER;
	hr = m_FrontBuffer->GetAttachedSurface( &ddcs, &m_BackBuffer);
	if (FAILED(hr)) goto error;

	// get info on the back buffer pixel format
	memset( &PixelFormat, 0, sizeof(PixelFormat) );
	PixelFormat.dwSize = sizeof(PixelFormat);
	hr = m_FrontBuffer->GetPixelFormat( &PixelFormat );
	if (FAILED(hr)) goto error;

	// setup color space shifts/masks
	m_RedMask = PixelFormat.dwRBitMask;
	m_RedRShift = 8 - (GetHighestBit( PixelFormat.dwRBitMask ) - GetLowestBit( PixelFormat.dwRBitMask ));
	m_RedLShift = GetLowestBit( PixelFormat.dwRBitMask );

	m_GreenMask = PixelFormat.dwGBitMask;
	m_GreenRShift = 8 - (GetHighestBit( PixelFormat.dwGBitMask ) - GetLowestBit( PixelFormat.dwGBitMask ));
	m_GreenLShift = GetLowestBit( PixelFormat.dwGBitMask );

	m_BlueMask = PixelFormat.dwBBitMask;
	m_BlueRShift = 8 - (GetHighestBit( PixelFormat.dwBBitMask ) - GetLowestBit( PixelFormat.dwBBitMask ));
	m_BlueLShift = GetLowestBit( PixelFormat.dwBBitMask );

	// allocate vpage
	m_VPage = (Color32_t *)malloc( 320*240*sizeof(Color32_t) );
	if (!m_VPage) goto error;

	return 0;

error:
	OutputDebugString( "DDraw Error" );

	if (m_BackBuffer)
	{
		m_BackBuffer->Release();
		m_BackBuffer = NULL;
	}

	if (m_FrontBuffer)
	{
		m_FrontBuffer->Release();
		m_FrontBuffer = NULL;
	}

	if (m_DDraw)
	{
		m_DDraw->SetCooperativeLevel( g_hWnd, DDSCL_NORMAL );
		m_DDraw->RestoreDisplayMode();
		m_DDraw->Release();
		m_DDraw = NULL;
	}

	if (m_VPage)
	{
		free( m_VPage );
		m_VPage = NULL;
	}
	
	return 1;
}

/*************************************************************************************************
*
*	Function:	DDraw_Close()
*
*	Desc:		Releases all resources
*
*	Notes:		
*
***************************************************************************************************/
void DDraw_Close( void )
{
	// release back buffer
	if (m_BackBuffer)
	{
		m_BackBuffer->Release();
		m_BackBuffer = NULL;
	}

	// relese front buffer
	if (m_FrontBuffer)
	{
		m_FrontBuffer->Release();
		m_FrontBuffer = NULL;
	}

	// release DDraw
	if (m_DDraw)
	{
		// reset coop
		m_DDraw->SetCooperativeLevel( g_hWnd, DDSCL_NORMAL );
		m_DDraw->RestoreDisplayMode();
		m_DDraw->Release();
		m_DDraw = NULL;
	}

	// release virtual page
	if (m_VPage)
	{
		free( m_VPage );
		m_VPage = NULL;
	}
}

/*************************************************************************************************
*
*	Function:	DDraw_Flip()
*
*	Desc:		Shows the VPage to the world
*
*	Notes:		
*
***************************************************************************************************/
void DDraw_Flip( void )
{
	HRESULT hr;
	DDSURFACEDESC ddsd;
	unsigned char *Scanline;
	unsigned short *Dest;
	Color32_t *Src;
	int x, y;
	unsigned r, g, b;

	// lock the surface
	memset( &ddsd, 0, sizeof(ddsd) );
	ddsd.dwSize = sizeof(ddsd);
	hr = m_BackBuffer->Lock( NULL, &ddsd, DDLOCK_SURFACEMEMORYPTR | DDLOCK_WAIT, NULL);
	if (!FAILED(hr))
	{
		// setup pointers
		Scanline = (unsigned char *)ddsd.lpSurface;
		Src = m_VPage;

		// for all scanlines
		for (y=0; y < 240; y++)
		{
			// for all pixels in the scan
			Dest = (unsigned short *)Scanline;
			for (x=0; x < 320; x++)
			{
				// conver 888 to whatever format the display is in
				r = (((Src->r >> m_RedRShift) << m_RedLShift) & m_RedMask);
				g = (((Src->g >> m_GreenRShift) << m_GreenLShift) & m_GreenMask);
				b = (((Src->b >> m_BlueRShift) << m_BlueLShift) & m_BlueMask);
				Src++;

				// write to back buffer
				*Dest = r + g + b;
				Dest++;
			}
			// NOTE: lPith might NOT be equal to 320*2.
			Scanline += ddsd.lPitch;
		}
		hr = m_BackBuffer->Unlock(NULL);
		if (FAILED(hr)) OutputDebugString("BackBuffer->Unlock() Failed!");
	}

	// page flip it
	hr = m_FrontBuffer->Flip(NULL, DDFLIP_WAIT);
	if (FAILED(hr)) OutputDebugString("FrontBuffer->Flip() Failed!");
}

/*************************************************************************************************
*
*	Function:	DDraw_GetAddress()
*
*	Desc:		Gets the address of our Virtual page
*
*	Notes:		
*
***************************************************************************************************/
Color32_t *DDraw_GetAddress( void )
{
	return m_VPage;
}

/*************************************************************************************************
*
*	Function:	DDraw_DisplayCallback()
*
*	Desc:		Called for each Displaymode enumerated in DDraw_Init()
*
*	Notes:		
*
***************************************************************************************************/
static HRESULT WINAPI DDraw_DisplayCallback( LPDDSURFACEDESC lpDDSurfaceDesc, LPVOID lpContext )
{
	char buf[256];

	// copy mode info into our structure
	m_ModeList[ m_NumberModes ].Width = lpDDSurfaceDesc->dwWidth;
	m_ModeList[ m_NumberModes ].Height = lpDDSurfaceDesc->dwHeight;
	m_ModeList[ m_NumberModes ].Bpp = lpDDSurfaceDesc->ddpfPixelFormat.dwRGBBitCount;
	m_NumberModes++;

	// show the modes
	sprintf( buf, "(%02i) - %03i x %03i x %i\n", m_NumberModes-1, m_ModeList[ m_NumberModes-1 ].Width, m_ModeList[ m_NumberModes-1 ].Height, m_ModeList[ m_NumberModes-1 ].Bpp );
	OutputDebugString( buf );

	// next mode
	return DDENUMRET_OK;
}
 
/*************************************************************************************************
*
*	Function:	GetLowestBit()
*
*	Desc:		Gets the lowest significant bit position
*
*	Notes:		
*
***************************************************************************************************/
static int GetLowestBit( unsigned int Value )
{
	int Position;

	// sanity check
	if (Value == 0) return 0;

	Position = 0;
	while ( (Value&1) == 0)
	{
		Position++;
		Value = Value>> 1;
	}

	return Position;
}

/*************************************************************************************************
*
*	Function:	GetHighestBit()
*
*	Desc:		Gets the highest significant bit position
*
*	Notes:		
*
***************************************************************************************************/
static int GetHighestBit( unsigned int Value )
{
	int Position;

	// sanity check
	if (Value == 0) return 0;

	Position = 0;
	while ( (Value&0x80000000) == 0)
	{
		Position++;
		Value = Value<< 1;
	}

	return 32 - Position;
}
