#ifndef _VL_QUAT_H_
#define _VL_QUAT_H_
// -----------------------------------------------------------------------------
//
//      Mimic Engine
//      Copyright (C) 1997-1999 by Maciej Sinilo
//
//      MODULE  : VL_QUAT.H - Quaternions
//      CREATED : 24-03-99
//
// -----------------------------------------------------------------------------


#include "vl_main.h"
#include "vl_vec3.h"


// --- vlQuaternion DEFINITION -------------------------------------------------
//
// Quaternion class, represented as 3D vector and additional scalar
//
class vlQuaternion
{
public:

        vlQuaternion();
        vlQuaternion(const vlVec3& inV, const vl_real inW);
        vlQuaternion(const vlVec4& inV4);
        vlQuaternion(const vl_real x, const vl_real y, const vl_real z,
                     const vl_real w);
        vlQuaternion(const vlQuaternion& other);
        vlQuaternion(const vl_real* pArray);

        // Get/set element (coordinate)
        vl_real& operator [] (const int index);
        const vl_real& operator [] (const int index) const;

        // Conversion
        // !WARNING! It assumes v always lies before w [in memory]!
        operator vl_real* ()             { return (vl_real *)v; }
        operator const vl_real*() const  { return (const vl_real *)v; }

        // Get/set element (coordinate)
        // Other fashion, more convenient when using pointers
        vl_real&            rX()             { return v[0]; }
        const vl_real&      rX() const       { return v[0]; }
        vl_real&            rY()             { return v[1]; }
        const vl_real&      rY() const       { return v[1]; }
        vl_real&            rZ()             { return v[2]; }
        const vl_real&      rZ() const       { return v[2]; }
        vl_real&            rW()             { return w; }
        const vl_real&      rW() const       { return w; }


        vlQuaternion& operator = (const vlQuaternion& other);
        vlQuaternion& operator *= (const vlQuaternion& q);
        // Conjugate (rotate in opposite direction)
        vlQuaternion& operator ~ ();

        vlMatrix4 ToMatrix() const;
        operator vlMatrix4 () const { return ToMatrix(); }

        // !WARNING! It assumes v always lies before w [in memory]!
        vl_real* GetPtr()                 { return (vl_real*)v;       }
        const vl_real* GetReadPtr() const { return (const vl_real*)v; }

        void FromArray(const vl_real* pArray);

// FRIENDS
        friend vlQuaternion operator * (const vlQuaternion&, const vlQuaternion&);
        friend vlQuaternion Slerp(const vlQuaternion& a, const vlQuaternion& b,
                const vl_real t);

// FRIENDS
        friend ostream& operator << (ostream& s, const vlQuaternion& q);

protected:

        vlVec3    v;
        vl_real   w;
};



// --- vlQuaternion IMPLEMENTATION ---------------------------------------------

VL_INL vlQuaternion::vlQuaternion() : v() { }
VL_INL vlQuaternion::vlQuaternion(const vlVec3& inV, const vl_real inW) : v(inV), w(inW) { }
VL_INL vlQuaternion::vlQuaternion(const vlVec4& inV4) : v(inV4[0], inV4[1], inV4[2]), w(inV4[3]) { }
VL_INL vlQuaternion::vlQuaternion(const vl_real x, const vl_real y, const vl_real z,
        const vl_real ww) : v(x, y, z), w(ww) { }
VL_INL vlQuaternion::vlQuaternion(const vlQuaternion& other) : v(other.v), w(other.w) { }
VL_INL vlQuaternion::vlQuaternion(const vl_real* pArray) { FromArray(pArray); }
VL_INL vl_real& vlQuaternion::operator [] (const int i) { return (v.GetPtr())[i]; }
VL_INL const vl_real& vlQuaternion::operator [] (const int i) const { return ((const vl_real*)&v[0])[i]; }
VL_INL vlQuaternion& vlQuaternion::operator = (const vlQuaternion& other) { v = other.v; w = other.w; return SELF; }
VL_INL vlQuaternion& vlQuaternion::operator *= (const vlQuaternion& q) { SELF = SELF * q; return SELF; }
VL_INL vlQuaternion& vlQuaternion::operator ~ () { v = -v; return SELF; }

VL_INL vlQuaternion operator * (const vlQuaternion& a, const vlQuaternion& b)
{
        return vlQuaternion(
                a.w*b.v[0] + b.w*a.v[0] + a.v[1]*b.v[2] - a.v[2]*b.v[1],
                a.w*b.v[1] + b.w*a.v[1] + a.v[2]*b.v[0] - a.v[0]*b.v[2],
                a.w*b.v[2] + b.w*a.v[2] + a.v[0]*b.v[1] - a.v[1]*b.v[0],
                a.w*b.w - (a.v * b.v));
}

VL_INL void vlQuaternion::FromArray(const vl_real* pArray)
{
        v.FromArray(pArray);
        w = pArray[4];
}

// -----------------------------------------------------------------------------
//      VL_QUAT.H - HEADER END
// -----------------------------------------------------------------------------
#endif  // !_VL_QUAT_H_

