#include <stdio.h>
#include <graphs.h>
#include <types.h>
#include <math.h>
#include <malloc.h>
#include <icp.h>
#include <string.h>

BYTE read_tga_map(BYTE *,BYTE *,BYTE *);
BYTE read_tga_picture(BYTE *, BYTE *, DWORD, DWORD);
void convert_pal(BYTE *pal);
void convert_pal_8(BYTE *palin, BYTE *palout);
BYTE precalc_gouraud(BYTE *, BYTE *, BYTE *, float, float, float, float);
void take_photo(BYTE *, BYTE *);

#pragma pack(1)
typedef struct {
    BYTE no_characters;         // Length of Image Identification Field
    BYTE map_type;              //  0 = no color map     1 = color map
    BYTE type;                  //  1 = Color-mapped image Uncommpressed
                                //  2 = RGB image Uncompressed
                                //  9 = Color-mapped image RLE Compressed
                                // 10 = RGB image Compressed

    WORD map_origin;            // Color Map Origin
    WORD map_length;            // Color Map Length
    BYTE map_size;              // Number of bits in each color map entry

    WORD x_origin;
    WORD y_origin;
    WORD width;
    WORD height;
    BYTE bits;
    BYTE descriptor;
}HEAD_TGA;
#pragma pack()

#define UNCOMP_8BITS   1
#define COMP_8BITS     9
#define UNCOMP_RGB     2
#define COMP_RGB      10
#define TEXT_SIZE    256

// --------------------------------------------------------------------------

// Lee el TGA que se pasa en name (debe incluir extensin). El TGA debe ser
//  del tipo 8 bits Comprimido o sin comprimir, y de tamao 256*256
// Devuelve en mem la textura en modo raw 256*256, y en pal la paleta en
//  formato 32 bits ORGB
// Si hubo algn error devuelve un 1

BYTE read_tga_map(BYTE *name,BYTE *mem,BYTE *pal) {

    HEAD_TGA head;
    SDWORD i,j,k;
    BYTE color,pack;
    BYTE *buffer;
    DWORD index;
    DWORD *dpal=(DWORD *)pal;

    buffer=lib_fopen(name);
    if(buffer==NULL) return 1;

    memcpy(&head,buffer,sizeof(HEAD_TGA));

    if(head.type!=COMP_8BITS && head.type!=UNCOMP_8BITS) {
        free(buffer);
        return 1;
    }
    if(head.width!=TEXT_SIZE || head.height!=TEXT_SIZE) {
        free(buffer);
        return 1;
    }
    if(head.map_size!=24 && head.map_size!=32) {
        free(buffer);
        return 1;
    }
    if(head.map_length!=256) {
        free(buffer);
        return 1;
    }

    index=sizeof(HEAD_TGA)+head.no_characters;

    // Lectura de la paleta
    for(i=0;i<head.map_length;i++) {
        for(k=0;k<head.map_size/8;k++)
            *(pal+i*4+k)=*(buffer+index++);
        *(dpal+i)<<=32-head.map_size;
    }


    if(head.type==UNCOMP_8BITS) {
        for(i=TEXT_SIZE-1;i>=0;i--) for(j=0;j<TEXT_SIZE;j++) {
            color=*(buffer+index++);
            *(mem+i*TEXT_SIZE+j)=color;
        }
    }
    else if(head.type==COMP_8BITS) {

        i=TEXT_SIZE-1;  j=0;
        while(i>=0) {
            pack=*(buffer+index++);
            if(pack&128) {
                pack=pack&127;
                color=*(buffer+index++);
                for(k=0;k<=pack;k++) {
                    *(mem+i*TEXT_SIZE+j)=color;
                    j++;
                    if(j==TEXT_SIZE) {j=0; i--;}
                }
            }
            else {
                pack=pack&127;
                for(k=0;k<=pack;k++) {
                    color=*(buffer+index++);
                    *(mem+i*TEXT_SIZE+j)=color;
                    j++;
                    if(j==TEXT_SIZE) {j=0; i--;}
                }
            }
        }
    }

    free(buffer);

    return 0;
}

static DWORD get_pixel(BYTE *mem,DWORD *index,HEAD_TGA *head) {

    WORD wcolor;
    DWORD red,green,blue;
    BYTE bred,bgreen,bblue;

    switch(head->bits) {
        case 16:    wcolor=*((WORD *)(mem+*index));
                    *index=*index+2;
                    blue=(wcolor&31)<<3;
                    green=((wcolor>>5)&31)<<3;
                    red=((wcolor>>10)&31)<<3;
                    break;
        case 24:    bblue=*(mem+*index);
                    bgreen=*(mem+*index+1);
                    bred=*(mem+*index+2);
                    *index=*index+3;
                    red=bred; green=bgreen; blue=bblue;
                    break;
        case 32:    bblue=*(mem+*index+1);
                    bgreen=*(mem+*index+2);
                    bred=*(mem+*index+3);
                    *index=*index+4;
                    red=bred; green=bgreen; blue=bblue;
    }

    red>>=8-graphics_system.redsize;
    green>>=8-graphics_system.greensize;
    blue>>=8-graphics_system.bluesize;
    red=red<<graphics_system.redpos;
    green=green<<graphics_system.greenpos;
    blue=blue<<graphics_system.bluepos;

    return red+green+blue;

}

static void put_pixel(BYTE *mem, DWORD color) {

    WORD *wmem=(WORD *)mem;
    DWORD *dmem=(DWORD *)mem;

    switch(graphics_system.bbp) {
        case 15:
        case 16:    *wmem=color;
                    break;
        case 24:    *mem=color&255;
                    *(mem+1)=(color>>8)&255;
                    *(mem+2)=(color>>16)&255;
                    break;
        case 32:    *dmem=color;
    }

}

// Lee el TGA que se pasa en name (debe incluir extensin). El TGA debe ser
//  del tipo 16/24/32 bits de color y del tamao (y,x).
// Devuelve en mem el dibujo en modo raw.
// Si hubo algn error devuelve un 1.

BYTE read_tga_picture(BYTE *name, BYTE *mem, DWORD x, DWORD y) {

    HEAD_TGA head;
    WORD bpp;
    SDWORD i,j,k,color;
    DWORD index;
    BYTE pack;
    BYTE *buffer;

    switch (graphics_system.bbp) {
        case 15:
        case 16:   bpp=2;
                   break;
        case 24:   bpp=3;
                   break;
        case 32:   bpp=4;
    }

    buffer=lib_fopen(name);
    if(buffer==NULL) return 1;

    memcpy(&head,buffer,sizeof(HEAD_TGA));


    if(head.type!=COMP_RGB && head.type!=UNCOMP_RGB) {
        free(buffer);
        return 1;
    }
    if(head.width!=x || head.height!=y) {
        free(buffer);
        return 1;
    }

    index=sizeof(HEAD_TGA)+head.no_characters+head.map_length;

    if(head.type==UNCOMP_RGB) {
        for(i=y-1;i>=0;i--) for(j=0;j<x;j++) {
            put_pixel(mem+(i*x+j)*bpp,get_pixel(buffer,&index,&head));
        }
    }
    else if(head.type==COMP_RGB) {

        i=y-1;  j=0;
        while(i>=0) {
            pack=*(buffer+index++);
            if(pack&128) {
                pack=pack&127;
                color=get_pixel(buffer,&index,&head);
                for(k=0;k<=pack;k++) {
                    put_pixel(mem+(i*x+j)*bpp,color);
                    if(++j==x) {j=0; i--;}
                }
            }
            else {
                pack=pack&127;
                for(k=0;k<=pack;k++) {
                    put_pixel(mem+(i*x+j)*bpp,get_pixel(buffer,&index,&head));
                    if(++j==x) {j=0; i--;}
                }
            }
        }
    }

    free(buffer);

    return 0;

}


// Convierte una paleta de 32 bits (ORGB, por ejemplo la devuelta por la
//  funcin read_tga() al formato VGA RGB con 6 bits por componente.

void convert_pal_8(BYTE *palin, BYTE *palout) {

    DWORD i;

    for(i=0;i<256;i++) {
        *(palout+i*3)=(*(palin+i*4+3))>>2;
        *(palout+i*3+1)=(*(palin+i*4+2))>>2;
        *(palout+i*3+2)=(*(palin+i*4+1))>>2;
    }
}

// Convierte una paleta de 32 bits (ORGB, por ejemplo la devuelta por la
//  funcin read_tga() a la correspondiente al modo activo.

void convert_pal(BYTE *pal) {

    DWORD i;
    DWORD red,green,blue;
    SBYTE redpos=graphics_system.redpos;
    BYTE redmask=((2<<graphics_system.redsize)-1)<<(8-graphics_system.redsize);
    SBYTE greenpos=graphics_system.greenpos;
    BYTE greenmask=((2<<graphics_system.greensize)-1)
                 <<(8-graphics_system.greensize);
    SBYTE bluepos=graphics_system.bluepos;
    BYTE bluemask=((2<<graphics_system.bluesize)-1)
                 <<(8-graphics_system.bluesize);
    DWORD *dpal=(DWORD *)pal;

    redpos-=8-graphics_system.redsize;
    greenpos-=8-graphics_system.greensize;
    bluepos-=8-graphics_system.bluesize;

    for(i=0;i<256;i++) {
        blue=(*(pal+i*4+1))&bluemask;
        green=(*(pal+i*4+2))&greenmask;
        red=(*(pal+i*4+3))&redmask;
        red=(redpos>=0)?red<<redpos:red>>-redpos;
        green=(greenpos>=0)?green<<greenpos:green>>-greenpos;
        blue=(bluepos>=0)?blue<<bluepos:blue>>-bluepos;
        *(dpal+i)=red+green+blue;
    }

}


//  Generacin de tablas precalculadas para Gouraud en True/Hi Color

// Entrada: Paleta 32 bits  ORGB
// Salida:  Paleta 32 bits  ORGB
//          Look Up Table 256*256 -> [intensidad][color]
// Si hubo algn error devuelve un 1

typedef struct {
    BYTE alpha,red,green,blue;
}ORGB;

#define NSHADES 256
#define PI 3.141592653589793238462643383279502884197169399375105820975

BYTE precalc_gouraud(BYTE *pal_in,BYTE *pal_out,BYTE *table,float ambient,
                     float diffuse, float specular, float coefficient) {

    SDWORD i,j,k,color;
    float angulo;
    BYTE *pal;
    ORGB *true_pal_in=(ORGB *)pal_in;
    ORGB *true_pal_out=(ORGB *)pal_out;
    reduce_RGB *shades=(reduce_RGB *)malloc(256*256*sizeof(reduce_RGB));
    if(shades==NULL) return 1;
    pal=malloc(768*sizeof(BYTE));
    if(pal==NULL) { free(shades); return 1; }

    for(j=0,k=0;j<NSHADES;j++) {
        angulo=PI/2-(PI*j)/(2*(NSHADES-1));
        for(i=0;i<256;i++,k++) {
            color=(ambient+diffuse*cos(angulo))*(true_pal_in+i)->red+
                  (specular*pow(cos(angulo),coefficient))*255;
            if(color>255) color=255;
            (shades+k)->color.r=color;

            color=(ambient+diffuse*cos(angulo))*(true_pal_in+i)->green+
                  (specular*pow(cos(angulo),coefficient))*255;
            if(color>255) color=255;
            (shades+k)->color.g=color;

            color=(ambient+diffuse*cos(angulo))*(true_pal_in+i)->blue+
                  (specular*pow(cos(angulo),coefficient))*255;
            if(color>255) color=255;
            (shades+k)->color.b=color;
            }
        }

    if(quantize(256*NSHADES,pal,shades,table,256)) return 1;
    for(i=0;i<256;i++) {
        (true_pal_out+i)->red=*(pal+i*3);
        (true_pal_out+i)->green=*(pal+i*3+1);
        (true_pal_out+i)->blue=*(pal+i*3+2);
    }

    free(pal);
    free(shades);
    return 0;
}

// ----------------------Capturador de pantallas-----------------------------
//  Solo soporta salida a 24 bits RGB sin comprimir

static void decode(BYTE *virtual, DWORD x, DWORD y, BYTE *r, BYTE *g, BYTE *b) {

    BYTE *pos=virtual+y*graphics_system.bytes_scanline;
    WORD *wpos=(WORD *)pos;
    DWORD *dpos=(DWORD *)pos;
    DWORD color;

    switch(graphics_system.bbp) {
        case 15:
        case 16:
                    *r=(*(wpos+x))>>graphics_system.redpos;
                    *r=*r&((1<<graphics_system.redsize)-1);
                    *r=*r<<(8-graphics_system.redsize);

                    *g=(*(wpos+x))>>graphics_system.greenpos;
                    *g=*g&((1<<graphics_system.greensize)-1);
                    *g=*g<<(8-graphics_system.greensize);

                    *b=(*(wpos+x))>>graphics_system.bluepos;
                    *b=*b&((1<<graphics_system.bluesize)-1);
                    *b=*b<<(8-graphics_system.bluesize);

                    break;
        case 24:
                    color=*(pos+3*x) + (*(pos+3*x+1)<<8) + (*(pos+3*x+2)<<16);

                    *r=color>>graphics_system.redpos;
                    *r=*r&((1<<graphics_system.redsize)-1);
                    *r=*r<<(8-graphics_system.redsize);

                    *g=color>>graphics_system.greenpos;
                    *g=*g&((1<<graphics_system.greensize)-1);
                    *g=*g<<(8-graphics_system.bluesize);

                    *b=color>>graphics_system.bluepos;
                    *b=*b&((1<<graphics_system.bluesize)-1);
                    *b=*b<<(8-graphics_system.bluesize);

                    break;

        case 32:

                    *r=(*(dpos+x))>>graphics_system.redpos;
                    *r=*r&((1<<graphics_system.redsize)-1);
                    *r=*r<<(8-graphics_system.redsize);

                    *g=(*(dpos+x))>>graphics_system.greenpos;
                    *g=*g&((1<<graphics_system.greensize)-1);
                    *g=*g<<(8-graphics_system.bluesize);

                    *b=(*(dpos+x))>>graphics_system.bluepos;
                    *b=*b&((1<<graphics_system.bluesize)-1);
                    *b=*b<<(8-graphics_system.bluesize);

                    break;
    }

}

// Crea un fichero TGA con el contenido de virtual

void take_photo(BYTE *virtual, BYTE *name) {

    HEAD_TGA header;
    FILE *fp;
    DWORD i,j;
    BYTE r,g,b;
    BYTE *line;

    if(graphics_system.bbp!=15 && graphics_system.bbp!=16 &&
       graphics_system.bbp!=24 && graphics_system.bbp!=32) return;

    line=malloc(graphics_system.xresolution*3);
    if(line==NULL) return;

    fp=fopen(name,"wb");
    if(fp==NULL) return;


    header.no_characters=0;
    header.map_type=0;
    header.type=UNCOMP_RGB;

    header.map_origin=0;
    header.map_length=0;
    header.map_size=0;

    header.x_origin=0;
    header.y_origin=0;
    header.width=graphics_system.xresolution;
    header.height=graphics_system.yresolution;
    header.bits=24;
    header.descriptor=0;

    fwrite(&header,1,sizeof(HEAD_TGA),fp);

    for(i=0;i<header.height;i++) {

        for(j=0;j<header.width;j++) {

            decode(virtual,j,header.height-1-i,&r,&g,&b);
            *(line+3*j)=b;
            *(line+3*j+1)=g;
            *(line+3*j+2)=r;
        }

        fwrite(line,header.width*3,sizeof(BYTE),fp);

    }

    free(line);
    fclose(fp);


}
