#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/mman.h>
#include "memory.h"
#include "derrno.h"

struct mcb_struct {
  int used;
  struct mcb_struct *next;
  int size;
  unsigned int magic;
};

#define MCB_MAGIC 0x19761999
#define MCB_SIZE 16

static struct mcb_struct *mcb_first;


int dosmem_largest ( void )
{
  struct mcb_struct *p=mcb_first;
  int largest=0;
  while (p) {
    if (!p->used&&p->size>largest) largest=p->size;
    p=p->next;
  }
  return largest>>4;
}


void dosmem_init ( int base , int ramsize )
{
  if (mmap(NULL,0x110000,PROT_READ|PROT_WRITE|PROT_EXEC,MAP_FIXED|MAP_PRIVATE|MAP_ANON,-1,0)) {
    perror("anonymous mmap()-ing 1M+64K failed");
    exit(1);
  }
  bzero(NULL,0x110000);
  mcb_first=(struct mcb_struct*)base;
  mcb_first->size=ramsize-base-MCB_SIZE;
  mcb_first->used=0;
  mcb_first->next=NULL;
  mcb_first->magic=MCB_MAGIC;
}


static void optimize_blocks ( void )
{
  struct mcb_struct *p=mcb_first,*q;
  while (p) {
    if (!p->used) {
      q=p;
      while (q->next&&!q->next->used) q=q->next;
      if (p!=q) {
        p->next=q->next;
	p->size=(int)q-(int)p-MCB_SIZE+q->size;
      }  /* if */
    }  /* if */
    p=p->next;
  } /* while */
}


int dosmem_alloc ( unsigned int size , int user )
{
  struct mcb_struct *p=mcb_first,*best=NULL;
  int remain,bestremain=1000000;
  size<<=4;  /* allocate paragraphs ! */
  while (p) {
    remain=p->size-size;
    if (remain<bestremain&&remain>=0&&!p->used) {
      best=p;
      bestremain=remain;
    }
    p=p->next;
  }
  if (!best) return -DNOMEM;  /* not enough memory */
  if (bestremain>=MCB_SIZE*4) {
    struct mcb_struct *new=(struct mcb_struct*)((int)best+size+MCB_SIZE);
    new->next=best->next;
    best->next=new;
    new->used=0;
    best->size=size;
    new->size=bestremain-size-MCB_SIZE;
    new->magic=MCB_MAGIC;
  }
  best->used=user;
  return (int)best+MCB_SIZE;
}


int dosmem_free ( int addr , int user )
{
  struct mcb_struct *p=(struct mcb_struct*)(addr-MCB_SIZE);
  if (p->used!=user) return 1;
  if (p->magic!=MCB_MAGIC) return 1;
  p->used=0;
  optimize_blocks();
  return 0;
}


int dosmem_realloc ( int addr , int newsize , int user )
{
  struct mcb_struct *p=(struct mcb_struct*)(addr-MCB_SIZE);
  int remain;
  return 0; /* get LOST ! */
  if (p->used!=user) return 1;
  if (p->magic!=MCB_MAGIC) return 1;
  newsize<<=4;
  if (newsize==p->size) return 0; /* actual size, no need to work */
  if (newsize>p->size) {  /* GROW */
    if (!p->next||p->next->used) return 1; /* no free block after this one */
    remain=p->next->size+p->size+MCB_SIZE-newsize;
    if (remain<0) return 0; /* not enough space in the two free blocks */
    if (remain>=MCB_SIZE*4) {
      struct mcb_struct *new=(struct mcb_struct *)((int)p+newsize+MCB_SIZE);
      new->magic=MCB_MAGIC;
      new->next=p->next;
      p->next=new;
      new->used=0;
      new->size=remain-MCB_SIZE;
      p->size=newsize;
    } else {
      p->size+=p->next->size+MCB_SIZE;
      p->next=p->next->next;
    }
    return 0;
  } else {             /* SHRINK */
    return 0;
  }
}


void dosmem_dump ( void )
{
  struct mcb_struct *p=mcb_first;
  fprintf(stderr,"base     size     next     used     magic\n");
  while (p) {
    fprintf(stderr,"%08X %08X %08X %08X %08X ",(int)p,p->size,(int)p->next,p->used,p->magic);
    if (p->magic!=MCB_MAGIC) fprintf(stderr,"??MAGIC ");
    if (p->next&&(int)p->next-(int)p-MCB_SIZE!=p->size) fprintf(stderr,"??SIZE");
    fprintf(stderr,"\n");
    p=p->next;
  }
}
