/*
 *
 *   HElliZER: the first portable demo in the world
 *
 *   Copyright (C) 1996  Queue Members Group Art Division
 *   Coded by Mad Max / Queue Members Group (Mike Shirobokov)
 *   <mad_max@dixon.volgacom.samara.su>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 */
#include "3d.h"
#include "misc.h"
#include "video.h"
#include "resource.h"
#include "parts.h"

int x1,x2,ls,zs,txs,tys;

PAGE shadow;
uchar** yOffset;
void (*DrawFace)();

inline short hshort( int x ) { return short(x>>16); }
inline ushort hushort( int x ) { return ushort(x>>16); }
inline uchar huchar( int x ) { return uchar(x>>24); }

void DrawLineShadow( int x1, int z1, int l1, int tx1, int ty1 )
{
  int shadow_offset=shadow-bw;
  for(;;) {
    if( zbuffer[x1] < hshort(z1) ) {
      zbuffer[x1] = hshort(z1);
      color[x1] = (yOffset[hushort(ty1)])[hushort(tx1)];
      bw[x1] = (uchar)hushort(l1);
      if( zbuffer[x1+shadow_offset]==MIN_Z ) shadow[x1]=30;
    }
    if( ++x1 >= x2 ) break;
    l1+=ls; z1+=zs; tx1+=txs; ty1+=tys;
  }
  if( zbuffer[x1] < hshort(z1) ) {
    zbuffer[x1] = hshort(z1);
    color[x1] = (yOffset[hushort(ty1)])[hushort(tx1)];
    bw[x1] = (uchar)hushort(l1);
    if( zbuffer[x1+shadow_offset]==MIN_Z ) shadow[x1]=30;
  }
}

void DrawLineNoShadow( int x1, int z1, int l1, int tx1, int ty1 )
{
  for(;;) {
    if( zbuffer[x1] < hshort(z1) ) {
      zbuffer[x1] = hshort(z1);
      color[x1] = (yOffset[hushort(ty1)])[hushort(tx1)];
      bw[x1] = (uchar)hushort(l1);
    }
    if( ++x1 >= x2 ) break;
    l1+=ls; z1+=zs; tx1+=txs; ty1+=tys;
  }
  if( zbuffer[x1] < hshort(z1) ) {
    zbuffer[x1] = hshort(z1);
    color[x1] = (yOffset[hushort(ty1)])[hushort(tx1)];
    bw[x1] = (uchar)hushort(l1);
  }
}

void DrawLineNoTexture( int x1, int z1, int l1, int tx1, int ty1 )
{
  for(;;) {
    if( zbuffer[x1] < hshort(z1) ) {
      zbuffer[x1] = hshort(z1);
      bw[x1] = (uchar)hushort(l1);
    }
    if( ++x1 >= x2 ) break;
    l1+=ls; z1+=zs; tx1+=txs; ty1+=tys;
  }
  if( zbuffer[x1] < hshort(z1) ) {
    zbuffer[x1] = hshort(z1);
    bw[x1] = (uchar)hushort(l1);
  }
}

int leftx, leftz, rightx, rightz;
int lefttx, leftty, leftl, righttx, rightty, rightl;
int rightxs, rightzs, righttxs, righttys, rightls,
    leftxs, leftzs, lefttxs, lefttys, leftls;
int tmpl, tmpr, offset, ys;

void DrawFaceShadow()
{
  for( int y=0; y<tmpr; y++ ) {
    x1 = offset+(leftx>>16);
    x2 = offset+(rightx>>16);
    if(x2 > offset && x1<offset+vidSizeX) {
      if(x2 > offset+vidSizeX-2) x2 = offset+vidSizeX-2;
      if(x1<offset) {
        register tmp = offset-x1;
        DrawLineShadow( offset, leftz+zs*tmp, leftl+ls*tmp,
                  lefttx+txs*tmp, leftty+tys*tmp );
      }
      else {
        DrawLineShadow( x1, leftz, leftl, lefttx, leftty );
      }
    }
    leftx += leftxs; leftz += leftzs;
    lefttx += lefttxs; leftty += lefttys; leftl += leftls;
    rightx += rightxs; rightz += rightzs;
    righttx += righttxs; rightty += righttys; rightl += rightls;
    offset += vidBytesPerLine;
  }
}

void DrawFaceNoShadow()
{
  for( int y=0; y<tmpr; y++ ) {
    x1 = offset+(leftx>>16);
    x2 = offset+(rightx>>16);
    if(x2 > offset && x1<offset+vidSizeX-1) {
      if(x2 > offset+vidSizeX-2) x2 = offset+vidSizeX-2;
      if(x1<offset) {
        register tmp = offset-x1;
        DrawLineNoShadow( offset, leftz+zs*tmp, leftl+ls*tmp,
                  lefttx+txs*tmp, leftty+tys*tmp );
      }
      else {
        DrawLineNoShadow( x1, leftz, leftl, lefttx, leftty );
      }
    }
    leftx += leftxs; leftz += leftzs;
    lefttx += lefttxs; leftty += lefttys; leftl += leftls;
    rightx += rightxs; rightz += rightzs;
    righttx += righttxs; rightty += righttys; rightl += rightls;
    offset += vidBytesPerLine;
  }
}

void DrawFaceNoTexture()
{
  for( int y=0; y<tmpr; y++ ) {
    x1 = offset+(leftx>>16);
    x2 = offset+(rightx>>16);
    if(x2 > offset && x1<offset+vidSizeX) {
      if(x2 > offset+vidSizeX-2) x2 = offset+vidSizeX-2;
      if(x1<offset) {
        register tmp = offset-x1;
        DrawLineNoTexture( offset, leftz+zs*tmp,
                           leftl+ls*tmp, 0, 0 );
      }
      else {
        DrawLineNoTexture( x1, leftz, leftl, 0, 0);
      }
    }
    leftx += leftxs; leftz += leftzs; leftl += leftls;
    rightx += rightxs; rightz += rightzs; rightl += rightls;
    offset += vidBytesPerLine;
  }
}

inline FacedObject::setDrawLineShadow( PAGE color, PAGE bw, int shadow_offset )
{
  DrawFace = &DrawFaceShadow;
  shadow = bw - shadow_offset;
}

inline FacedObject::setDrawLineNoShadow( PAGE color, PAGE bw )
{
  DrawFace = &DrawFaceNoShadow;
}

inline FacedObject::setDrawLineNoTexture( PAGE bw )
{
  DrawFace = &DrawFaceNoTexture;
}

Point _vert[3];

void Face::Draw()
{
  bool is_right;
  ::yOffset = yOffset;

  if( vert[0]->y < vert[1]->y ) {
    if( vert[1]->y < vert[2]->y ) {
      _vert[0] = *vert[0]; _vert[1] = *vert[1]; _vert[2] = *vert[2];
    }
    else {
      _vert[2] = *vert[1];
      if( vert[0]->y < vert[2]->y ) {
        _vert[0] = *vert[0]; _vert[1] = *vert[2];
      }
      else {
        _vert[0] = *vert[2]; _vert[1] = *vert[0];
      }
    }
  }
  else {
    if( vert[0]->y < vert[2]->y ) {
      _vert[0] = *vert[1]; _vert[1] = *vert[0]; _vert[2] = *vert[2];
    }
    else {
      _vert[2] = *vert[0];
      if( vert[1]->y < vert[2]->y ) {
        _vert[0] = *vert[1]; _vert[1] = *vert[2];
      }
      else {
        _vert[0] = *vert[2]; _vert[1] = *vert[1];
      }
    }
  }
//  _vert[0].l=0; _vert[1].l=0;
  int y0=_vert[0].y>>16, y1=_vert[1].y>>16, y2=_vert[2].y>>16;
  if( y0>=vidSizeY || y2<0 ) return;
  rightx = leftx = _vert[0].x;
  rightz = leftz = _vert[0].z;
  righttx = lefttx = _vert[0].tx;
  rightty = leftty = _vert[0].ty;
  rightl = leftl = _vert[0].l;
  tmpl = y2-y0;
  if( !tmpl ) return;
  leftxs = (_vert[2].x-leftx)/tmpl;
  tmpr = y1-y0;
  if( tmpr ) {
    rightxs = (_vert[1].x-rightx)/tmpr;
    if(leftxs == rightxs) return;
    if(leftxs < rightxs) {
      rightzs = (_vert[1].z-rightz)/tmpr;
      righttxs = (_vert[1].tx-righttx)/tmpr;
      righttys = (_vert[1].ty-rightty)/tmpr;
      rightls = (_vert[1].l-rightl)/tmpr;
      leftzs = (_vert[2].z-leftz)/tmpl;
      lefttxs = (_vert[2].tx-lefttx)/tmpl;
      lefttys = (_vert[2].ty-leftty)/tmpl;
      leftls = (_vert[2].l-leftl)/tmpl;
      is_right = true;
    }
    else {
      { int tmp=leftxs; leftxs=rightxs; rightxs=tmp; }
      rightzs = (_vert[2].z-rightz)/tmpl;
      righttxs = (_vert[2].tx-righttx)/tmpl;
      righttys = (_vert[2].ty-rightty)/tmpl;
      rightls = (_vert[2].l-rightl)/tmpl;
      leftzs = (_vert[1].z-leftz)/tmpr;
      lefttxs = (_vert[1].tx-lefttx)/tmpr;
      lefttys = (_vert[1].ty-leftty)/tmpr;
      leftls = (_vert[1].l-leftl)/tmpr;
      is_right = false;
    }
    offset = y0*vidBytesPerLine;
    if( y0<0 ) {
      if(y1<0) y0-=y1;
      rightx -= rightxs*y0; rightz -= rightzs*y0; rightl -= rightls*y0;
      righttx -= righttxs*y0; rightty -= righttys*y0;
      leftx -= leftxs*y0; leftz -= leftzs*y0; leftl -= leftls*y0;
      lefttx -= lefttxs*y0; leftty -= lefttys*y0;
      offset = 0; tmpr += y0;
    }
    {
      register tmp = (rightxs-leftxs)>>8;
      if( tmp ) {
        ls = (rightls-leftls)/tmp         << 8;
        zs = (rightzs-leftzs)/tmp         << 8;
        txs = (righttxs-lefttxs)/tmp      << 8;
        tys = (righttys-lefttys)/tmp      << 8;
      }
      else {
        zs = ls = txs = tys = 0;
      }
    }
    if( y1 >= 0 ) {
      if(y1>vidSizeY) {
        tmpr -= y1-vidSizeY;
        DrawFace();
        return;
      }
      else {
        DrawFace();
      }
    }
  }
  else {
    if(_vert[1].x<leftx) {
      leftx = _vert[1].x;
      leftz = _vert[1].z;
      lefttx = _vert[1].tx;
      leftty = _vert[1].ty;
      leftl = _vert[1].l;
      leftxs = (_vert[2].x-leftx)/tmpl;
      leftzs = (_vert[2].z-leftz)/tmpl;
      lefttxs = (_vert[2].tx-lefttx)/tmpl;
      lefttys = (_vert[2].ty-leftty)/tmpl;
      leftls = (_vert[2].l-leftl)/tmpl;
      is_right = true;
    }
    else {
      rightx = _vert[1].x;
      rightz = _vert[1].z;
      righttx = _vert[1].tx;
      rightty = _vert[1].ty;
      rightl = _vert[1].l;
      rightxs = (_vert[2].x-rightx)/tmpl;
      rightzs = (_vert[2].z-rightz)/tmpl;
      righttxs = (_vert[2].tx-righttx)/tmpl;
      righttys = (_vert[2].ty-rightty)/tmpl;
      rightls = (_vert[2].l-rightl)/tmpl;
      is_right = false;
    }
    offset = y1*vidBytesPerLine;
  }
  tmpr = y2-y1;
  if( !tmpr ) return;
  if( is_right ) {
    rightxs = (_vert[2].x-rightx)/tmpr,
    rightzs = (_vert[2].z-rightz)/tmpr,
    rightls = (_vert[2].l-rightl)/tmpr,
    righttxs = (_vert[2].tx-righttx)/tmpr,
    righttys = (_vert[2].ty-rightty)/tmpr;
  }
  else {
    leftxs = (_vert[2].x-leftx)/tmpr,
    leftzs = (_vert[2].z-leftz)/tmpr,
    lefttxs = (_vert[2].tx-lefttx)/tmpr,
    lefttys = (_vert[2].ty-leftty)/tmpr,
    leftls = (_vert[2].l-leftl)/tmpr;
  }
  if( y1<0 ) {
    rightx -= rightxs*y1; rightz -= rightzs*y1; rightl -= rightls*y1;
    righttx -= righttxs*y1; rightty -= righttys*y1;
    leftx -= leftxs*y1; leftz -= leftzs*y1; leftl -= leftls*y1;
    lefttx -= lefttxs*y1; leftty -= lefttys*y1;
    offset = 0; tmpr += y1;
  }
  if(y2>vidSizeY) {
    tmpr -= y2-vidSizeY;
  }
  {
    register tmp = (rightxs-leftxs)>>8;
    if( tmp ) {
      ls = (rightls-leftls)/tmp         << 8;
      zs = (rightzs-leftzs)/tmp         << 8;
      txs = (righttxs-lefttxs)/tmp      << 8;
      tys = (righttys-lefttys)/tmp      << 8;
    }
    else {
      zs = ls = txs = tys = 0;
    }
  }
  DrawFace();
}

Face::Face( Point* p1, Point* p2, Point* p3, uchar** _yOffset )
{
  vert[0]=p1; vert[1]=p2; vert[2]=p3;
  yOffset = _yOffset;
  int x1 = vert[0]->x, y1 = vert[0]->y,
      z1 = vert[0]->z, x2 = vert[1]->x,
      y2 = vert[1]->y, z2 = vert[1]->z,
      x3 = vert[2]->x, y3 = vert[2]->y,
      z3 = vert[2]->z;
   normal.x = (y2-y1)*(z3-z1)-(z2-z1)*(y3-y1);
   normal.y = (z2-z1)*(x3-x1)-(x2-x1)*(z3-z1);
   normal.z = (x2-x1)*(y3-y1)-(y2-y1)*(x3-x1);
}


FacedObject::FacedObject( uint _flags ) :
 Object3D(), points(), faces(), normals(), tmp_points(), tmp_faces(),
 flags(_flags), perspect(1)
{
}

FacedObject::FacedObject( FacedObject& o ) :
 Object3D(o.dx,o.dy,o.dz,o.rx,o.ry,o.rz,o.lx,o.ly,o.lv,o.lo,o.sx,o.sy,o.sz),
 points(), faces(), normals(), tmp_points(), tmp_faces(), flags(o.flags),
 perspect(1)
{
  int i;
  for( i=0; i<o.points.Count; i++ ) {
    Point* p = (Point*)o.points.At(i);
    points.Insert( new Point(*p) );
    tmp_points.Insert( new Point(*p) );
    Vector* n = (Vector*)o.normals.At(i);
    normals.Insert( new Vector(*n) );
  }
  for( i=0; i<o.faces.Count; i++ ) {
    Face* f = (Face*)o.faces.At(i);
    int n[3];
    for( int j=0; j<3; j++ ) {
      n[j] = o.points.IndexOf( f->vert[j] );
    }
    faces.Insert( new Face( (Point*)points.At(n[0]), (Point*)points.At(n[1]),
                            (Point*)points.At(n[2]),
                            f->yOffset, f->normal ) );
    tmp_faces.Insert( new Face( (Point*)tmp_points.At(n[0]),
                                (Point*)tmp_points.At(n[1]),
                                (Point*)tmp_points.At(n[2]),
                                f->yOffset, f->normal ) );
  }
}

const char Spaces[] = {' ',9,10,13,':',',','\"'};

#define GetToken(x)     getToken(x,Spaces)

double GetNumber( FILE* fp ) { return atof( GetToken(fp) ); }

bool WaitToken( char* token, FILE* fp )
{
//  printf( "WaitToken: %s\n", token );
  char* s;
  do {
    s = GetToken(fp);
    if( !strcmp( s, "EOF" ) ) return false;
  } while ( strcmp(s,token) );
  return true;
}

void FacedObject::ImportASC( char* filename, char* name, double Scale,
                             Image* texture, int mapping )
{
  FILE* fp = resOpenFile(filename);
  uchar** yOffset = new uchar*[texture->sizeY];
  int i;
  for( i=0; i<texture->sizeY; i++ ) {
    yOffset[i] = texture->data+i*texture->bytesPerLine;
  }
  fpos_t start;
  fgetpos( fp, &start );
  double MinX = 999999, MinY = 999999, MinZ = 999999,
         MaxX = -999999, MaxY = -999999, MaxZ = -999999;
  while( WaitToken( "Vertices", fp ) ) {
    int NVertices = GetNumber(fp);
    if( NVertices > 2000 ) error( "Too large ASC" );
    for( i=0; i<NVertices; i++ ) {
      WaitToken( "Vertex", fp );
      WaitToken( "X", fp );
      double X = GetNumber(fp);
      MinX = min( X, MinX );
      MaxX = max( X, MaxX );
      WaitToken( "Y", fp );
      double Y = GetNumber(fp);
      MinY = min( Y, MinY );
      MaxY = max( Y, MaxY );
      WaitToken( "Z", fp );
      double Z = GetNumber(fp);
      MinZ = min( Z, MinZ );
      MaxZ = max( Z, MaxZ );
    }
  };
  double Max = max( MaxX-MinX, MaxZ-MinZ ); Max = max( Max, MaxY-MinY );
  maxx = (MaxX-(MaxX+MinX)/2)*Scale / Max;
  minx = (MinX-(MaxX+MinX)/2)*Scale / Max;
  maxz = (MaxY-(MaxY+MinY)/2)*Scale / Max;
  minz = (MinY-(MaxY+MinY)/2)*Scale / Max;
  miny = -(MaxZ-(MaxZ+MinZ)/2)*Scale / Max;
  maxy = -(MinZ-(MaxZ+MinZ)/2)*Scale / Max;

  clearerr(fp);
  fsetpos( fp, &start );
  char* Name;
  do {
    WaitToken( "Named", fp );
    WaitToken( "object", fp );
    Name = GetToken(fp);
  } while( name && strcmp(Name,name) && !resEof(fp) );
  WaitToken( "Vertices", fp );
  int NVertices = GetNumber(fp);
  WaitToken( "Faces", fp );
  int NFaces = GetNumber(fp);
  int wasPoints = points.Count;
  for( i=0; i<NVertices; i++ ) {
    WaitToken( "X", fp );
    int X = (GetNumber(fp)-(MaxX+MinX)/2)*Scale / (Max);
    WaitToken( "Y", fp );
    int Y = (GetNumber(fp)-(MaxY+MinY)/2)*Scale / (Max);
    WaitToken( "Z", fp );
    int Z = (GetNumber(fp)-(MaxZ+MinZ)/2)*Scale / (Max);
    int TX, TY;
    if( mapping & MAP_NOSCALE ) {
      if( mapping & MAP_XY ) {
        TX=X+VID_MAX_SIZE_X/2;
        TY=-Z+VID_MAX_SIZE_Y/2;
      }
      if( mapping & MAP_XZ ) {
        TX=X+VID_MAX_SIZE_X/2;
        TY=Y+VID_MAX_SIZE_Y/2;
      }
      if( mapping & MAP_YZ ) {
        TX=-Z+VID_MAX_SIZE_X/2;
        TY=Y+VID_MAX_SIZE_Y/2;
      }
    }
    else {
      if( mapping & MAP_XY ) {
        TX=(X+Scale/2)*(texture->sizeX-1)/Scale,
        TY=(-Z+Scale/2)*(texture->sizeY-1)/Scale;
      }
      if( mapping & MAP_XZ ) {
        TX=(X+Scale/2)*(texture->sizeX-1)/Scale,
        TY=(Y+Scale/2)*(texture->sizeY-1)/Scale;
      }
      if( mapping & MAP_YZ ) {
        TX=(-Z+Scale/2)*(texture->sizeX-1)/Scale,
        TY=(Y+Scale/2)*(texture->sizeY-1)/Scale;
      }
    }
    if( mapping & MAP_SPHERE ) {
      TX = Y ? (atan2(Y,X*2)+M_PI)/M_PI/2*(texture->sizeX-1) : texture->sizeX-1,
      TY = ( atan2(Z,sqrt(sqr(X)+sqr(Y)))+M_PI )/M_PI/2*(texture->sizeY-1);
    }
/*
    if( flags & MAP_NORMAL ) {
      p->tx = (n->x/2+128);
      p->ty = (n->y/2+128);
    }
*/
    points.Insert( new Point( X, -Z, Y, TX, TY ) );
    tmp_points.Insert( new Point(X, -Z, Y, TX, TY ) );
    normals.Insert( new Vector(0,0,0) );
  }
  for( i=0; i<NFaces; i++ ) {
    Point* p[3];
    int n[3];
    char token[3]=" ";
    *token='A';
    for( int j=0; j<3; j++ ) {
      WaitToken( token, fp ); (*token)++;
      p[j] = (Point*)points.At(n[j]=(GetNumber(fp)+wasPoints));
    }
/*
    int xs = abs(p[0]->x-p[1]->x)+abs(p[1]->x-p[2]->x)+abs(p[0]->x-p[2]->x),
        ys = abs(p[0]->y-p[1]->y)+abs(p[1]->y-p[2]->y)+abs(p[0]->y-p[2]->y),
        zs = abs(p[0]->z-p[1]->z)+abs(p[1]->z-p[2]->z)+abs(p[0]->z-p[2]->z),
        tx,ty,tz;
    for( j=0; j<3; j++ ) {
      tx = ((p[j]->x+Scale/2)*255/Scale)*65536,
      ty = ((p[j]->y+Scale/2)*255/Scale)*65536,
      tz = ((p[j]->z+Scale/2)*255/Scale)*65536;
      if( xs > zs ) {
        p[j]->tx = tx;
        if( ys > zs ) {
          p[j]->ty = ty;
        }                             //x = tx
        else {                        //       y = ty
          p[j]->ty = tz;              //         = tz
        }                             //  = tz
      }                               //       y = ty
      else {                          //         = tx
        p[j]->tx = tz;                //
        if( ys > xs ) {               //       tx,ty
          p[j]->ty = ty;              //       tx,tz
        }                             //       tz,ty
        else {                        //       tz,tx <->
          p[j]->tx = tx;
          p[j]->ty = tz;
        }
      }
    }
*/
    faces.Insert( new Face( p[0], p[1], p[2], yOffset ) );
    tmp_faces.Insert( new Face( (Point*)tmp_points.At(n[0]),
                                (Point*)tmp_points.At(n[1]),
                                (Point*)tmp_points.At(n[2]),
                                yOffset ) );
    }
  resClose(fp);
}

void FacedObject::Prepare()
{
  int i;
  for( i=0; i<faces.Count; i++ ) {
    Face* f = (Face*)faces.At(i);
    for( int j=0; j<3; j++ ) {
      Vector* n = (Vector*)normals.At( points.IndexOf(f->vert[j]) );
      n->x += f->normal.x;
      n->y += f->normal.y;
      n->z += f->normal.z;
    }
    f = (Face*)tmp_faces.At(i);
    f->normal.Normalize();
  }
  for( i=0; i<normals.Count; i++ ) {
    Vector* n = (Vector*)normals.At(i);
    n->Normalize();
    Point* p = (Point*)points.At(i);
  }
}

void FacedObject::Draw( short* zbuffer, PAGE color, PAGE bw )
{
  ::zbuffer = zbuffer;
  int i;
  for( i=0; i<points.Count; i++ ) {
    Point* p = (Point*)tmp_points.At(i);
    *p = *(Point*)points.At(i);
    Vector n = *(Vector*)normals.At(i);
    n.Rotate(0,0,0,rx,ry,rz);
    if( (ly||lx) && abs(ly-dy)<VID_MAX_SIZE_Y/2 &&
                    abs(lx-dx)<VID_MAX_SIZE_X/2 ) {
      n.Rotate(0,0,0,asinTable[(ly-dy+VID_MAX_SIZE_Y/2)*asinSteps/VID_MAX_SIZE_Y],
                     asinTable[(lx-dx+VID_MAX_SIZE_X/2)*asinSteps/VID_MAX_SIZE_X],0);
    }
    if( flags & TWO_SIDE ) {
      p->l = abs(n.z) << 16;
    }
    else {
      p->l = n.z > 0 ? n.z << 16 : 0;
    }
    p->l = (255<<16)*lo + p->l*lv;
    if( p->l > (255<<16) ) p->l = (255<<16);
    p->Rotate(0,0,0,rx,ry,rz);
    p->x *= sx; p->y *= sy; p->z *= sz;
    p->Move(dx,dy,dz);
    if( flags & PERSPECT ) {
      double scale = (0.5+0.5*(MAX_Z-p->z*perspect)/MAX_Z);
      p->x = (int)vidScaleX((p->x/scale+VID_MAX_SIZE_X/2))<<16;
      p->y = (int)vidScaleY((p->y/scale+VID_MAX_SIZE_Y/2))<<16;
    }
    else {
      p->x = vidScaleX(p->x+VID_MAX_SIZE_X/2)<<16;
      p->y = vidScaleY(p->y+VID_MAX_SIZE_Y/2)<<16;
    }
    p->z <<= 16;
    p->tx <<= 16; p->ty <<= 16;
  }
  if( flags & MAP_BKGRND ) {
    setDrawLineNoTexture(bw);
  }
  else {
    if( flags & SHADOW ) {
      setDrawLineShadow( color, bw,
        int(vidScaleY(ly/2.5-dy)*sy)*vidBytesPerLine - vidScaleX(lx/2.5-dx)*sx );
    }
    else {
      setDrawLineNoShadow( color, bw );
    }
  }
  for( i=0; i<faces.Count; i++ ) {
    Face* f= (Face*)tmp_faces.At(i);
    if( flags & TWO_SIDE ) {
      f->Draw();
    }
    else {
      Vector n = f->normal;
      n.Rotate(0,0,0,rx,ry,rz);
      if( n.z > 0 ) {
        f->Draw();
      }
    }
  }
}

inline int morph( int from, int to, int step, int steps )
{
  return from+(to-from)*step/steps;
}

void FacedObject::Morph( FacedObject& o1, FacedObject& o2,
                         int step, int steps, bool morphTexture )
{
  if( points.Count != o1.points.Count || points.Count != o2.points.Count ||
      faces.Count != o1.faces.Count || faces.Count != o2.faces.Count ) {
    error( "Trying to morph different objects" );
  }
//  if( step<0 || step>=steps ) error( "Morph step is out of range" );
  if( step<0 ) step=0;
  if( step>steps ) step=steps;
  int i;
  for( i=0; i<points.Count; i++ ) {
    Point* p = (Point*)points.At(i);
    Point* p1 = (Point*)o1.points.At(i);
    Point* p2 = (Point*)o2.points.At(i);
    p->x = morph( p1->x, p2->x, step, steps );
    p->y = morph( p1->y, p2->y, step, steps );
    p->z = morph( p1->z, p2->z, step, steps );
    if( morphTexture ) {
      p->tx = morph( p1->tx, p2->tx, step, steps );
      p->ty = morph( p1->ty, p2->ty, step, steps );
    }
    Vector* n = (Vector*)normals.At(i);
    Vector* n1 = (Vector*)o1.normals.At(i);
    Vector* n2 = (Vector*)o2.normals.At(i);
    n->x = morph( n1->x, n2->x, step, steps );
    n->y = morph( n1->y, n2->y, step, steps );
    n->z = morph( n1->z, n2->z, step, steps );
  }
  for( i=0; i<faces.Count; i++ ) {
    Vector* n = &(((Face*)faces.At(i))->normal);
    Vector* n1 = &(((Face*)o1.faces.At(i))->normal);
    Vector* n2 = &(((Face*)o2.faces.At(i))->normal);
    n->x = morph( n1->x, n2->x, step, steps );
    n->y = morph( n1->y, n2->y, step, steps );
    n->z = morph( n1->z, n2->z, step, steps );
  }
}

void FacedObject::MoveAbs( int dx, int dy, int dz )
{
  for( int i=0; i<points.Count; i++ ) {
    ((Point*)(points.At(i)))->Move( dx, dy, dz );
  }
}

void FacedObject::RotateAbs( int rx, int ry, int rz )
{
  for( int i=0; i<points.Count; i++ ) {
    ((Point*)(points.At(i)))->Rotate( 0,0,0, rx, ry, rz );
  }
}

void FacedObject::Store( char* filename )
{
  FILE* fp = fopen( filename, "wt" );
  fprintf( fp, "%d %d %d %d %d %d %d %d %d %d %d %d %lf %lf %lf %lf %lf"
	       " %d %lf %d ",
	   maxx, maxy, maxz, minx, miny, minz, dx, dy, dz, rx, ry, rz, lv, lo,
	   sx, sy, sz, flags, perspect, points.Count );
  int i;
  for( i=0; i<points.Count; i++ ) {
    ((Point*)points.At(i))->Store(fp);
    ((Vector*)normals.At(i))->Store(fp);
  }
  fprintf( fp, "%d ", faces.Count );
  for( i=0; i<faces.Count; i++ ) {
    Face* f = (Face*)faces.At(i);
    for( int j=0; j<3; j++ ) {
      int tmp=points.IndexOf( f->vert[j] );
      fprintf( fp, "%d ", tmp );
    }
    f->normal.Store(fp);
  }
  resClose(fp);
}

#define TEXT_LOAD

void FacedObject::Load( char* filename, Image* texture )
{
  FILE* fp = resOpenFile( filename );
  uchar** yOffset = new uchar*[texture->sizeY];
  int i;
  for( i=0; i<texture->sizeY; i++ ) {
    yOffset[i] = texture->data+i*texture->bytesPerLine;
  }
  int oldPointsCount = points.Count;
  int pointsCount;
#ifndef TEXT_LOAD
  fread( &maxx, sizeof(int), 1, fp );
  fread( &maxy, sizeof(int), 1, fp );
  fread( &maxz, sizeof(int), 1, fp );
  fread( &minx, sizeof(int), 1, fp );
  fread( &miny, sizeof(int), 1, fp );
  fread( &minz, sizeof(int), 1, fp );
  fread( &dx, sizeof(int), 1, fp );
  fread( &dy, sizeof(int), 1, fp );
  fread( &dz, sizeof(int), 1, fp );
  fread( &rx, sizeof(int), 1, fp );
  fread( &ry, sizeof(int), 1, fp );
  fread( &rz, sizeof(int), 1, fp );
  fread( &lv, sizeof(double), 1, fp );
  fread( &lo, sizeof(double), 1, fp );
  fread( &sx, sizeof(double), 1, fp );
  fread( &sy, sizeof(double), 1, fp );
  fread( &sz, sizeof(double), 1, fp );
  fread( &flags, sizeof(int), 1, fp );
  fread( &perspect, sizeof(double), 1, fp );
  fread( &pointsCount, sizeof(int), 1, fp );
#else
  fscanf( fp, "%d %d %d %d %d %d %d %d %d %d %d %d %lf %lf %lf %lf %lf"
	       "%d %lf %d ",
	   &maxx, &maxy, &maxz, &minx, &miny, &minz, &dx, &dy, &dz, 
	   &rx, &ry, &rz, &lv, &lo, &sx, &sy, &sz, &flags, &perspect, 
	   &pointsCount );
#endif
  for( i=0; i<pointsCount; i++ ) {
    Point* p = new Point; 
#ifdef TEXT_LOAD
    p->Load(fp);
#else
    fread( p, sizeof(Point), 1, fp );
#endif
    points.Insert(p);
    tmp_points.Insert(new Point(*p));
    Vector* n = new Vector; 
#ifdef TEXT_LOAD
    n->Load(fp);
#else
    fread( n, sizeof(Vector), 1, fp );
#endif
    normals.Insert(n);
  }
  int facesCount;
#ifdef TEXT_LOAD
  fscanf( fp, "%d ", &facesCount );
#else
  fread( &facesCount, sizeof(int), 1, fp );
#endif
  for( i=0; i<facesCount; i++ ) {
    int n[3];
#ifndef TEXT_LOAD
    for( int j=0; j<3; j++ ) {
      fread( &(n[j]), sizeof(int), 1, fp );
    }
#else
    fscanf( fp, "%d %d %d ", &n[0], &n[1], &n[2] );
#endif
    Face* f = new Face( (Point*)points.At(n[0]+oldPointsCount),
                        (Point*)points.At(n[1]+oldPointsCount),
                        (Point*)points.At(n[2]+oldPointsCount),
                        yOffset);
#ifndef TEXT_LOAD
    fread( &f->normal, sizeof(Vector), 1, fp );
#else
    f->normal.Load(fp);
#endif
    faces.Insert(f);
    tmp_faces.Insert( new Face( (Point*)tmp_points.At(n[0]+oldPointsCount),
                                (Point*)tmp_points.At(n[1]+oldPointsCount),
                                (Point*)tmp_points.At(n[2]+oldPointsCount),
                                yOffset ) );
  }
  resClose(fp);
//  Store(filename);
}

void Vector::Rotate( int cx, int cy, int cz, int rx, int ry, int rz )
{
  int TX = x-cx, TY = y-cy, TZ = z-cz;
  if(ry) {
    if( ry < 0 ) ry = ry % cosSteps + cosSteps;
    else ry %= cosSteps;
    int T1 = TX * cosTable3D[ ry ];
    int T2 = TZ * sinTable3D[ ry ];
    int T3 = TX * sinTable3D[ ry ];
    int T4 = TZ * cosTable3D[ ry ];
    TX = ( T1 - T2 ) >> cosScale3D;
    TZ = ( T3 + T4 ) >> cosScale3D;
  }
  if(rx) {
    if( rx < 0 ) rx = rx % cosSteps + cosSteps;
    else rx %= cosSteps;
    int T1 = TY * cosTable3D[ rx ];
    int T2 = TZ * sinTable3D[ rx ];
    int T3 = TY * sinTable3D[ rx ];
    int T4 = TZ * cosTable3D[ rx ];
    TY = ( T1 - T2 ) >> cosScale3D;
    TZ = ( T3 + T4 ) >> cosScale3D;
  }
  if(rz) {
    if( rz < 0 ) rz = rz % cosSteps + cosSteps;
    else rz %= cosSteps;
    int T1 = TY * cosTable3D[ rz ];
    int T2 = TX * sinTable3D[ rz ];
    int T3 = TY * sinTable3D[ rz ];
    int T4 = TX * cosTable3D[ rz ];
    TY = ( T1 - T2 ) >> cosScale3D;
    TX = ( T3 + T4 ) >> cosScale3D;
  }
  x = TX+cx; y = TY+cy; z = TZ+cz;
}
