/*
 *
 *   HElliZER: the first portable demo in the world
 *
 *   Copyright (C) 1996  Queue Members Group Art Division
 *   Coded by Mad Max / Queue Members Group (Mike Shirobokov)
 *   <mad_max@dixon.volgacom.samara.su>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 */
#include <stdio.h>
#include <string.h>
#include "image.h"
#include "video.h"
#include "resource.h"
#include "misc.h"

struct rgb { uchar r,g,b; };

Image::Image( char* name, int stretch ) {
  uchar* buf = (uchar*)resGetResource(name);
  bytesPerLine = sizeX = (*(buf+8)<<8) + *(buf+9);
  sizeY = (*(buf+10)<<8) + *(buf+11);
//  printf( "name=%s sizeX=%d sizeY=%d\n", name, sizeX, sizeY );
  rgb* tmp = (rgb*)(buf+32);
  palette = (RGB*)cmalloc(sizeof(RGB)*256);
  for( int i=0; i<256; i++ ) {
    palette[i].red = tmp[i].r;
    palette[i].green = tmp[i].g;
    palette[i].blue = tmp[i].b;
  }
  data = buf+800; loaded = true;
  switch(stretch) {
    case STRETCH: {
      StretchTo(vidScaleX(sizeX),vidScaleY(sizeY),vidScaleX(bytesPerLine));
      break;
    }
    case STRETCH_BIT0: {
      StretchToBit( vidScaleX(sizeX), vidScaleY(sizeY), false );
      break;
    }
    case STRETCH_BIT1: {
      StretchToBit( vidScaleX(sizeX), vidScaleY(sizeY), true );
      break;
    }
    case STRETCH_SCREEN: {
      StretchTo( vidSizeX, vidSizeY, vidBytesPerLine);
      break;
    }
  }
}

Image::Image( int sizex, int sizey, int bytesperline, RGB pal[256], uchar* buf )
{
  sizeX = sizex;
  sizeY = sizey;
  bytesPerLine = bytesperline;
  palette = pal;
  data = buf;
  loaded = false;
}

Image::~Image()
{
  if( loaded )
    cfree(data-800);
  else
    cfree(data);
  cfree(palette);
}

void Image::Show( int x, int y, PAGE page )
{
  if( x > vidSizeX || y > vidSizeY ) return;
  int offsetx=0, offsety=0, offsetpage=0;
  if( x < 0 ) {
    offsetx = -x;
    x = 0;
  }
  int sizex = min( sizeX - offsetx, vidSizeX-x );
  if( sizex <= 0 ) return;
  if( y < 0 ) {
    offsety = -y;
    y = 0;
  }
  int sizey = min( sizeY - offsety, vidSizeY-y );
  if( sizey <= 0 ) return;

//  printf( "sizex=%d  sizey=%d\n",sizex,sizey);

  uchar* from = data + offsety*bytesPerLine + offsetx;
  uchar* to = page + y*vidBytesPerLine + x;
  for( int i=0; i<sizey; i++ ) {
    memcpy( to, from, sizex );
    to += vidBytesPerLine;
    from += bytesPerLine;
  }
}

void Image::ShowT( int x, int y, PAGE page )
{
  if( x > vidSizeX || y > vidSizeY ) return;
  int offsetx=0, offsety=0, offsetpage=0;
  if( x < 0 ) {
    offsetx = -x;
    x = 0;
  }
  int sizex = min( sizeX - offsetx, vidSizeX-x );
  if( sizex <= 0 ) return;
  if( y < 0 ) {
    offsety = -y;
    y = 0;
  }
  int sizey = min( sizeY - offsety, vidSizeY-y );
  if( sizey <= 0 ) return;

//  printf( "sizex=%d  sizey=%d\n",sizex,sizey);

  uchar* from = data + offsety*bytesPerLine + offsetx;
  uchar* to = page + y*vidBytesPerLine + x;
  for( int i=0; i<sizey; i++ ) {
    for( int j=0; j<sizex; j++ ) {
      if( from[j] ) to[j]=from[j];
    }
    to += vidBytesPerLine;
    from += bytesPerLine;
  }
}

void Image::ShowAdd( int x, int y, PAGE page )
{
  if( x > vidSizeX || y > vidSizeY ) return;
  int offsetx=0, offsety=0, offsetpage=0;
  if( x < 0 ) {
    offsetx = -x;
    x = 0;
  }
  int sizex = min( sizeX - offsetx, vidSizeX-x );
  if( sizex <= 0 ) return;
  if( y < 0 ) {
    offsety = -y;
    y = 0;
  }
  int sizey = min( sizeY - offsety, vidSizeY-y );
  if( sizey <= 0 ) return;

//  printf( "sizex=%d  sizey=%d\n",sizex,sizey);

  uchar* from = data + offsety*bytesPerLine + offsetx;
  uchar* to = page + y*vidBytesPerLine + x;
  for( int i=0; i<sizey; i++ ) {
    for( int j=0; j<sizex; j++ ) {
      to[j] = from[j]+to[j] <= 255 ? from[j]+to[j] : 255;
    }
    to += vidBytesPerLine;
    from += bytesPerLine;
  }
}

void Image::ShowTB( int x, int y, uchar color, PAGE page )
{
  if( x > vidSizeX || y > vidSizeY ) return;
  int offsetx=0, offsety=0, offsetpage=0;
  if( x < 0 ) {
    offsetx = -x;
    x = 0;
  }
  int sizex = min( sizeX - offsetx, vidSizeX-x );
  if( sizex <= 0 ) return;
  if( y < 0 ) {
    offsety = -y;
    y = 0;
  }
  int sizey = min( sizeY - offsety, vidSizeY-y );
  if( sizey <= 0 ) return;

//  printf( "sizex=%d  sizey=%d\n",sizex,sizey);

  uchar* from = data + offsety*bytesPerLine + offsetx/8;
  uchar* to = page + y*vidBytesPerLine + x;
  for( int i=0; i<sizey; i++ ) {
    register mask = 0x80 >> (offsetx&7);
    for( int j=0,k=0; j<sizex; j++ ) {
      if( from[k] & mask ) to[j]=color;
      mask >>= 1; if( !mask ) { mask = 0x80; k++; }
    }
    to += vidBytesPerLine;
    from += bytesPerLine;
  }
}

void Image::ShowR( int x, int y, int w, int h, PAGE page )
{
  if( x > vidSizeX || y > vidSizeY ) return;
  int offsetx=0, offsety=0, offsetpage=0;
  if( x < 0 ) {
    offsetx = -x;
    x = 0;
  }
  int size_x = min( w - offsetx, vidSizeX-x );
  if( size_x <= 0 ) return;
  if( y < 0 ) {
    offsety = -y;
    y = 0;
  }
  int size_y = min( h - offsety, vidSizeY-y );
  if( size_y <= 0 ) return;

  uchar* from = data + (offsety*(sizeY-1)/h)*bytesPerLine +
                       offsetx*(sizeX-1)/w, *from1=from;
  uchar* to = page + y*vidBytesPerLine + x;
  int stepy = (sizeY<<16)/h; y=0;
  for( int i=0; i<size_y; i++, y+=stepy ) {
    from1 = from+(y>>16)*bytesPerLine;
    int stepx = (sizeX<<16)/w;
    for( int j=0, x=0; j<size_x; j++, x+=stepx ) {
      to[j] = from1[x>>16];
    }
    to += vidBytesPerLine;
  }
}

void Image::ShowRT( int x, int y, int w, int h, PAGE page )
{
  if( x > vidSizeX || y > vidSizeY ) return;
  int offsetx=0, offsety=0, offsetpage=0;
  if( x < 0 ) {
    offsetx = -x;
    x = 0;
  }
  int size_x = min( w - offsetx, vidSizeX-x );
  if( size_x <= 0 ) return;
  if( y < 0 ) {
    offsety = -y;
    y = 0;
  }
  int size_y = min( h - offsety, vidSizeY-y );
  if( size_y <= 0 ) return;

  uchar* from = data + (offsety*(sizeY-1)/h)*bytesPerLine +
                       offsetx*(sizeX-1)/w, *from1=from;
  uchar* to = page + y*vidBytesPerLine + x;
  int stepy = (sizeY<<16)/h; y=0;
  for( int i=0; i<size_y; i++, y+=stepy ) {
    from1 = from+(y>>16)*bytesPerLine;
    register stepx = (sizeX<<16)/w;
    for( int j=0, x=0; j<size_x; j++, x+=stepx ) {
      if( from1[x>>16] ) to[j] = from1[x>>16];
    }
    to += vidBytesPerLine;
  }
}

void Image::ShowRTC( int x, int y, int w, int h, uchar color, PAGE page )
{
  if( x > vidSizeX || y > vidSizeY ) return;
  int offsetx=0, offsety=0, offsetpage=0;
  if( x < 0 ) {
    offsetx = -x;
    x = 0;
  }
  int size_x = min( w - offsetx, vidSizeX-x );
  if( size_x <= 0 ) return;
  if( y < 0 ) {
    offsety = -y;
    y = 0;
  }
  int size_y = min( h - offsety, vidSizeY-y );
  if( size_y <= 0 ) return;

  uchar* from = data + (offsety*(sizeY-1)/h)*bytesPerLine +
                       offsetx*(sizeX-1)/w, *from1=from;
  uchar* to = page + y*vidBytesPerLine + x;
  int stepy = (sizeY<<16)/h; y=0;
  for( int i=0; i<size_y; i++, y+=stepy ) {
    from1 = from+(y>>16)*bytesPerLine;
    register stepx = (sizeX<<16)/w;
    for( int j=0, x=0; j<size_x; j++, x+=stepx ) {
      if( from1[x>>16] ) to[j] = color;
    }
    to += vidBytesPerLine;
  }
}

void Image::ShowRTAdd( int x, int y, int w, int h, PAGE page )
{
  if( x > vidSizeX || y > vidSizeY ) return;
  int offsetx=0, offsety=0, offsetpage=0;
  if( x < 0 ) {
    offsetx = -x;
    x = 0;
  }
  int size_x = min( w - offsetx, vidSizeX-x );
  if( size_x <= 0 ) return;
  if( y < 0 ) {
    offsety = -y;
    y = 0;
  }
  int size_y = min( h - offsety, vidSizeY-y );
  if( size_y <= 0 ) return;

  uchar* from = data + (offsety*(sizeY-1)/h)*bytesPerLine +
                       offsetx*(sizeX-1)/w, *from1=from;
  uchar* to = page + y*vidBytesPerLine + x;
  int stepy = (sizeY<<16)/h; y=0;
  for( int i=0; i<size_y; i++, y+=stepy ) {
    from1 = from+(y>>16)*bytesPerLine;
    register stepx = (sizeX<<16)/w;
    for( int j=0, x=0; j<size_x; j++, x+=stepx ) {
      to[j] = from1[x>>16]+to[j] <= 255 ? from1[x>>16]+to[j] : 255;
    }
    to += vidBytesPerLine;
  }
}

void Image::StretchTo( int newx, int newy, int newbytesperline )
{
//  printf( "vidBytesPerLine=%d vidSizeX=%d vidSizeY=%d sizeX=%d sizeY=%d\n",
//          vidBytesPerLine, vidSizeX, vidSizeY, sizeX, sizeY );

  if( newbytesperline >= bytesPerLine && newy >= sizeY ) return;
  for( int y=0; y<newy; y++ ) {
    for( int x=0; x<newx; x++ ) {
      data[ y*newbytesperline+x ] =
        data[ y*(sizeY-1)/newy*bytesPerLine+x*(sizeX-1)/newx ];
    }
  }
  sizeX = newx; sizeY = newy; bytesPerLine = newbytesperline;
  if(loaded)
    data=(uchar*)crealloc(data-800,sizeY*bytesPerLine+800)+800;
  else
    data=(uchar*)crealloc(data,sizeY*bytesPerLine);
}

void Image::StretchToBit( int newx, int newy, bool neg )
{
//  printf( "vidBytesPerLine=%d vidSizeX=%d vidSizeY=%d sizeX=%d sizeY=%d\n",
//          vidBytesPerLine, vidSizeX, vidSizeY, sizeX, sizeY );
  int newbytesperline = (newx+7)/8;
  uchar* newdata=(uchar*)cmalloc( 800+newy*newbytesperline );
  memcpy( newdata, data-800, 800 );
  newdata+=800;
  if( newx > sizeX && newy > sizeY ) return;
  for( int y=0; y<newy; y++ ) {
    memset( newdata+y*newbytesperline, 0, newbytesperline );
    uchar mask=0x1;
    for( int x=0; x<newx; x++ ) {
      newdata[ y*newbytesperline+x/8 ] <<= 1;
      if( (data[ y*(sizeY-1)/newy*bytesPerLine+x*(sizeX-1)/newx ]==0) == neg ) {
        newdata[ y*newbytesperline+x/8 ] |= 1;
      }
    }
  }
  cfree(data-800); data=newdata;
  sizeX = newx; sizeY = newy; bytesPerLine = newbytesperline;
}
