/* ball.c - 
   Copyright (C) 2000 Tijs van Bakel and Jorik Blaas.
   Tijs van Bakel <smoke@casema.net>
   Jorik Blaas <jrk@panic.et.tudelft.nl>
 
 This file is part of a silly intro
 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with This program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <stdlib.h>
#include <math.h>

#include "crap_png.h"
#include "crap_video.h"
#include "crap_lib.h"
#include "ball.h"
#include "crap_mod.h"
#include "crap_font.h"

uint8 multable[64*64];

#define XCHG(x,y) { int temp; temp=x; x=y; y=temp; }

void
glenz_line ( Image* dest_image, int x0, int y0, int x1, int y1, uint8 c0, uint8 c1 )
{
  int w,h;
  int dx, dy;
  int dcolor;
  int color;
  int x,y;
  uint8* pos;
  uint8* dest;
  int vwidth = dest_image->width;
  int vheight = dest_image->height;

  dest = dest_image->buffer;

  if ((x0 < 0) || (x1 < 0) || (y0 < 0) || (y1 < 0) ||
      (x0 >= vwidth) || (x1 >= vwidth) || (y0 >= vheight) || (y1 >= vheight))
    {
      return;
    }

  w = (x1-x0);
  h = (y1-y0);

  if ( abs(h) > abs(w) )
    {
      if ( y1 < y0 )
	{
	  h = -h;
	  w = -w;
	  XCHG ( y1, y0 );
	  XCHG ( x1, x0 );
	  XCHG ( c1, c0 );
	}

      if ( h == 0 )
	return;
      
      dest += y0 * vwidth + x0;

      dcolor = ( (c1-c0)<<16 ) / h;
      dx = ( w << 16 ) / h;
      color = (c0<<16);
      x = 0;
      for ( y = 0; y < h; y++ )
	{
	  x += dx;
	  color += dcolor;
	  pos = dest + (x>>16);
	  *pos = multable[ (*pos) * 64 + (color>>16)];
	  dest += vwidth;
	}
    }
  else
    {
      if ( x1 < x0 )
	{
	  w = -w;
	  h = -h;
	  XCHG ( y1, y0 );
	  XCHG ( x1, x0 );
	  XCHG ( c1, c0 );
	}

      if ( w == 0 )
	return;
     
      dest += y0 * vwidth + x0;

      dcolor = ( (c1-c0)<<16 ) / w;
      dy = ( h << 16 ) / w;
      color = (c0<<16);
      y = 0;
      for ( x = 0; x < w; x++ )
	{
	  y += dy;
	  color += dcolor;
	  pos = dest + (y>>16)*320;
	  *pos = multable[ (*pos) * 64 + (color>>16)];
	  dest ++;
	}
    }
}

/* transform WORLD to VIEW coordinates */
void
rotate_sphere ( WorldCoord* world_coord,
		ViewCoord* view_coord,
		int ncoords, float zoom,
		float xangle, float yangle, float zangle )
{
  float sx,sy,sz;
  float cx,cy,cz;
  int i;
  float x,y,z;

  sx = sin(xangle) * zoom; cx = cos(xangle) * zoom;
  sy = sin(yangle) * zoom; cy = cos(yangle) * zoom;
  sz = sin(zangle) * zoom; cz = cos(zangle) * zoom;
  
  for (i=0;i<ncoords;i++)
    {
      x = world_coord[i].x;
      y = world_coord[i].y;
      z = world_coord[i].z;

      /* some z axis */
      view_coord[i].x = cz*x - sz*y;
      view_coord[i].y = sz*x + cz*y;

      x = view_coord[i].x;
      y = view_coord[i].y;

      /* some x axis */
      view_coord[i].y = cx*y - sx*z;
      view_coord[i].z = sx*y + cx*z;

      y = view_coord[i].y;
      z = view_coord[i].z;
      
      /* some y axis */

      view_coord[i].x = cy*x + sy*z;
      view_coord[i].z = -sy*x + cy*z;
    }
}

/* transform VIEW to SCREEN coordinates */
void
perspective_sphere ( ViewCoord* view_coord, ScreenCoord* screen_coord, int ncoords,
		     float center_x, float center_y, float radius )
{
  int i;
  float z;
  
  for (i=0;i<ncoords;i++)
    {
      z = view_coord[i].z + 384.0;
      screen_coord[i].x = center_x + (view_coord[i].x * (1.3 * radius) / z);
      screen_coord[i].y = center_y + (view_coord[i].y * (radius) / z);
    }
}

void
draw_sphere ( Image* dest,
	      float counter,
	      Ball_data* data,
	      int n_coords,
	      float center_x, float center_y,
	      float distort_amount )
{
  int x0,x1,y0,y1;
  int i;
  float xangle, yangle, zangle;
  int color0, color1;
  ViewCoord* view_coords_l;
  ScreenCoord* screen_coords_l;
#define LAYERS 4 
  int l;
  float counter_l;
  float zoom;

  /* rotate and project coords */
  for (l = 0; l < LAYERS; l++)
    { 
      counter_l = counter + (LAYERS-l) * ( 3.0 * sin ( counter/500.0 ) );
      xangle = counter_l / 32.0;
      yangle = counter_l / 20.0; 
      zangle = counter_l / 54.0;
      
      view_coords_l = data->view_coords + l * (data->n_coords/LAYERS);
      screen_coords_l = data->screen_coords + l * (data->n_coords/LAYERS);
      
      zoom = 0.95 + l * 0.0150;
      rotate_sphere ( data->world_coords, view_coords_l, data->n_coords / LAYERS, zoom,
		      xangle , yangle , zangle );
      perspective_sphere ( view_coords_l, screen_coords_l, data->n_coords / LAYERS,
			   center_x, center_y,
			   90.0 + distort_amount * 50.0 );
    }

  /* draw */
  for ( l = 0; l < LAYERS-1 ; l++ ) 
    for ( i = 0; i < data->n_coords/LAYERS; i++ )
    {
      view_coords_l = data->view_coords + l * (data->n_coords/LAYERS);
      screen_coords_l = data->screen_coords + l * (data->n_coords/LAYERS);

      x0 = screen_coords_l[i].x;
      y0 = screen_coords_l[i].y;
      color0 = (int) ( (float) l * ( -view_coords_l[i].z + 256.0 ) / LAYERS ) >> 3;

      view_coords_l = data->view_coords + (l+1) * (data->n_coords/LAYERS);
      screen_coords_l = data->screen_coords + (l+1) * (data->n_coords/LAYERS);

      x1 = screen_coords_l[i].x;
      y1 = screen_coords_l[i].y;
      color1 = (int) ( (float) (l+1.0) * ( -view_coords_l[i].z + 256.0 ) / LAYERS ) >> 3;

/*        glenz_line ( dest, x0, y0, x1, y1, color0*(l)/LAYERS, color1*(l+1)/LAYERS ); */
      glenz_line ( dest, x0, y0, x1, y1, 63-color0, 63-color1 );

    }
}

void calc_sphere ( WorldCoord* coords, int n_coords, float R )
{
  int i;

  float x,y,z;
  float n;

  for ( i = 0; i < n_coords; i++ )
    {
      n = 2.0;
      while ( n > 1.0 )
	{
	  x = frand() * 2.0 - 1.0;
	  y = frand() * 2.0 - 1.0;
	  z = frand() * 2.0 - 1.0;
	  
	  n = sqrt(x*x+y*y+z*z);
	}
      
      x = 1/n * x * R;
      y = 1/n * y * R;
      z = 1/n * z * R;
  
      coords[i].x = x;
      coords[i].y = y;
      coords[i].z = z;
    }
}

void
calc_multable()
{
  int i,j;
  for (i=0;i<64;i++)
    for (j=0;j<64;j++)
      {
	multable[i*64+j] = ( (i-1) * (j-1) ) / 64 + 1;
      }
}

void
ball_palette( Palette* p )
{
  int i;
  
  for ( i = 0; i < 65; i++)
    {
      p->color[i].r = (((i)*3) >> 3) + 5;
      p->color[i].g = 0 + 10;
      p->color[i].b = ((i) >> 1) + 15;
    }
  p->color[0].r = 0;
  p->color[0].g = 0;
  p->color[0].b = 0;
}

Ball_data* create_fx_ball ( Image* dest, int n_coords )
{
  Ball_data* data;
  
  data = (Ball_data*) malloc ( sizeof(Ball_data) );

  data->n_coords = n_coords;
  data->view_coords = malloc ( sizeof (ViewCoord) * n_coords );
  data->screen_coords = malloc ( sizeof (ScreenCoord) * n_coords );
  data->world_coords = malloc ( sizeof (WorldCoord) * n_coords );

  calc_multable();
  calc_sphere( data->world_coords, n_coords, 256.0 );

  data->palette.color = malloc ( sizeof(Color) * 256 );

  data->ball_x_pos = dest->width / 2.0;
  data->ball_y_pos = - dest->height / 2.0;
  
  crap_png_load( &data->background_image, &data->palette, "ball.png" );
  
  return data;
}

void fade_curtain ( Image* dest )
{
  static float curtain_v = 0.0;
  static float curtain_y = 2.0;

  curtain_y += curtain_v;
  curtain_v += 0.2;

  if ( curtain_y > dest->height )
    {
      curtain_y = dest->height;
      curtain_v = 0.0;
    }

  crap_image_rect_fill ( dest, 63,
			 0, 0,
			 dest->width, curtain_y );
  crap_image_rect_fill ( dest, 0,
			 0, curtain_y,
			 dest->width, dest->height + 1 - curtain_y );
}

void fx_ball ( Image* dest, Ball_data* data, int ball_visible, int distort_ball )
{
  float x_distort;
  static float counter = 1500.0;
  static int first_time = 1;

  static float distort_amount = 0.0;
  static int distort_counter = 0;
  
  static float ball_y = 0.0;
  static float ball_v = 0.0;
  static float ball_a = 0.0;

  fade_curtain ( dest );
  
  if ( first_time )
    {
      first_time = 0;
      crap_image_fill ( dest, 0 );
      crap_setpalette ( &data->palette, 0, 128 );
    }

  counter += 1.0;

  if ( ball_visible )
    ball_a = 0.2;

  ball_y += ball_v;
  ball_v += ball_a;

  if ( ball_y > dest->height )
    {
      ball_v = 0.0;
      ball_y = dest->height;
    }
  
  distort_counter++;
  if ( distort_ball )
    {
      distort_counter = 0;
      distort_amount = 1.0;
    }
  distort_amount *= 0.95;

  x_distort = -0.36 + 1.0 / ( 1.75 + sin ( 2.0*M_PI*(1-distort_amount) + 0.5*M_PI));
  draw_sphere ( dest, counter,
		data, data->n_coords,
		data->ball_x_pos, ball_y - dest->height/2,
		x_distort );
}
