/***********************************************************************************
 * FxDemo.cpp           -       Startup code for this application
 *
 * Author:              Ron Bakker        (madrigal@gmx.net)
 * Description:         Starfield screensaver
 * Date:                04-03-1998
 ***********************************************************************************/

#include <windows.h>
#include <dos.h>
#include "glidesystem.h"
#include "4dmath.h"

void FxStars (void);

////////////////////////////////////////////////////////////////////////////////////
// Globals
////////////////////////////////////////////////////////////////////////////////////

long            numberOfNoisePixels = 3000;            // Noise pixels
long            numberOfStars = 1000;                  // the moving stars
long            numberOfRigidStars = 100;              // the rigid stars
GrVertex3D      *stars3D;                               // the moving stars vertices
GrVertex        *noisePixels;                           // the noise pixels vertices
GrVertex        *rigidStars;                            // the rigid stars vertices
GrFog_t         fogtable[GR_FOG_TABLE_SIZE];            // a fogtable
int angle = 0;                                          // start-angle
float rotangle = 1.0f;                                  // cumulative rotation in degrees
float distance = -0.5f;    // distance increase (cumulative translation of the 3d stars)
BOOL rotation = FALSE;                                  // Do rotation ?
float tailLength = 15.0f;  // length of star-tail .. bigger is smaller here
float speed = 0.05f;                                   // moving speed (WARP 1)
float accel = 1.0;         // internal variable for acceleration when noise disappear
float colorGray = 0.0f;    // internal variable for increasing gray-color at the start (3d stars and rigid stars)
static unsigned long randx = 1; // internal random seed

/***********************************************************************************
 * iRandom        - Generate a random integer
 ***********************************************************************************/
static unsigned int iRandom (unsigned int maxr)
{
    unsigned int n,retval;

    if (maxr > 0xFFFFFFF) {
        do {
            retval = iRandom(0xFFFFFFF);
            retval |= iRandom(maxr>>28)<<28;
        } while (retval > maxr);
        return retval;
    }
    for (n=1; n<32; n++)
        if (((unsigned)1 << n) > maxr) break;
    do {
        randx = randx*1103515245 + 12345;
        retval = (randx & 0x7fffffff) >> (31-n);
    } while (retval > maxr);
    return retval;
}

/***********************************************************************************
 * rRandom        - Randomize between boundries s and e
 ***********************************************************************************/
static int rRandom(int s, int e)
{
    return s + iRandom(e-s);
}

/***********************************************************************************
 * InitFxDemo        - Here we initialize any demo data
 ***********************************************************************************/
int InitFxDemo (void)
{
    // allocate star data
    stars3D = (GrVertex3D *) malloc (numberOfStars*sizeof(GrVertex3D));
    rigidStars = (GrVertex *) malloc (numberOfRigidStars*sizeof(GrVertex));
    // randomize initial 3d-stars
    for( int i = 0; i < numberOfStars; i++ )
    {
        stars3D[i].x = (((float)rRandom( 2, 100 )) / 100.0f )  - 0.5f;
        stars3D[i].y = (((float)rRandom( 2, 100 )) / 100.0f )  - 0.5f;
        stars3D[i].z = (((float)rRandom( 2, 1000 )) / 100.0f ) - 0.5f;
        stars3D[i].w = 1.0f;
    }
    // initialize rigid stars
    for( i = 0; i < numberOfRigidStars; i++ )
    {
        rigidStars[i].oow = ((float)rRandom( 1.0f, 10.0f )/10.0f);
        rigidStars[i].a = 255.0f;
        rigidStars[i].r = 255.0f;
        rigidStars[i].g = 255.0f;
        rigidStars[i].b = 255.0f;
        rigidStars[i].x = rand() % 640;
        rigidStars[i].y = rand() % 480;
    }

    // set some state variables
    #ifndef __DEBUG_TEST__        
    grColorCombine( GR_COMBINE_FUNCTION_LOCAL,
                    GR_COMBINE_FACTOR_NONE,
                    GR_COMBINE_LOCAL_CONSTANT,
                    GR_COMBINE_OTHER_NONE,
                    FXFALSE );
    grAlphaCombine( GR_COMBINE_FUNCTION_LOCAL,
                    GR_COMBINE_FACTOR_NONE,
                    GR_COMBINE_LOCAL_ITERATED,
                    GR_COMBINE_OTHER_NONE,
                    FXFALSE );
    grFogMode( GR_FOG_WITH_TABLE );
    grFogColorValue( 0x0 );
    guFogGenerateExp( fogtable, .9f );
    grFogTable( fogtable );
    grAlphaBlendFunction( GR_BLEND_SRC_ALPHA, GR_BLEND_ONE_MINUS_SRC_ALPHA,
                          GR_BLEND_ZERO, GR_BLEND_ZERO );
    #endif

    // Allocate the noise
    noisePixels = (GrVertex *) malloc (numberOfNoisePixels*sizeof(GrVertex));
    // initialize the noise
    for (i = 0; i<numberOfNoisePixels; i++)
    {
        noisePixels[i].x = rand() % 640;
        noisePixels[i].y = rand() % 480;
        noisePixels[i].oow = 1.0f;
        noisePixels[i].a = 255.0f;
        noisePixels[i].r = 255.0f;
        noisePixels[i].g = 255.0f;
        noisePixels[i].b = 255.0f;
     }
    return FXTRUE;      // we succeeded
}

/***********************************************************************************
 * FxIntro              - Renders the intro once and decreases the color of the pixels
 ***********************************************************************************/
BOOL FxIntro (void)
{
    if (noisePixels[0].r == 0.0f)
    // if all noisePixels are black ...
    {
        // when noise-pixel disappear the speed of the 3d-stars must be set down in order
        // to provide a smooth transition
        accel = 7.0;
        return 0;
    }
    else
    // fade and randomize the noise pixels
    {
        int random_number1, random_number2;
        for (long i = 0; i<numberOfNoisePixels; i++)
        {
            random_number1 = rand();
            random_number2 = rand();
            noisePixels[i].x = random_number1 % 640;
            noisePixels[i].y = random_number2 % 480;
            noisePixels[i].r--;   
            noisePixels[i].g--;   
            noisePixels[i].b--;
            #ifndef __DEBUG_TEST__        
            grDrawPoint(&noisePixels[i]);
            #endif
            // this draws twice as much noise pixels (*2)
            noisePixels[i].x = (random_number1*2) % 640;
            noisePixels[i].y = (random_number2*2) % 480;
            #ifndef __DEBUG_TEST__        
            grDrawPoint(&noisePixels[i]);
            #endif
        }
        // the gray-color of the 3d-stars and rigids is increased here as the noise is decreased
        colorGray+=1.0f;
        FxStars();
    }
    return 1;
}

/***********************************************************************************
 * FxRigidStars              - Renders the rigid stars of the star-theme
 ***********************************************************************************/
void FxRigidStars (void)
{
    for( int i = 0; i < numberOfRigidStars; i++ )
    {
        // make the rigids glow a bit ....
        rigidStars[i].oow = 0.5f + ((float)rRandom( 1.0f, 3.0f )/10.0f);
        // set the increasing (or already maximum) color
        rigidStars[i].r = colorGray;
        rigidStars[i].g = colorGray;
        rigidStars[i].b = colorGray;
        #ifndef __DEBUG_TEST__        
        grAADrawPoint( &rigidStars[i] );
        #endif
    }
}

/***********************************************************************************
 * FxStars              - Renders the star-theme
 ***********************************************************************************/
void FxStars (void)
{
        static GrVertex3D *xfVerts;             // transformed verts
        static GrVertex3D *prjVerts;            // projected verts
        GrVertex vtxA, vtxB;                     // line-section verts
        int i;                                  // i

        // allocate some verts
        xfVerts = (GrVertex3D *) malloc (numberOfStars*sizeof(GrVertex3D));
        prjVerts = (GrVertex3D *) malloc (numberOfStars*sizeof(GrVertex3D));

        // 3D Transformations and variables
        angle += rotangle;                      // adjust the angle
        if (accel > 1.0) accel -= 0.5;          // decrease acceleration
        distance += (speed/accel);              // increase the translation parameter 'distance'
        if ( angle >= 360.0f ) angle -= 360.0f; // let angle go round in rotations
        
        SetViewTransformMatrix_4D (Indentity_4D()); // set the viewtransformmatrix to identity
        if (rotation)
                ViewMatrixMult_4D (RotationZ_4D (angle));  // multily a rotation in Z
        // Adds a translation of -distance (move the stars closer!)
        ViewMatrixMult_4D (Translation_4D (0.0f, 0.0f, -distance)); 
        // Transform the 3d-stars into screen-space
        TransformVerticesToView_4D (xfVerts, stars3D, numberOfStars);
        // map the 3d-stars to the viewport
        ProjectVerticesToViewport (prjVerts, xfVerts, numberOfStars);
        // first paint the rigid stars
        FxRigidStars();

        // now paint the moving stars
        for( i = 0; i < numberOfStars; i++ )
        {
            // scale the projected vertices to screen space (they are in the [0,1] scale)
            vtxA.x = prjVerts[i].x * 640;
            vtxA.y = prjVerts[i].y * 480;
            if ((vtxA.x < 640) && (vtxA.x > -1) && (vtxA.y < 480) && (vtxA.y > -1))
            // if one is in screen space than we can paint it
            {
                vtxA.oow = 1.0f / prjVerts[i].w;        // depth value
                vtxA.a = 255.0f;                        // alpha value
                vtxA.r = colorGray;                     // red
                vtxA.g = colorGray;                     // green
                vtxA.b = colorGray;                     // blue
                memcpy (&vtxB, &vtxA, sizeof (GrVertex));       // make the tail-part vertex
                vtxB.x = (int) (vtxA.x + ((320 - vtxA.x)/tailLength));  // position it relative to the center
                vtxB.y = (int) (vtxA.y + ((240 - vtxA.y)/tailLength));  // idem dito
                vtxB.r = 0;                             // let red fade to 0
                vtxB.g = 0;                             // let green fade to 0
                vtxB.b = 0;                             // let blue fade to 0
                vtxB.oow = 1.0f/500.0f;                 // let the depth (artificially) fade so it gets a comet look
                #ifndef __DEBUG_TEST__        
                grAADrawLine ( &vtxB , &vtxA );
                #endif
             }
            else
            // if it is outside the screen-coordinates ... reset (randomize) the coordinate
            {
                stars3D[i].z = (((float)rRandom( 500, 1000 )) / 100.0f ) - 0.5f + distance;
            }
        }
        // de-allocate the variables
        free (xfVerts);
        free (prjVerts);
}

/***********************************************************************************
 * RenderFxScene        - The scene is animated and rendered ONCE to give Windows
 *                        back the message-processing
 ***********************************************************************************/
void RenderFxScene (void)
{
    // clear the buffers
    #ifndef __DEBUG_TEST__        
    grBufferClear(0,0,GR_WDEPTHVALUE_FARTHEST);
    guColorCombineFunction (GR_COLORCOMBINE_ITRGB);
    #endif    
    if (!FxIntro())
    // if FxIntro is FALSE then noise has faded to black and it will only return FALSE
    // from now on
    {
        // set color to maximum .. a little redundant but you can also play with it
        colorGray = 255.0f;
        // paint all stars
        FxStars();
    }
    #ifndef __DEBUG_TEST__        
    grBufferSwap( 1 );
    #endif
}

/***********************************************************************************
 * CloseFxDemo        - Here we de-allocate any demo data
 ***********************************************************************************/
void CloseFxDemo (void)
{
    // on closing clear the ever getting smaller clip-window with white
    #ifndef __DEBUG_TEST__        
    for (int i = 640; i > 0; i-=32)
    {        
        grClipWindow (0, 0, 640, 480);
        grBufferClear(0,0,GR_WDEPTHVALUE_FARTHEST);
        grClipWindow (640-i, 480-(i*480/640), i, (i*480/640));
        grBufferClear(0xffffffff,0,GR_WDEPTHVALUE_FARTHEST);
        grBufferSwap(1);
    }
    #endif

    // De-allocate the stars and noise
    free (noisePixels);
    free (stars3D);
    free (rigidStars);
}
