/* framecount.c - 
   Copyright (C) 1999 Tijs van Bakel.
   Tijs van Bakel <smoke@casema.net>, 
 
 This file is part of the bizarre99 linux invitation intro.

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <sys/time.h>
#include <unistd.h>
#include <math.h>

#include "framecount.h"

void init_frame_counter ( struct frame_count* fc, double r )
{
    set_frame_rate ( fc, r );
    reset_frame_counter ( fc );
}

void set_frame_rate ( struct frame_count* fc, double r )
{
    fc->rate = r;
}

void reset_frame_counter ( struct frame_count* fc )
{
    gettimeofday ( &fc->t0, NULL );
    fc->frame = 0;
}

double get_frame_counter ( struct frame_count* fc )
{
    return ( fc->frame );
}

double update_frame_counter ( struct frame_count* fc )
{
    struct timeval tv;
    double diff;

    gettimeofday ( &tv, NULL );
    diff = ( tv.tv_sec - fc->t0.tv_sec ) * 1000000.0 + ( tv.tv_usec - fc->t0.tv_usec );

    fc->frame = diff * fc->rate / 1000000.0;

    return ( get_frame_counter ( fc ) );
}

double wait_next_frame ( struct frame_count* fc )
{
    struct timeval tv;
    double diff;
    
#ifdef DONT_IDLE_WAIT_FOR_TIMER
    gettimeofday ( &tv, NULL );
    diff = ( tv.tv_sec - fc->t0.tv_sec ) * 1000000.0 + ( tv.tv_usec - fc->t0.tv_usec );

    fc->frame = diff * fc->rate / 1000000.0;

    return ( get_frame_counter ( fc ) );
#else
    double frame_now;

    do {
        gettimeofday ( &tv, NULL );
        diff = ( tv.tv_sec - fc->t0.tv_sec ) * 1000000.0 + ( tv.tv_usec - fc->t0.tv_usec );
        
        frame_now = diff * fc->rate / 1000000.0;
    } while ( rint(frame_now) < rint(fc->frame) );

    fc->frame = frame_now;

    return ( get_frame_counter ( fc ) );
#endif
}

#define FC_SPEEDUP_STEP 500000 /* should be < 1000000 */
void inc_frame_counter ( struct frame_count* fc )
{
    if ( fc->t0.tv_usec < FC_SPEEDUP_STEP )
    {
        fc->t0.tv_sec--;
        fc->t0.tv_usec = 1000000 - fc->t0.tv_usec;
    }
    else
        fc->t0.tv_usec -= FC_SPEEDUP_STEP;
}

