/* invtro.c - invitation intro for the bizarre 99 demoparty
   Copyright (C) 1999 Tijs van Bakel.
   Tijs van Bakel <smoke@casema.net>, 
 
 This file is part of the bizarre99 linux invitation intro.

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#define INVTRO_NAME "invtro: "

#include <stdlib.h>

#include "config.h"
#include "file.h"
#include "image.h"
#include "modplay.h"
#include "invtro.h"
#include "interface.h"
#include "title.h"
#include "credits.h"
#include "endscreen.h"
#include "graphics.h"
#include "screen.h"
#include "text.h"

#include <stdio.h>
#include <getopt.h>

#ifdef HAVE_SCHED_H
#include <sched.h>
#endif

struct music m;
int debug_flag = 0;

void debug_msg ( char* str )
{
    if ( debug_flag )
        fprintf ( stderr, "INVTRO-DEBUG: %s\n", str );
}

void splashScreen()
{
    printf("\nbizarre invitation intro "VERSION"\nCopyright (C) 1999 Tijs van Bakel, aka Smoke/CRAP <smoke@casema.net>\n\n");
    printf("This program is free software; you may redistribute it under the terms of\n");
    printf("the GNU General Public License. (See the file COPYING for more details).\n");
    printf("This program has absolutely no warranty.\n\n");
}

int realtime_process ()
{
#ifdef HAVE_SCHED_H
    struct sched_param sp;

    sp.sched_priority = sched_get_priority_min ( SCHED_RR );
    if ( sched_setscheduler ( 0, SCHED_RR, &sp ) )
    {
        fprintf ( stderr, INVTRO_NAME "could not enable realtime scheduling, continuing.\n" );
        return -1;
    };
#endif
    return 0;
}

int renice_process ()
{
    if ( nice ( -20 ) )
    {
        fprintf ( stderr, INVTRO_NAME "could not renice process by -20, continuing.\n" );
        return -1;
    };
    return 0;
}

void showHelp()
{
   printf("Usage: invtro-bizarre99 [OPTIONS]\n\n"
          "  -z, --zoom=FACTOR            zoom the screen by FACTOR\n"
          "  -m, --mute                   do not play sound\n"
          "  -v, --video=TARGET           force display target to TARGET\n"
          "                               TARGET is one of svgalib,x11,libggi\n\n"
          "  -r, --realtime               employ realtime scheduling (root only)\n"
          "  -n, --nice                   renice the process (root only)\n"
          "      --screw-vr               do not wait for the vertical retrace\n\n"
          "  -d, --debug                  show what is going on (for bugreports)\n\n"
          "      --help                   display this help and exit\n"
          "      --version                output version information and exit\n"
          "\n");
}

int main(int argc, char *argv[])
{
    char* data_path; /* path for loading data from */
    char* audio_so_path; /* path for loading shared object libraries from */
    char* video_so_path; /* path for loading shared object libraries from */
    
    struct graphics g;
    struct screen s;
    struct text t;
    
    int playSound = 1;

    int opt_index = 0;
    int argCount = 0;

    char *data_searchpaths[] = {
        PREFIX_INSTALL_PATH "/share/invtro-bizarre99/",
        "/usr/local/share/invtro-bizarre99/",
        "/usr/share/invtro-bizarre99/",
        NULL
    };
    
    char *video_so_searchpaths[] = {
        PREFIX_INSTALL_PATH "/lib/invtro-bizarre99/",
        "/usr/local/lib/invtro-bizarre99/",
        "/usr/lib/invtro-bizarre99/",
        NULL
    };
    
    char *audio_so_searchpaths[] = {
        PREFIX_INSTALL_PATH "/lib/invtro-bizarre99/",
        "/usr/local/lib/invtro-bizarre99/",
        "/usr/lib/invtro-bizarre99/",
        NULL
    };
    
    struct option long_opts[] = {
        { "help",      0, 0, 'H' },
        { "version",   0, 0, 'V' },
        { "mute",      0, 0, 'm' },
        { "zoom",      1, 0, 'z' },
        { "video",     1, 0, 'v' },
        { "screw-vr",  0, 0, 'S' },
        { "debug",     0, 0, 'd' },
        { "realtime",  0, 0, 'r' },
        { "nice",      0, 0, 'n' },
        { 0, 0, 0, 0 }
    };

    /* initialize the screen variables */
    s.preferred_target = SCREEN_TARGET_LIBGGI | SCREEN_TARGET_X11 | SCREEN_TARGET_SVGALIB;
    s.zoom_factor = 1;
    s.wait_retrace = 1; /* wait for the vertical retrace if possible */
    
    /* parse command line */
    while (1) {
        argCount = getopt_long(argc, argv, "HVmz:v:Sdrn", long_opts, &opt_index);
        if (argCount == -1) {
            break;
        }
  
        switch (argCount) {
                case 'H':
                    showHelp();
                    return 1;
                case 'V':
                    splashScreen();
                    return 1;
                case 'm':
                    playSound = 0;
                    break;
                case 'z':
                    s.zoom_factor = atoi(optarg);
                    if (s.zoom_factor < 1)
                    {
                        fprintf(stderr, INVTRO_NAME"the zoom factor should be at least 1\n");
                        return 1;
                        }
                    if (s.zoom_factor > 6)
                    {
                        fprintf(stderr,INVTRO_NAME"the zoom factor is extremely high. think about what will happen.\n");
                        return 1;
                    }
                    break;
                case 'S':
                    s.wait_retrace = 0;
                    break;
                case 'v':
                    s.preferred_target = 0;
                    if ( !strcmp ( optarg, "svgalib" ) )
                        s.preferred_target = SCREEN_TARGET_SVGALIB;
                    if ( !strcmp ( optarg, "x11" ) )
                        s.preferred_target = SCREEN_TARGET_X11;
                    if ( !strcmp ( optarg, "libggi" ) )
                        s.preferred_target = SCREEN_TARGET_LIBGGI;
                    break;
                case 'd':
                    debug_flag = 1;
                    break;
                case 'r':
                    realtime_process ();
                    break;
                case 'n':
                    renice_process ();
                    break;
                default:
                    return 1;
        }
    }

            /* general init stuff goes here */

    splashScreen();

    debug_msg ( "searching datapath\n" );
    data_path = get_data_path ( argv[0], data_searchpaths, "../Data/" );
    debug_msg ( data_path );
    audio_so_path = get_so_path ( argv[0], audio_so_searchpaths, "audio-plugins/.libs/" );
    debug_msg ( audio_so_path );
    video_so_path = get_so_path ( argv[0], video_so_searchpaths, "video-plugins/.libs/" );
    debug_msg ( video_so_path );

    debug_msg ( "registering modplayer driver" );
    if ( modplay_register ( &m, audio_so_path ) )
    {
        fprintf ( stderr, INVTRO_NAME"could not initialize a modplayer, continuing without sound\n" );
        playSound = 0;
    }
    
            /* initialization code */
    
    if (playSound)
    {
        debug_msg ( "initializing modplayer" );
        if ( modplay_init( &m ) ) {
            fprintf(stderr, INVTRO_NAME"Could not initialize music code\n");
            return 1;
        }
        debug_msg ( "loading music" );
        if ( modplay_load( &m, data_path, "music.xm" ) ) {
            fprintf(stderr, INVTRO_NAME"Bailing out because of libmikmod's doings\n");
            return 1;
        }
    }

    debug_msg ( "initializing screen driver" );
    if ( screen_init ( &s, video_so_path ) ) {
        fprintf(stderr, INVTRO_NAME"Could not initialize screen code\n");
        return 1;
    }

    debug_msg ( "initializing graphics" );
    if ( initGraphics ( &g, &s ) ) {
        fprintf(stderr, INVTRO_NAME"Could not initialize graphics code\n");
        return 1;
    }

    debug_msg ( "loading graphics" );
    if ( loadGraphics( data_path, &g ) ) {
        fprintf(stderr, INVTRO_NAME"Could not read graphics files\n");
        return 1;
    }
    
    debug_msg ( "starting modplayer" );
    if ( loadWriterText( data_path, &t ) ) {
        fprintf(stderr, INVTRO_NAME "Could not read text file\n");
        return 1;
    }

           /* here we go */

    if (playSound)
    {
        debug_msg ( "starting modplayer" );
        modplay_start( &m );
    }

    debug_msg ( "setting videomode" );

    if ( video_set_mode ( &s, SCR_WIDTH, SCR_HEIGHT, SCR_BPP ) )
    {
        fprintf ( stderr, "\n" INVTRO_NAME "please try forcing another video target with the -v switch.\n" );
        return 1;
    }

    debug_msg ( "starting credits part" );
    doCredits( &s, &g, &m );
    debug_msg ( "starting title part" );
    doTitleScreen( &s, &g, &m );
    debug_msg ( "starting interface part" );
    doInterface( &s, &g, &m, &t );
    debug_msg ( "starting endscreen part" );
    doEndScreen( &s, &g, &m );

            /* over so soon ? */

    debug_msg ( "ending" );
    if (playSound)
        modplay_stop( &m ); 

    video_done ( &s );
    
    if (playSound)
        modplay_done( &m );
    
    doneGraphics( &g );

    return 0;
}

