/* x11.c - x11 window system support for the invtro 
   Copyright (C) 1999 Tijs van Bakel.
   Tijs van Bakel <smoke@casema.net>, 
 
 This file is part of the bizarre99 linux invitation intro.

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#define PLUGIN_NAME "invtro, x11-plugin: "

#include <stdlib.h>
#include <unistd.h>

#include "screen.h"
#include "types.h"

#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>

#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/extensions/XShm.h>
#include <X11/keysym.h>

#define X11_8_BPP    0
#define X11_16_BPP   1
#define X11_24_BPP   2

int X11_bpp;
uint16 palette_lookup_16 [ 256 ];
uint32 palette_lookup_24 [ 256 ];

Display *display;
int screen;
Screen *scrptr;
Visual *visual;
Window window, root;

int depth;
Colormap colormap;
XColor xcolors[256];

XSetWindowAttributes attributes;
unsigned long attribute_mask;
XShmSegmentInfo shminfo;
GC gc;
XImage *ximage;

uint16* convert_buf_16;
uint32* convert_buf_24;

int init ( struct screen* s )
{
    int i;
    
    if ( ( display = XOpenDisplay ( NULL ) ) == NULL )
    {
        fprintf ( stderr, PLUGIN_NAME "could not open display %s\n", XDisplayName(NULL));
        return -1;
    }
    
    screen = DefaultScreen (display);
    scrptr = DefaultScreenOfDisplay (display);
    visual = DefaultVisual (display, screen);
    root = DefaultRootWindow (display);
    depth = DefaultDepth (display, screen);

    s->image = malloc ( sizeof (struct image) );
    if (!s->image) {
        fprintf(stderr,PLUGIN_NAME "could not get memory for screen\n");
        return 1;
    }
    
    s->image->width = SCR_WIDTH;
    s->image->height = SCR_HEIGHT;

    if ( ( visual->class == PseudoColor ) && ( depth == 8 ) )
    {
        X11_bpp = X11_8_BPP;
    }
    else if ( ( visual->class == TrueColor ) && ( depth == 16 ) )
    {
        X11_bpp = X11_16_BPP;
    }
    else if ( ( visual->class == TrueColor ) && ( depth == 24 ) )
    {
        X11_bpp = X11_24_BPP;
    }
    else
    {
        fprintf ( stderr, PLUGIN_NAME "x11 bitdepth not supported\n");
        return -1;
    }

    if ( X11_bpp == X11_8_BPP )
    {
        colormap = XCreateColormap (display, root, visual, AllocAll);
        for ( i = 0; i < 256; i++ )
        {
            xcolors[i].flags = DoRed|DoGreen|DoBlue;
            xcolors[i].pixel = i;
            xcolors[i].pad = 0;
        }
    }
    else
    {
        colormap = DefaultColormap (display, screen);
    }

    s->wait_retrace = 0;
    return 0;
}

int set_mode ( struct screen* s, uint32 width, uint32 height, uint32 bpp )
{
    width *= s->zoom_factor;
    height *= s->zoom_factor;
    
    attribute_mask = CWEventMask;
    
    if ( X11_bpp == X11_8_BPP )
        attribute_mask |= CWColormap;
    
    attributes.colormap = colormap;
    attributes.event_mask |= ExposureMask | ButtonPressMask | KeyPressMask;
    
    window = XCreateWindow (display, root, 0, 0, width, height, 1,
        DefaultDepthOfScreen (scrptr), InputOutput, visual,
        attribute_mask, &attributes);

    if (!window) {
        fprintf ( stderr, PLUGIN_NAME "could not open a window in x11\n");
        return -1;
    }

    gc = XCreateGC ( display, window, 0, NULL);
    ximage = XShmCreateImage ( display, visual, depth, ZPixmap,
                               NULL, &shminfo, width, height );

    if (!ximage) {
        fprintf (stderr, "can't create image\n");
        return -1;
    }
    shminfo.shmid = shmget (IPC_PRIVATE, ximage->bytes_per_line *
        (ximage->height + 1), IPC_CREAT | 0600);

    if (shminfo.shmid == -1) {
        fprintf (stderr, "can't allocate X shared memory\n");
        return -1;
    }
    shminfo.shmaddr = ximage->data = shmat (shminfo.shmid, 0, 0);
    shminfo.readOnly = 0;
    XShmAttach (display, &shminfo);

    switch ( X11_bpp ) {
            case X11_8_BPP:
                break;
            case X11_16_BPP:
                convert_buf_16 = malloc ( width * height * 2 );
                if ( !convert_buf_16 )
                {
                    fprintf ( stderr, PLUGIN_NAME "could not get memory for zoom buffer\n" );
                    return -1;
                }
                break;
            case X11_24_BPP:
                convert_buf_24 = malloc ( width * height * 4 );
                if ( !convert_buf_24 )
                {
                    fprintf ( stderr, PLUGIN_NAME "could not get memory for zoom buffer\n" );
                    return -1;
                }
                break;
    }
                
    if ( ( X11_bpp != X11_8_BPP ) || ( s->zoom_factor > 1 ) )
    {
                /* we use a dummy to paint the demo in,
                   then later on we convert it to fit into the
                   image->data to go on the screen */
        s->image->buffer = (uint8*) malloc( SCR_WIDTH * SCR_HEIGHT );
        if (!s->image->buffer) {
            fprintf(stderr,PLUGIN_NAME "could not get memory for screen\n");
            return 1;
        }
    }
    else
    {
                /* in the unzoomed 8bpp mode we can use the image->data
                   directly */
        s->image->buffer = ximage->data;
    }

    XMapWindow ( display, window );
    XFlush ( display );
    XSync ( display, False );

    return 0;
}

void done ( struct screen* s )
{
    XSync ( display, False );
    XShmDetach ( display, &shminfo );

    XCloseDisplay (display);
    shmctl (shminfo.shmid, IPC_RMID, NULL);
    shmdt (shminfo.shmaddr);
}

/* cut and paste code follows. these 3 look alike, but it's a tiny bit faster
   then having a switch(bitdepth) statement around every pixelcopy */

void zoom_screen_8(struct screen *s, uint8* buffer)
{
    uint8* dest, * src;
    uint8* destptr, * line_src_ptr;
    uint32 x,y;
    uint8 pixel;
    uint32 width;
    int i;
    int zoom = s->zoom_factor;

    width = s->image->width * s->zoom_factor;
    src = buffer;
    destptr = (uint8*) ximage->data;
    
    for (y=0; y < s->image->height; y++)
    {
        dest = destptr;
        line_src_ptr = destptr;

                /* zoom one line for horizontal zooming */
        for ( x = 0; x < s->image->width; x++ )
        {
            pixel = *src++;
            for (i = zoom; i--; )
                *dest++ = pixel;
        }
                /* copy the line for vertical zooming */
        for (i = zoom-1; i--; )
        {
            destptr += s->image->width * zoom;
            memcpy ( destptr, line_src_ptr, s->image->width * zoom * 1 );
        }

        destptr += s->image->width * zoom;
    }

    XShmPutImage (display, window, gc, ximage, 0, 0, 0, 0, ximage->width, ximage->height, 0);
}

void zoom_screen_16(struct screen *s, uint16* converted_buffer)
{
    uint16* dest, * src;
    uint16* destptr, * line_src_ptr;
    uint32 x,y;
    uint16 pixel;
    uint32 width;
    int i;
    int zoom = s->zoom_factor;

    width = s->image->width * s->zoom_factor;
    src = converted_buffer;
    destptr = (uint16*) ximage->data;
    
    for (y=0; y < s->image->height; y++)
    {
        dest = destptr;
        line_src_ptr = destptr;

                /* zoom one line for horizontal zooming */
        for ( x = 0; x < s->image->width; x++ )
        {
            pixel = *src++;
            for (i = zoom; i--; )
                *dest++ = pixel;
        }
                /* copy the line for vertical zooming */
        for (i = zoom-1; i--; )
        {
            destptr += s->image->width * zoom;
            memcpy ( destptr, line_src_ptr, s->image->width * zoom * 2 );
        }

        destptr += s->image->width * zoom;
    }

    XShmPutImage (display, window, gc, ximage, 0, 0, 0, 0, ximage->width, ximage->height, 0);
}

void zoom_screen_24(struct screen *s, uint32* converted_buffer)
{
    uint32* dest, * src;
    uint32* destptr, * line_src_ptr;
    uint32 x,y;
    uint32 pixel;
    uint32 width;
    int i;
    int zoom = s->zoom_factor;

    width = s->image->width * s->zoom_factor;
    src = converted_buffer;
    destptr = (uint32*) ximage->data;
    
    for (y=0; y < s->image->height; y++)
    {
        dest = destptr;
        line_src_ptr = destptr;

                /* zoom one line for horizontal zooming */
        for ( x = 0; x < s->image->width; x++ )
        {
            pixel = *src++;
            for (i = zoom; i--; )
                *dest++ = pixel;
        }
                /* copy the line for vertical zooming */
        for (i = zoom-1; i--; )
        {
            destptr += s->image->width * zoom;
            memcpy ( destptr, line_src_ptr, s->image->width * zoom * 4 );
        }

        destptr += s->image->width * zoom;
    }

    XShmPutImage (display, window, gc, ximage, 0, 0, 0, 0, ximage->width, ximage->height, 0);
}

        /* LAME ASS HELL conversion routines. */
void convert_buffer_8_to_24 ( uint8* in, uint8* out, uint32 length )
{
    int i;
    uint32 c;

    for (i = 0; i < length; i++)
    {
        c = palette_lookup_24 [ in[i] ];
        out[i*4+2] = c >> 16;
        out[i*4+1] = (c >> 8) & 255;
        out[i*4] = c & 255;
    }
}

void convert_buffer_8_to_16 ( uint8* in, uint8* out, uint32 length )
{
    int i;
    uint16 c;
    
    for (i = 0; i < length; i++)
    {
        c = palette_lookup_16 [ in[i] ];
        out[i*2+1] = c >> 8;
        out[i*2] = c & 255;
    }
}

void update_buffer ( struct screen* s )
{
    if (! s->directBuffered ) {

        if ( s->zoom_factor != 1 )
        {
            switch ( X11_bpp ) {
                    case X11_8_BPP:
                        zoom_screen_8( s, s->image->buffer );
                        break;
                    case X11_16_BPP:
                        convert_buffer_8_to_16 ( s->image->buffer, (uint8*) convert_buf_16, SCR_WIDTH * SCR_HEIGHT );
                        zoom_screen_16 ( s, convert_buf_16 );
                        break;
                    case X11_24_BPP:
                        convert_buffer_8_to_24 ( s->image->buffer, (uint8*) convert_buf_24, SCR_WIDTH * SCR_HEIGHT );
                        zoom_screen_24 ( s, convert_buf_24 );
                        break;
            }
        }
        else 
        {
            switch ( X11_bpp ) {
                    case X11_8_BPP:
                        XShmPutImage (display, window, gc, ximage, 0, 0, 0, 0, s->image->width, s->image->height, 0);
                        break;
                    case X11_16_BPP:
                        convert_buffer_8_to_16 ( s->image->buffer, (uint8*) ximage->data, SCR_WIDTH * SCR_HEIGHT );
                        XShmPutImage (display, window, gc, ximage, 0, 0, 0, 0, s->image->width, s->image->height, 0);
                        break;
                    case X11_24_BPP:
                        convert_buffer_8_to_24 ( s->image->buffer, ximage->data, SCR_WIDTH * SCR_HEIGHT );
                        XShmPutImage (display, window, gc, ximage, 0, 0, 0, 0, s->image->width, s->image->height, 0);
                        break;
            }
        }
        
        XSync ( display, False );

    }
}

void update_palette ( struct screen* s, uint32 begincolor, uint32 length )
{
    int i;
    uint16 r,g,b;

    if ( s->paletteDirty )
    {
        switch ( X11_bpp ) {
            case X11_8_BPP:
                for ( i = 0; i < 256; i++ )
                {
                    if ( ( i > begincolor ) && ( i < begincolor+length ) )
                    {
                        xcolors[i].red = s->colormap[i].r;
                        xcolors[i].green = s->colormap[i].g;
                        xcolors[i].blue = s->colormap[i].b;
                    }
                }
                XStoreColors ( display, colormap, xcolors, 256 );
                break;
            case X11_16_BPP:
                for ( i = begincolor; i < begincolor+length; i++ )
                {
                    r = ( s->colormap[i].r >> 11 );
                    g = ( s->colormap[i].g >> 10 );
                    b = ( s->colormap[i].b >> 11 );

                    palette_lookup_16 [ i ] = ( ( r << 11 ) + ( g << 5 ) + b );
                }
                break;
            case X11_24_BPP:
                for ( i = begincolor; i < begincolor+length; i++ )
                {
                    r = ( s->colormap[i].r >> 8 );
                    g = ( s->colormap[i].g >> 8 );
                    b = ( s->colormap[i].b >> 8 );

                    palette_lookup_24 [ i ] = ( ( r << 16 ) + ( g << 8 ) + b );
                }
                break;
        }
        s->paletteDirty = 0;
    }
}

int poll_input ( struct screen* s )
{
    static XEvent event;
    
    while (XEventsQueued (display, QueuedAfterReading))
    {
        XNextEvent (display, &event);
        switch (event.type) {
                case KeyPress:
                    switch ( XKeycodeToKeysym(display,event.xkey.keycode,0) ) {
                            case XK_Left:
                                return INPUT_LEFT;
                            case XK_Right:
                                return INPUT_RIGHT;
                            case XK_Up:
                                return INPUT_UP;
                            case XK_Down:
                                return INPUT_DOWN;
                            case XK_Escape:
                                return INPUT_QUIT;
                            case XK_space:
                                return INPUT_CONTINUE;
                            case XK_Delete:
                                return INPUT_BACK;
                    }
        }
    }
    return INPUT_NOKEY;
}
