#include "gem.h"
#include "gemmath.h"
#include "gemutil.h"
#include <math.h>


GEM_OBJECT* FindObject( GEM_SCENE* scene, TCHAR* name )
{
	GEM_OBJECT*			obj = scene->objectList;

	if( !obj )
		return NULL;

	for( ; obj ; obj=obj->next )
	{
		if( !lstrcmpi( name, obj->name ) )
			return obj;
	}

	return NULL;
}

void AddKey(GEM_KEY* key, GEM_TRACK* track)
{
	GEM_KEY*			tmp = track->keyList;

	if( !tmp )
	{
		key->next = NULL;
		key->prev = NULL;
		track->keyList = key;
	}
	else
	{
		for( ; tmp->next ; tmp=tmp->next );

		tmp->next = key;
		key->prev = tmp;
		key->next = NULL;
	}
}

void AddKeyframer(GEM_KEYFRAMER* keyframer, GEM_SCENE* scene)
{
	GEM_KEYFRAMER*		tmp = scene->keyframerList;

	if( !tmp )
	{
		keyframer->next = NULL;
		keyframer->prev = NULL;
		scene->keyframerList = keyframer;
	}
	else
	{
		for( ; tmp->next ; tmp=tmp->next );

		tmp->next = keyframer;
		keyframer->prev = tmp;
		keyframer->next = NULL;
	}
}


void AddObject(GEM_OBJECT* object, GEM_SCENE* scene)
{
	GEM_OBJECT*			tmp = scene->objectList;

	if( !tmp )
	{
		object->next = NULL;
		object->prev = NULL;
		scene->objectList = object;
	}
	else
	{
		for( ; tmp->next ; tmp=tmp->next );

		tmp->next = object;
		object->prev = tmp;
		object->next = NULL;
	}
}

D3DVALUE LensToFOV( D3DVALUE lens)
{
	DWORD  			i;
	D3DVALUE 		fov;

	struct 
	{
		D3DVALUE	lens; 
		D3DVALUE	fov;
	} lens_table[] = {
						{15.0f,  115.0f}, 
						{20.0f,	 94.28571f}, 
						{24.0f,	 84.0f}, 
						{28.0f,  76.36364f},
						{35.0f,  63.0f},  
						{50.0f,	 48.0f},     
						{85.0f,	 28.0f}, 
						{135.0f, 18.0f},
						{200.0f, 12.0f}
					};

	for( i = 0 ; i < 9 ; i++ )
		if (lens == lens_table[i].lens)
		{			
			fov = lens_table[i].fov;
			break;
		}
		else
			fov = ( 15.0f / lens ) * 160.0f;

  return fov;
}

void CalculateNormals(GEM_MESH* mesh)
{
	GEM_VECTOR		norm;
	GEM_VECTOR		vec1;
	GEM_VECTOR		vec2;
	D3DVERTEX		v1, v2, v3;

	for( DWORD i = 0 ; i<mesh->vertexNbr ; i++ )
	{
		norm = GEM_VECTOR(0,0,0);
		for( DWORD j =0 ; j<mesh->faceNbr ; j++ )
			if( mesh->faces[j].indices[0]==i||mesh->faces[j].indices[1]==i||mesh->faces[j].indices[2]==i )
			{
				v1 = mesh->vertices[mesh->faces[j].indices[0]];
				v2 = mesh->vertices[mesh->faces[j].indices[1]];
				v3 = mesh->vertices[mesh->faces[j].indices[2]];

				vec1.x = v2.x - v1.x;
				vec1.y = v2.y - v1.y;
				vec1.z = v2.z - v1.z;

				vec2.x = v3.x - v1.x;
				vec2.y = v3.y - v1.y;
				vec2.z = v3.z - v1.z;

				norm = norm + Cross(vec1,vec2);
			}
		
		norm = Normalize( norm );
		mesh->vertices[i].nx = norm.x;
		mesh->vertices[i].ny = norm.y;
		mesh->vertices[i].nz = norm.z;
	}

}

int InitAllLights( GEM_SCENE *scene )
{
	if( !scene )
		return gem_error_null_pointer;

	if( !scene->pd3dDevice )
		return gem_error_scene_not_initialized;;
		

	GEM_OBJECT*			tmp = scene->objectList;
	GEM_LIGHT*			light;
	LPDIRECT3DVIEWPORT3	pd3dViewport;
	LPDIRECT3D3			pD3D;
	
	if( scene->pd3dDevice->GetCurrentViewport( &pd3dViewport )!=D3D_OK )
		return gem_error_no_viewport;

	if( scene->pd3dDevice->GetDirect3D( &pD3D )!=D3D_OK )
		return gem_error_invalid_device;

	for( ; tmp ; tmp=tmp->next )
	{
		if( tmp->type == LIGHT )
		{
			light = (GEM_LIGHT*)tmp->object;
			pD3D->CreateLight( &light->pd3dLight, NULL );

			D3DLIGHT2			lightInfo;
			ZeroMemory( &lightInfo, sizeof(D3DLIGHT2) );
			lightInfo.dwSize		= sizeof(D3DLIGHT2);
			lightInfo.dltType		= D3DLIGHT_POINT;
			lightInfo.dcvColor.r	= 1.0f;
			lightInfo.dcvColor.g	= 1.0f;
			lightInfo.dcvColor.b	= 1.0f;
			lightInfo.dvPosition.x	= 0.0f;
			lightInfo.dvPosition.y	= 0.0f;
			lightInfo.dvPosition.z	= 0.0f;	
			lightInfo.dvAttenuation0= 0.9f;     
			lightInfo.dvAttenuation1= 0.5f;     
			lightInfo.dvAttenuation2= 0.2f;     
			lightInfo.dvRange		= 1000.0f;
			lightInfo.dwFlags		= D3DLIGHT_ACTIVE;

			light->pd3dLight->SetLight( (LPD3DLIGHT)&lightInfo );
			pd3dViewport->AddLight( light->pd3dLight );
		}
	}

	pd3dViewport->Release();
	pD3D->Release();

	return gem_error_ok;
}
			

void PrepareKeyframer( GEM_SCENE *s )
{
	GEM_KEYFRAMER*		tmp=s->keyframerList;
	GEM_OBJECT*			obj;
	GEM_MESH*			mesh;
	
	while (tmp)
	{
		obj = FindObject( s, tmp->objName );
		tmp->obj = obj->object;
		if (tmp->type==MESHTRACK)
		{
			mesh = (GEM_MESH *)obj->object;
			for( DWORD i=0 ; i<mesh->vertexNbr ; i++ )
			{
				mesh->vertices[i].x=mesh->vertices[i].x-tmp->pivot.x;
				mesh->vertices[i].y=mesh->vertices[i].y-tmp->pivot.y;
				mesh->vertices[i].z=mesh->vertices[i].z-tmp->pivot.z;
			}
		}

		tmp=tmp->next;
	}
}



/*
 *  znajduje rodzica sciezki track w scenie s, zwraca null jezeli nie istnieje
 *  wykozystuje identyfikatory hierarchi
 */
GEM_KEYFRAMER *FindParent(GEM_SCENE *s, GEM_KEYFRAMER *track)
{
	GEM_KEYFRAMER*		tmp = s->keyframerList;

	while (tmp)
	{
		if( tmp->hierarchy==track->link )		
			return tmp;
		
		tmp=tmp->next;
	}

	return NULL;
}


/*
 *  wiaze hierarchie sciezek w scenie s
 */
void LinkHierarchy(GEM_SCENE *s)
{
	GEM_KEYFRAMER*		tmp = s->keyframerList;

	while (tmp)
	{
		if (tmp->link==0xFFFF)
			tmp->parent=NULL;
		else
			tmp->parent=FindParent(s,tmp);

		tmp=tmp->next;
	}

}

GEM_MATRIX PrepareMatrix(GEM_VECTOR& trans, GEM_VECTOR& scale, GEM_QUATERNION& rot)
{
	return SetScaleMatrix(scale)*FromQuat(rot)*SetTranslationMatrix(trans);
}

int TransformObjects(GEM_SCENE* scene, DWORD frame)
{
	if( !scene )
		return gem_error_null_pointer;

	GEM_KEYFRAMER*		 tmp = scene->keyframerList;
	GEM_MESH_TRACK*		 meshtrack;
	GEM_CAMERA_TRACK*	 camtrack;
	GEM_CAMERATRG_TRACK* camtrgtrack;
	GEM_LIGHT_TRACK*	 lighttrack;
	GEM_LIGHT*			 light;
	GEM_MESH*			 mesh;
	GEM_CAMERA*			 camera;

	for( ; tmp ; tmp=tmp->next )
	{
		switch( tmp->type )
		{
			case MESHTRACK:
				meshtrack = (GEM_MESH_TRACK *)tmp->track;
				mesh = (GEM_MESH *)tmp->obj;
				
				mesh->rot = QuatSpline  ( meshtrack->rotation, frame );
				mesh->trn = VectorSpline( meshtrack->position, frame );
				mesh->scl = VectorSpline( meshtrack->scale   , frame );

				mesh->mtxTransform = PrepareMatrix( mesh->trn, mesh->scl, mesh->rot );
			break;

			case LIGHTTRACK:
				lighttrack = (GEM_LIGHT_TRACK *)tmp->track;
				light = (GEM_LIGHT *)tmp->obj;

				light->position = VectorSpline( lighttrack->position, frame );
			break;

			case CAMERATRACK:
				camtrack = (GEM_CAMERA_TRACK *)tmp->track;
				camera = (GEM_CAMERA *)tmp->obj;

				camera->position = VectorSpline( camtrack->position, frame );
				camera->roll = FloatSpline( camtrack->roll, frame );
				camera->fov = FloatSpline( camtrack->fov, frame );
			break;

			case CAMERATRGTRACK:
				camtrgtrack = (GEM_CAMERATRG_TRACK *)tmp->track;
				camera = (GEM_CAMERA *)tmp->obj;

				camera->target = VectorSpline( camtrgtrack->position, frame );
			break;
		}
	}

	return gem_error_ok;
}

int TransformHierarchy(GEM_SCENE* scene)
{
	if( !scene )
		return gem_error_null_pointer;

	GEM_KEYFRAMER*		tmp = scene->keyframerList;
	GEM_MESH*			mesh;
	GEM_LIGHT*			light;
	GEM_CAMERA*			camera;
	GEM_MATRIX			parentmtx;

	for( ; tmp ; tmp=tmp->next )
	{
		if( tmp->parent && tmp->parent->type == MESHTRACK )
		{
			mesh = (GEM_MESH *)tmp->parent->obj;
			parentmtx = mesh->mtxTransform;

			switch( tmp->type )
			{
				case MESHTRACK:
					mesh = (GEM_MESH *)tmp->obj;
					mesh->mtxTransform = mesh->mtxTransform*parentmtx;
				break;

				case LIGHTTRACK:
					light = (GEM_LIGHT *)tmp->obj;
					light->position = light->position*parentmtx;
				break;

				case CAMERATRACK:
					camera = (GEM_CAMERA *)tmp->obj;
					camera->position = camera->position*parentmtx;
				break;

				case CAMERATRGTRACK:
					camera = (GEM_CAMERA *)tmp->obj;
					camera->target = camera->target*parentmtx;
				break;
			}
		}
	}

	return gem_error_ok;
}


int UpdateCamera(GEM_CAMERA* camera)
{
	if( !camera )
		return gem_error_null_pointer;

	camera->mtxProjection = SetProjectionMatrix(camera->fov, camera->nearZ, camera->farZ, camera->aspect);
	camera->mtxView = SetCameraMatrix( camera->position, camera->target, camera->roll );

	return gem_error_ok;
}


int TransformScene(GEM_SCENE* scene, DWORD frame)
{
	
	if( TransformObjects( scene, frame )!=gem_error_ok )
		return gem_error_null_pointer;

	if( TransformHierarchy( scene )!=gem_error_ok )
		return gem_error_null_pointer;

	if( UpdateCamera( scene->camera )!=gem_error_ok )
		return gem_error_no_active_camera;

	D3DMATRIX		d3dmtxView = GetD3DMATRIX( scene->camera->mtxView );
	D3DMATRIX		d3dmtxProjection = GetD3DMATRIX( scene->camera->mtxProjection ); 

	if( !scene->pd3dDevice )
		return gem_error_scene_not_initialized;

	scene->pd3dDevice->SetTransform( D3DTRANSFORMSTATE_VIEW, &d3dmtxView );
	scene->pd3dDevice->SetTransform( D3DTRANSFORMSTATE_PROJECTION, &d3dmtxProjection );

	GEM_OBJECT*			tmp = scene->objectList;
	GEM_LIGHT*			light;
	
	for( ; tmp ; tmp=tmp->next )
	{
		if( tmp->type == LIGHT )
		{
			light = (GEM_LIGHT *)tmp->object;

			D3DLIGHT2	lightInfo;
			ZeroMemory( &lightInfo, sizeof(D3DLIGHT2) );
			lightInfo.dwSize = sizeof(D3DLIGHT2);
			
			light->pd3dLight->GetLight( (LPD3DLIGHT)&lightInfo );
			lightInfo.dvPosition = GetD3DVECTOR(light->position);
			light->pd3dLight->SetLight( (LPD3DLIGHT)&lightInfo );
			
		}
	}

	return gem_error_ok;
}

GEM_MATRIX SetLitMatrix(GEM_VECTOR& CamPos, GEM_VECTOR& CamTrg, D3DVALUE roll)
{
	GEM_VECTOR		dir = CamTrg-CamPos;
	D3DVALUE		focus = Length( dir );

	
	D3DVALUE		alpha = -atan2(dir.x,dir.z);
	D3DVALUE		beta  = asin(dir.y/focus);
	D3DVALUE		gamma = -roll*3.14159f/180.0f;

	D3DVALUE		sina = (D3DVALUE)sin(alpha); 
	D3DVALUE		cosa = (D3DVALUE)cos(alpha);
	D3DVALUE		sinb = (D3DVALUE)sin(beta);  
	D3DVALUE		cosb = (D3DVALUE)cos(beta);
	D3DVALUE		sing = (D3DVALUE)sin(gamma); 
	D3DVALUE		cosg = (D3DVALUE)cos(gamma);

	GEM_MATRIX		mtx = IdentMatrix();

	mtx(1,1) = sina*sinb*sing + cosa*cosg;
	mtx(2,1) = cosb*sing;
	mtx(3,1) = sina*cosg - cosa*sinb*sing;

	mtx(1,2) = sina*sinb*cosg - cosa*sing;
	mtx(2,2) = cosb*cosg;
    mtx(3,2) = -cosa*sinb*cosg - sina*sing;

	mtx(1,3) = -sina*cosb;
    mtx(2,3) = sinb;
    mtx(3,3) = cosa*cosb;

	return Invert( mtx );
}


void RenderLight(GEM_SCENE* scene, GEM_LIGHT* light)
{
	D3DVERTEX	vertex[4];
	D3DMATRIX	d3dmtx = GetD3DMATRIX( SetLitMatrix(scene->camera->position,scene->camera->target, scene->camera->roll)*
									   SetTranslationMatrix( light->position ) );	
	

	D3DVECTOR vNorm = D3DVECTOR( 0.0f, 0.0f, 1.0f );
    vertex[0] = D3DVERTEX( D3DVECTOR( -scene->lsize, -scene->lsize, 0.0f ), vNorm, 0.0f, 1.0f );
    vertex[1] = D3DVERTEX( D3DVECTOR( -scene->lsize, +scene->lsize, 0.0f ), vNorm, 0.0f, 0.0f );
    vertex[2] = D3DVERTEX( D3DVECTOR( +scene->lsize, -scene->lsize, 0.0f ), vNorm, 1.0f, 1.0f );
    vertex[3] = D3DVERTEX( D3DVECTOR( +scene->lsize, +scene->lsize, 0.0f ), vNorm, 1.0f, 0.0f );

	scene->pd3dDevice->SetRenderState( D3DRENDERSTATE_SRCBLEND, D3DBLEND_ONE);
    scene->pd3dDevice->SetRenderState( D3DRENDERSTATE_DESTBLEND, D3DBLEND_ONE );
	scene->pd3dDevice->SetRenderState( D3DRENDERSTATE_ALPHABLENDENABLE, TRUE );
	scene->pd3dDevice->SetRenderState( D3DRENDERSTATE_ZWRITEENABLE , FALSE );
	
	scene->pd3dDevice->SetTexture(0, scene->ptexLightText);

	scene->pd3dDevice->SetTextureStageState( 0, D3DTSS_COLORARG1, D3DTA_TEXTURE );
	scene->pd3dDevice->SetTextureStageState( 0, D3DTSS_COLORARG2, D3DTA_DIFFUSE );
	scene->pd3dDevice->SetTextureStageState( 0, D3DTSS_COLOROP,   D3DTOP_SELECTARG1);
		
	scene->pd3dDevice->SetTransform( D3DTRANSFORMSTATE_WORLD, &d3dmtx );

	scene->pd3dDevice->BeginScene();	
	scene->pd3dDevice->DrawPrimitive( D3DPT_TRIANGLESTRIP, D3DFVF_VERTEX, vertex, 4, D3DDP_DONOTLIGHT );
	scene->pd3dDevice->EndScene();
	scene->pd3dDevice->SetRenderState( D3DRENDERSTATE_ALPHABLENDENABLE, FALSE );
	scene->pd3dDevice->SetRenderState( D3DRENDERSTATE_ZWRITEENABLE , TRUE );
}


int RenderMesh(GEM_SCENE* scene, GEM_MESH* mesh)
{
	D3DMATRIX		d3dmtx = GetD3DMATRIX( mesh->mtxTransform );
	HRESULT			hr;
	
	hr = scene->pd3dDevice->SetTransform( D3DTRANSFORMSTATE_WORLD, &d3dmtx );
	
	hr = scene->pd3dDevice->BeginScene();

	for (GEM_ELEMENT* tmp = mesh->elemList ; tmp ; tmp = tmp->next )
	{
		if( tmp->mat )
		{
			if( tmp->mat->texture1 )
			{
				LPDIRECT3DTEXTURE2 lptex = GetTexture( tmp->mat->texture1->filename );
				hr = scene->pd3dDevice->SetTexture( 0, lptex );
				hr = scene->pd3dDevice->SetTextureStageState( 0, D3DTSS_COLORARG1, D3DTA_TEXTURE );
				hr = scene->pd3dDevice->SetTextureStageState( 0, D3DTSS_COLOROP,   D3DTOP_MODULATE );
				hr = scene->pd3dDevice->SetTextureStageState( 0, D3DTSS_COLORARG2, D3DTA_DIFFUSE );
			}
			else						
				hr = scene->pd3dDevice->SetTextureStageState( 0, D3DTSS_COLOROP, D3DTOP_DISABLE );
							
		}
		else		
			hr = scene->pd3dDevice->SetTextureStageState( 0, D3DTSS_COLOROP, D3DTOP_DISABLE );				
			
		

		hr = scene->pd3dDevice->DrawIndexedPrimitive(D3DPT_TRIANGLELIST, D3DFVF_VERTEX,
										        mesh->vertices, mesh->vertexNbr, 
												tmp->idx, tmp->number, 0 );		

		if( FAILED( hr ) )
		{
			hr = scene->pd3dDevice->EndScene();
			return 1;
		}

		
	}

/*	mesh->scl.x += 0.1f;
	mesh->scl.y += 0.1f;
	mesh->scl.z += 0.1f;

	d3dmtx = GetD3DMATRIX( PrepareMatrix( mesh->trn, mesh->scl, mesh->rot ) );
	scene->pd3dDevice->SetTransform( D3DTRANSFORMSTATE_WORLD, &d3dmtx );
	scene->pd3dDevice->SetTexture( 0, NULL );
	scene->pd3dDevice->DrawIndexedPrimitive(D3DPT_LINELIST, D3DFVF_VERTEX,
										    mesh->vertices, mesh->vertexNbr, 
											mesh->linelist, mesh->faceNbr*4, D3DDP_DONOTLIGHT );*/

	hr = scene->pd3dDevice->EndScene();

	return gem_error_ok;
}

/*void RenderLineMesh(GEM_SCENE* scene, GEM_MESH* mesh)
{
	mesh->scl.x += 0.1f;
	mesh->scl.y += 0.1f;
	mesh->scl.z += 0.1f;
	
	d3dmtx = GetD3DMATRIX( PrepareMatrix( mesh->trn, mesh->scl, mesh->rot ) );
	scene->pd3dDevice->SetTransform( D3DTRANSFORMSTATE_WORLD, &d3dmtx );
	scene->pd3dDevice->SetTexture( 0, NULL );
	scene->pd3dDevice->DrawIndexedPrimitive(D3DPT_LINELIST, D3DFVF_VERTEX,
										    mesh->vertices, mesh->vertexNbr, 
											mesh->linelist, mesh->faceNbr*4, D3DDP_DONOTLIGHT );
}*/



int RenderScene(GEM_SCENE* scene, DWORD frame)
{
	GEM_OBJECT*			tmp;

	if( !scene )
		return gem_error_null_pointer;
	
	TransformScene( scene, frame );

	tmp = scene->objectList;
	
	for( ; tmp ; tmp=tmp->next )
	{
		if( tmp->type==MESH )
			if( RenderMesh( scene, (GEM_MESH*)tmp->object ) != gem_error_ok)
				return 1;
	}	

	tmp = scene->objectList;
	
	for( ; tmp ; tmp=tmp->next )
	{
		if( tmp->type==LIGHT )
			RenderLight( scene, (GEM_LIGHT*)tmp->object );		
	}	


	return gem_error_ok;
}


void InitCamera(GEM_CAMERA* camera, D3DVALUE aspect, D3DVALUE nearZ, D3DVALUE farZ)
{
	camera->aspect = aspect;
	camera->nearZ = nearZ;
	camera->farZ = farZ;	
}


int InitAllCameras(GEM_SCENE* scene, D3DVALUE aspect, D3DVALUE nearZ, D3DVALUE farZ)
{
	if (!scene)
		return gem_error_null_pointer;

	GEM_OBJECT*		tmp = scene->objectList;

	for( ; tmp ; tmp=tmp->next )
	{
		if( tmp->type==CAMERA )
			InitCamera( (GEM_CAMERA*)tmp->object, aspect, nearZ, farZ );
	}

	return gem_error_ok;
}


void AddElement(GEM_ELEMENT* elem, GEM_MESH* mesh)
{
	GEM_ELEMENT*			tmp = mesh->elemList;

	if( !tmp )
	{
		elem->next = NULL;
		elem->prev = NULL;
		mesh->elemList = elem;
	}
	else
	{
		for( ; tmp->next ; tmp=tmp->next );

		tmp->next = elem;
		elem->prev = tmp;
		elem->next = NULL;
	}
}


int InitScene(GEM_SCENE* scene, LPDIRECT3DDEVICE3 pd3dDevice, D3DVALUE aspect, D3DVALUE nearZ, D3DVALUE farZ, DWORD flags)
{
	int			err;

	if (!scene)
		return gem_error_null_pointer;
	
	scene->pd3dDevice = pd3dDevice;
	scene->flags = flags;
	scene->lsize = 20.0f;
	

	if( (err=InitAllLights( scene ))!=gem_error_ok )
		return err;

	if( (err=InitAllCameras( scene, aspect, nearZ, farZ ))!=gem_error_ok )
		return err;

	return gem_error_ok;
}

int CloseScene(GEM_SCENE* scene)
{
	LPDIRECT3DVIEWPORT3		pd3dViewport;

	if( !scene )
		return gem_error_null_pointer;

	if( !scene->pd3dDevice )
		return gem_error_scene_not_initialized;

	if( scene->pd3dDevice->GetCurrentViewport( &pd3dViewport )!=D3D_OK )
		return gem_error_no_viewport;

	for( GEM_OBJECT* tmp = scene->objectList ; tmp ; tmp=tmp->next )
	{
		if( tmp->type==LIGHT )
		{
			GEM_LIGHT*	light = (GEM_LIGHT*)tmp->object;
			pd3dViewport->DeleteLight( light->pd3dLight );				
		}
	}

	return gem_error_ok;
}



int ReleaseScene(GEM_SCENE* scene)
{
	return gem_error_ok;
}

int SetActiveCamera(GEM_SCENE* scene, GEM_CAMERA* camera)
{
	if( !scene || !camera )
		return gem_error_null_pointer;

	scene->camera = camera;

	return gem_error_ok;
}

int SetLightTexture(GEM_SCENE* scene, LPDIRECT3DTEXTURE2 ptextTexture)
{
	if( !scene || !ptextTexture )
		return gem_error_null_pointer;

	scene->ptexLightText = ptextTexture;

	return gem_error_ok;
}

void SetLightSize(GEM_SCENE* scene, D3DVALUE size)
{
	scene->lsize = size;
}

		


















