//---GEM--------------------------------------------------------------------------
//	textures.cpp - GEM texture engine
//
//	version:                                                                       
//		0.1		19/08/99		ACiD	initial version                                               
//		0.2		13/09/99		ACiD	added tga & jpeg readers
//
//	desc:                                                                        
//		this version handling bmp, tga and jpeg files (to achive jpeg decompresion
//		jpeg.lib by IJP was included ) this engine loaded a file, create a surface
//		to which a image are copied and then obtaining a Texture interface
//		engine creates a 16bit surface for full color image and 8bit for paletized
//		i decided to use only 16bit textures because many graphic devices cannot
//		utilize textures from system memory so all textures must fit in video 
//		memory in 4MB you can fit about 23 256x256 16bit textures and only 16
//		textures 256x256 in 24bit, and becaus there is vere little difrence in
//		quality between 16 an 24 bit textures choice was clear
// 
//	(c)	ZONE51 1999
//--------------------------------------------------------------------------------

#define		STRICT	
#include	"gem.h"
#include	"convert.h"
#include	<stdio.h>
#include	<tchar.h>
extern "C"{
#include	"jpeglib.h"
}



enum TGAImageType
{
	TGAColorMapped = 1,
	TGAFullColor,
	TGAGrayScale,
	TGACompressedColorMapped = 9,
	TGACompressedFullColor,
	TGACompressedGrayScale
};

#pragma pack(1)
struct TGAHeader 
{
    BYTE	IDLength;		/* length of Identifier String */
    BYTE	ColorMapType;		/* 0 = no map */
    BYTE	ImgType;		/* image type (see below for values) */
    WORD    ColorMapIndex;	/* index of first color map entry */
    WORD	ColorMapLenght;	/* number of entries in color map */
    BYTE	ColorSize;		/* size of color map entry (15,16,24,32) */
    WORD	xorg;			/* x origin of image */
    WORD	yorg;			/* y origin of image */
    WORD    width;			/* width of image */
    WORD	height;			/* height of image */
    BYTE	bpp;			/* pixel size (8,16,24,32) */
	BYTE	attribs;    
};
#pragma pack()

#pragma pack (1)
struct BMPHeader 
{		
	WORD		ID;	
	DWORD		filesize;
	DWORD		res;
	DWORD		pixeloffset;

	DWORD		bmisize;
	DWORD		width;
	DWORD		height;
	WORD		planes;
	WORD		bpp;
	DWORD		compression;
	DWORD		cmpsize;	/* size of compressed image */
	DWORD		xscale;		/* pixels per meter */
	DWORD		yscale;
	DWORD		colors;		/* number of colors used */
	DWORD		impcolors;	/* number of important colors */	
};
#pragma pack ()



#define SAFE_DELETE(p)  { if(p) { delete (p);     (p)=NULL; } }
#define SAFE_RELEASE(p) { if(p) { (p)->Release(); (p)=NULL; } }

GEM_TEXTURE::~GEM_TEXTURE()
{
	SAFE_DELETE( next );		
	SAFE_RELEASE( pddsSurface );
	SAFE_RELEASE( pd3dTexture );
	SAFE_DELETE( pBuffer );
}


struct	TEXTUREFORMATSEARCH
{
	LPDDPIXELFORMAT			pddpf;
		
	BOOL					bAlpha;
	BOOL					bFound;
	WORD					bpp;
};


//--------------------------------------------------------------------------------
//name: GetDDFromDevice()
//desc: retrives a DD interface from D3DDevice interface
//--------------------------------------------------------------------------------
LPDIRECTDRAW4 GetDDFromDevice( LPDIRECT3DDEVICE3 pd3dDevice )
{
	LPDIRECTDRAW4        pDD = NULL;
	LPDIRECTDRAWSURFACE4 pddsRender;

    if( pd3dDevice )
	{
	    // Get the current render target
		if( SUCCEEDED( pd3dDevice->GetRenderTarget( &pddsRender ) ) )
		{
		    // Get the DDraw4 interface from the render target
			pddsRender->GetDDInterface( (VOID**)&pDD );
			pddsRender->Release();
		}
	}

	return pDD;
}


//--------------------------------------------------------------------------------
//name: TextureSearch()
//desc: using in pixel format enumeration to find desire format of texture
//		looking either for 8bit palettizet format or any 16bit format, if
//		16bit format was found BPP member of struct TFS are set to actual 
//		bit foramt it means 16 or 15 cause DX dont diffrents it
//--------------------------------------------------------------------------------
static HRESULT CALLBACK TextureSearch( DDPIXELFORMAT* pddpf, LPVOID pparam )
{
    if( NULL==pddpf || NULL==pparam )
        return DDENUMRET_OK;

	TEXTUREFORMATSEARCH* pTFS = (TEXTUREFORMATSEARCH*)pparam;

    // Skip any funky modes
    if( pddpf->dwFlags & (DDPF_LUMINANCE|DDPF_BUMPLUMINANCE|DDPF_BUMPDUDV) )
        return DDENUMRET_OK;

	// Else, skip any paletized formats (all modes under 16bpp)
	if( pddpf->dwRGBBitCount < 16 )
		return DDENUMRET_OK;

	// skip any FourCC formats
	if( pddpf->dwFourCC != 0 )
		return DDENUMRET_OK;

	if( !pTFS->bAlpha )
	{				
		if( pddpf->dwRGBBitCount == 16 )
		{	
			if( pddpf->dwFlags&DDPF_ALPHAPIXELS )
				return DDENUMRET_OK;
			
			if( pddpf->dwRBitMask == 0xF800 )
				pTFS->bpp = 16;
			else
				if( pddpf->dwRBitMask == 0x7C00 )	//this lines skip a funky BGR modes
					pTFS->bpp = 15;
				else
					return DDENUMRET_OK;

			memcpy( pTFS->pddpf, pddpf, sizeof(DDPIXELFORMAT) );
			
			pTFS->bFound = TRUE;
			return DDENUMRET_CANCEL;
		}		
    }
	else
	{		
		if( pddpf->dwRGBBitCount == 32 && pddpf->dwFlags&DDPF_ALPHAPIXELS )
		{
			if( pddpf->dwRBitMask != 0x00FF0000 ) //this skip funky BGR modes
				return DDENUMRET_OK;

			memcpy( pTFS->pddpf, pddpf, sizeof(DDPIXELFORMAT) );			
			pTFS->bpp = 32;
			pTFS->bFound = TRUE;
			return DDENUMRET_CANCEL;
		}
	}
		
    return DDENUMRET_OK;
}

//----------------------------------------------------------------------------
//	main texture engine
//----------------------------------------------------------------------------

static	GEM_TEXTURE*		g_TextureList = NULL;

class GEM_TEXTUREENGINE
{
public:

	GEM_TEXTUREENGINE();
   ~GEM_TEXTUREENGINE();

} g_gteTextureEngine;

GEM_TEXTUREENGINE::GEM_TEXTUREENGINE()
{
}

GEM_TEXTUREENGINE::~GEM_TEXTUREENGINE()
{
	SAFE_DELETE( g_TextureList );
}

//----------------------------------------------------------------------------
//	loaders function
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
//	name: LoadBMPFile()
//	desc: only true color and 256 colors images are handled
//----------------------------------------------------------------------------

HRESULT	LoadBMPFile(TCHAR* name, GEM_TEXTURE* texture)
{
	BMPHeader	header;	
	FILE*		imgFile = fopen( name, "rb" );
	LONG		i, j;
	BYTE		r, g, b;	
	WORD		size = (texture->bpp ==32 ? 4 : 2);

	if(	!imgFile )
		return E_FAIL;

	fread( &header, sizeof(BMPHeader), 1, imgFile );

	if( header.compression )
		return E_FAIL;
	
	texture->width	= header.width;
	texture->height = header.height;
	texture->pBuffer= new BYTE[texture->width*texture->height*size];

	if( header.bpp == 24 )
	{							
		for( i = 0 ; i<(LONG)header.height; i++ )
			for( j = 0 ; j<(LONG)header.width ; j++ )
			{
				fread( &b, 1, 1, imgFile );
				fread( &g, 1, 1, imgFile );
				fread( &r, 1, 1, imgFile );

				switch( texture->bpp )
				{
					case 32: 
						*((DWORD*)texture->pBuffer+i*header.width+j) = (r<<16) + (g<<8) +b;
					break;

					case 16:
						*((WORD*)texture->pBuffer+i*header.width+j) = C24Convert16( r, g, b );
					break;

					case 15:
						*((WORD*)texture->pBuffer+i*header.width+j) = C24Convert15( r, g, b );
					break;
				}
			}							
	}

	fclose( imgFile );

	return S_OK;
}

//----------------------------------------------------------------------------
//name: LoadJPGfile()
//desc: loades a .jpg file, decompresion rutine comes from IJP (IndependedJPEG
//		Group)
//----------------------------------------------------------------------------
HRESULT LoadJPGFile(TCHAR* name, GEM_TEXTURE* texture)
{
	WORD		size = (texture->bpp ==32 ? 4 : 2);	
	DWORD		line;
	DWORD		width;

	//seting up a decompresion stuctures
	struct jpeg_decompress_struct	cinfo;
	struct jpeg_error_mgr			jerr;	

	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_decompress(&cinfo);

	//seting input source
	FILE*							infile;
	
	if( ( infile = fopen( name, "rb" ) ) == NULL ) 
		return E_FAIL;

	jpeg_stdio_src( &cinfo, infile );
	jpeg_read_header( &cinfo, TRUE );
	jpeg_start_decompress( &cinfo );

	texture->width		 = cinfo.output_width;
	texture->height		 = cinfo.output_height;	
	texture->pBuffer	 = new BYTE[texture->width*texture->height*size];	

	line = cinfo.output_height - 1;	

	width = cinfo.output_width;

	if( cinfo.out_color_components != 3 )
		return E_FAIL;

	JSAMPROW	sl = new JSAMPLE[cinfo.output_width*cinfo.out_color_components];

	while (cinfo.output_scanline < cinfo.output_height)
	{
		jpeg_read_scanlines( &cinfo, &sl, 1);		
		for( DWORD i = 0 ; i<cinfo.output_width ; i++ )		
		{
			switch( texture->bpp )
			{
				case 32: 
					*((DWORD*)texture->pBuffer+line*width+i) = (sl[i*3]<<16) + (sl[3*i+1]<<8) + sl[3*i+2];
				break;

				case 16:
					*((WORD*)texture->pBuffer+line*width+i) = C24Convert16( sl[i*3], sl[i*3+1], sl[3*i+2] );
				break;

				case 15:
					*((WORD*)texture->pBuffer+line*width+i) = C24Convert15( sl[i*3], sl[i*3+1], sl[3*i+2] );
				break;
			}
		}
		
		line--;
	}

	jpeg_finish_decompress(&cinfo);
	jpeg_destroy_decompress(&cinfo);
	fclose( infile );

	delete sl;

	return S_OK;
}


HRESULT LoadTGAFile(TCHAR* name, GEM_TEXTURE* texture)
{
	TGAHeader			header;	
	FILE*				imgFile = fopen( name, "rb" );
	LONG				i, j;	
	WORD				size = (texture->bpp ==32 ? 4 : 2);	
	WORD				c1;
	DWORD				c2;
	WORD*				buff16;
	DWORD*				buff32;

	if(	!imgFile )
		return E_FAIL;

	fread( &header, sizeof(TGAHeader), 1, imgFile );
	
	if( header.ImgType!=TGAFullColor )
		return E_FAIL;

	texture->width		 = header.width;
	texture->height		 = header.height;
	texture->pBuffer	 = new BYTE[header.width*header.height*size];			
	switch( header.bpp )
	{
		case 16:					
			for( i = (LONG)header.height-1 ; i>=0 ; i--)
				for( j = 0 ; j<(LONG)header.width ; j++ )
				{
					fread( &c1, sizeof(WORD), 1, imgFile );
					switch( texture->bpp )										
					{
						case 32:							 
							 *((DWORD*)texture->pBuffer+i*header.width+j) = C15Convert24(c1);
						break;

						case 16:
							 *((WORD*)texture->pBuffer+i*header.width+j) = C15Convert16(c1);	 
						break;

						case 15:
							 *((WORD*)texture->pBuffer+i*header.width+j) = c1;	
						break;	
					}
				}			
		break;

		case 32:
			for( i = (LONG)header.height-1 ; i>=0 ; i--)
				for( j = 0 ; j<(LONG)header.width ; j++ )
				{
					fread( &c2, sizeof(DWORD), 1, imgFile );					
					switch( texture->bpp )										
					{
						case 32:
							 *((DWORD*)texture->pBuffer+i*header.width+j) = c2;
						break;
						case 16:
							 *((WORD*)texture->pBuffer+i*header.width+j) = C24Convert16(c2);	
						break;
						case 15:
							 *((WORD*)texture->pBuffer+i*header.width+j) = C24Convert15(c2);	
						break;	
					}
				}			
		break;

		default:
			return E_FAIL;
		break;
	}

	return S_OK;
}


HRESULT CopyToSurface(GEM_TEXTURE* texture)
{
	LPDIRECTDRAWSURFACE4	pdds;
	LPDIRECTDRAW4			pDD;
	LPDIRECTDRAWPALETTE		pddPalette;
	WORD					size = (texture->bpp ==32 ? 4 : 2);	
	

	if ( FAILED (texture->pddsSurface->GetDDInterface( (LPVOID*)&pDD ) ) )
		return E_FAIL;
		
	DDSURFACEDESC2			ddsd;
	ZeroMemory( &ddsd, sizeof(ddsd) );
	ddsd.dwSize = sizeof(ddsd);
	ddsd.ddpfPixelFormat.dwSize = sizeof(ddsd.ddpfPixelFormat);

	ddsd.dwFlags		= DDSD_WIDTH | DDSD_HEIGHT | DDSD_PIXELFORMAT | DDSD_PITCH | DDSD_LPSURFACE | DDSD_CAPS;
	ddsd.dwHeight		= texture->height;
	ddsd.dwWidth		= texture->width;
	
	ddsd.lpSurface		= texture->pBuffer;
	ddsd.ddsCaps.dwCaps = DDSCAPS_TEXTURE | DDSCAPS_SYSTEMMEMORY;
	ddsd.lPitch			= texture->width*size;

	texture->pddsSurface->GetPixelFormat( &ddsd.ddpfPixelFormat );
	
	if( FAILED( pDD->CreateSurface( &ddsd, &pdds, NULL ) ) )
		return E_FAIL;		
				
	if( FAILED( texture->pddsSurface->BltFast( 0, 0, pdds, NULL, NULL ) ) )
		return E_FAIL;
	
	pdds->Release();

	return DD_OK;
}

/*
HRESULT LoadGIFFile(
HRESULT LoadPCXFile(
HRESULT LoadPNGFile(
*/
//----------------------------------------------------------------------------
//name: FindTexture()
//desc:	look for texture in global texture list and retur its pointer or null 
//		in case that texture doesnt exist
//----------------------------------------------------------------------------
GEM_TEXTURE* FindTexture(TCHAR* name)
{
	for( GEM_TEXTURE* tmp=g_TextureList ; tmp ; tmp=tmp->next )
	{
		if( !lstrcmpi( name, tmp->name ) )
			return tmp;
	}

	return NULL;
}


//----------------------------------------------------------------------------
//name: LoadImageFile()
//desc: loads a image from .bmp file, check a resource first and then try to
//		load image from file
//----------------------------------------------------------------------------
HRESULT LoadImageFile(GEM_TEXTURE* lpgtTexture)
{
	TCHAR*			strExt;
	TCHAR*			strName = lpgtTexture->name;
	HRESULT			hr;

	strExt = _tcsrchr( strName, TEXT('.') );

	if( strExt == NULL )
		return E_FAIL;

	if( !lstrcmpi( strExt, ".bmp" ) )
	{
		hr = LoadBMPFile( strName, lpgtTexture );
		return hr;
	}
	else
		if( !lstrcmpi( strExt, ".jpg") )
		{
			hr = LoadJPGFile( strName, lpgtTexture );
			return hr;
		}
		else
			if( !lstrcmpi( strExt, ".tga" ) )
			{
				hr = LoadTGAFile( strName, lpgtTexture );
				return hr;
			}
			else
				return E_FAIL;

	return S_OK;
}


//----------------------------------------------------------------------------
//name: GEM_LoadTexture()
//desc: this function create a new texture and add it to global list, load a
//		image from file or resourece, so far only .bmp files can be loaded
//		this function do not create any DDSurfaces or D3DTextures, you must
//		use one of update function
//----------------------------------------------------------------------------
HRESULT AddTexture(TCHAR* name, DWORD stage, DWORD flags)
{
	FILE*			file;

	if( FindTexture( name ) )
		return S_OK;

	GEM_TEXTURE*	pgtTexture = new GEM_TEXTURE;
	if( !pgtTexture )
		return DDERR_OUTOFMEMORY;

	ZeroMemory( pgtTexture, sizeof(GEM_TEXTURE) );

	pgtTexture->dwStage = stage;
	pgtTexture->bAlphaChanel = flags;
	lstrcpy( pgtTexture->name, name );

	if( (file = fopen( name, "rb")) == NULL )
	{
		delete pgtTexture;
		return E_FAIL;
	}

	fclose( file );

	//add texture to list
	pgtTexture->next = g_TextureList;	
	if( g_TextureList )
		g_TextureList->prev=pgtTexture;

	g_TextureList = pgtTexture;

	return S_OK;
}


//----------------------------------------------------------------------------
//name: CopyToSurface()
//desc: copise a bitmap to surface
//----------------------------------------------------------------------------
/*
HRESULT CopyToSurface(LPDIRECTDRAWSURFACE4 lpddsTarget, HBITMAP hbm )
{
	HDC						hdcBitmap;
	HDC						hdcSurface;
	DDSURFACEDESC2			ddsd;
	HRESULT					hr;
	//LPDIRECTDRAW4			pDD4;

	ddsd.dwSize = sizeof(ddsd);
	ddsd.ddpfPixelFormat.dwSize = sizeof(ddsd.ddpfPixelFormat);

	hr = lpddsTarget->GetSurfaceDesc( &ddsd );
	if( FAILED(hr) )
		return hr;

	hdcBitmap = CreateCompatibleDC( NULL );
	if( !hdcBitmap )
		return E_FAIL;

	if( !SelectObject( hdcBitmap, hbm ) )
		return E_FAIL;

	DWORD	retval;

	if( SUCCEEDED( lpddsTarget->GetDC( &hdcSurface ) ) )
	{
		retval=BitBlt( hdcSurface, 0, 0, ddsd.dwWidth, ddsd.dwHeight, hdcBitmap, 0, 0, SRCCOPY );		
		if ( FAILED( lpddsTarget->ReleaseDC( hdcSurface ) ) )
			return E_FAIL;
	}

	DeleteDC(hdcBitmap);
	if (!retval) 
		return E_FAIL;

	return S_OK;
}
*/

//----------------------------------------------------------------------------
//name: GEM_UpdateTexture()
//desc: create a DDSurface and DDTexture and copy image to surface
//----------------------------------------------------------------------------
HRESULT UpdateTexture(TCHAR* name, LPDIRECT3DDEVICE3 pd3dDevice)
{
	HRESULT					hr;
	GEM_TEXTURE*			lpgtTexture = FindTexture( name );
	DDSURFACEDESC2			ddsd;

	ZeroMemory( &ddsd, sizeof(ddsd) );
	ddsd.dwSize					= sizeof(ddsd);
	ddsd.ddpfPixelFormat.dwSize = sizeof(ddsd.ddpfPixelFormat);

	if( lpgtTexture==NULL )
		return DDERR_NOTFOUND;

	LPDIRECTDRAW4			lpDD4 = GetDDFromDevice( pd3dDevice );

	if( !lpDD4 )
		return DDERR_NOTFOUND;
	

	//searching for apropiate pixel format
	TEXTUREFORMATSEARCH			tfs;
	ZeroMemory( &tfs, sizeof(tfs) );

	tfs.pddpf		= &ddsd.ddpfPixelFormat;
	tfs.bAlpha		= lpgtTexture->bAlphaChanel;
	tfs.bFound		= FALSE;	

	pd3dDevice->EnumTextureFormats( &TextureSearch, &tfs );
	if( !tfs.bFound )
		return DDERR_NOTFOUND;

	lpgtTexture->bpp = tfs.bpp;

	if( FAILED( LoadImageFile( lpgtTexture ) ) )
		return E_FAIL;
		
	DWORD					dwHeight = (DWORD)lpgtTexture->height;
	DWORD					dwWidth  = (DWORD)lpgtTexture->width;
		
	ddsd.dwFlags		= DDSD_HEIGHT | DDSD_WIDTH | DDSD_PIXELFORMAT | 
						  DDSD_CAPS | DDSD_TEXTURESTAGE;
	ddsd.dwHeight		= dwHeight;
	ddsd.dwWidth		= dwWidth;
	ddsd.dwTextureStage = lpgtTexture->dwStage;
	ddsd.ddsCaps.dwCaps = DDSCAPS_TEXTURE ;

	//ajusting dimension of the bitmap to be a pow of 2 
	for( ddsd.dwHeight=1 ; dwHeight>ddsd.dwHeight ; ddsd.dwHeight<<=1 );
	for( ddsd.dwWidth=1  ; dwWidth>ddsd.dwWidth   ; ddsd.dwWidth<<=1  );

	//set a texture dimension to be a square
	if( ddsd.dwHeight>ddsd.dwWidth )
		ddsd.dwWidth=ddsd.dwHeight;
	else
		if( ddsd.dwHeight<ddsd.dwWidth )
			ddsd.dwHeight=ddsd.dwWidth;	
	

	hr = lpDD4->CreateSurface( &ddsd, &lpgtTexture->pddsSurface, NULL); 
	if(	FAILED(hr) )
		return hr;

	hr = lpgtTexture->pddsSurface->QueryInterface( IID_IDirect3DTexture2, (LPVOID*)&lpgtTexture->pd3dTexture );
	if( FAILED(hr) )
		return hr;	

	return CopyToSurface( lpgtTexture );
}

HRESULT UpdateAllTextures(LPDIRECT3DDEVICE3 pd3dDevice)
{
	HRESULT				hr;

	for( GEM_TEXTURE* tmp=g_TextureList ; tmp ; tmp=tmp->next )
	{
		hr = UpdateTexture( tmp->name, pd3dDevice );
		if( FAILED(hr) )
			return hr;
	}

	return S_OK;
}

LPDIRECT3DTEXTURE2 GetTexture(TCHAR* name)
{
	GEM_TEXTURE*	lpgtTexture = FindTexture( name );

	if( !lpgtTexture )
		return NULL;

	return lpgtTexture->pd3dTexture;
}

LPDIRECTDRAWSURFACE4 GetSurface(TCHAR* name)
{
	GEM_TEXTURE*	lpgtTexture = FindTexture( name );

	if( !lpgtTexture )
		return NULL;

	return lpgtTexture->pddsSurface;
}

HRESULT RemoveTexture(TCHAR* name)
{
	GEM_TEXTURE*	lpgtTexture = FindTexture( name );

	if( !lpgtTexture )
		return DDERR_NOTFOUND;

	if( lpgtTexture->prev )
		lpgtTexture->prev->next = lpgtTexture->next;

	if( lpgtTexture->next )
		lpgtTexture->next->prev = lpgtTexture->prev;

	lpgtTexture->prev = NULL;
	lpgtTexture->next = NULL;

	SAFE_DELETE( lpgtTexture );
	
	return S_OK;
}


HRESULT LoadAlphaChanel(TCHAR* name, GEM_TEXTURE* texture)
{
	TCHAR*					strExt;
	TCHAR*					strName = name;
	HRESULT					hr;	
	BMPHeader				header;
	DDSURFACEDESC2			ddsd;
	
	ZeroMemory( &ddsd, sizeof(ddsd) );
	ddsd.dwSize = sizeof(ddsd);
	ddsd.ddpfPixelFormat.dwSize = sizeof(ddsd.ddpfPixelFormat);
	
	
	strExt = _tcsrchr( strName, TEXT('.') );

	if( strExt == NULL )
		return E_FAIL;

	if( !lstrcmpi( strExt, ".bmp" ) )
	{
		FILE*			imgFile = fopen( strName, "rb" );

		fread( &header, sizeof(BMPHeader), 1, imgFile );

		if( header.compression )
			return E_FAIL;

		if( header.bpp!=8 )
			return E_FAIL;

		if( header.width!=texture->width || header.height!=texture->height )
			return E_FAIL;				
		
		fseek( imgFile, 1078, SEEK_SET );		
		
		if( SUCCEEDED( texture->pddsSurface->Lock( NULL, &ddsd, DDLOCK_NOSYSLOCK, NULL ) ) )
		{
			BYTE*	buff = (BYTE*)ddsd.lpSurface;
			BYTE	val;
			for( LONG i = (LONG)header.height-1 ; i>=0 ; i-- )
				for( LONG j = 0 ; j<(LONG)header.width ; j++ )				
				{
					fread( &val, 1, 1, imgFile );
					buff[i*ddsd.lPitch+j*4-1] = val;
				}

			texture->pddsSurface->Unlock( NULL );
		}
		else
			return E_FAIL;
		
		fclose( imgFile );
	}
	else 
		return E_FAIL;

	return S_OK;
}


HRESULT AddAlphaChanel(TCHAR* texName, TCHAR* alphaName)
{
	GEM_TEXTURE*		texture = FindTexture( texName );	

	if( !texture )
		return E_FAIL;

	if( !texture->pddsSurface )
		return E_FAIL;	

	if( !texture->bAlphaChanel )
		return E_FAIL;

	return LoadAlphaChanel( alphaName, texture );
}


HRESULT SetColorKey(TCHAR* textName, DWORD colKey)
{
	GEM_TEXTURE*		texture = FindTexture( textName );
	
	if( !texture )
		return E_FAIL;

	if( !texture->pddsSurface )
		return E_FAIL;	

	DDCOLORKEY colorKey;

	colorKey.dwColorSpaceLowValue	= colKey;     
	colorKey.dwColorSpaceHighValue	= colKey;
	
	texture->pddsSurface->SetColorKey( DDCKEY_SRCBLT, &colorKey );
	texture->pddsSurface->SetColorKey( DDCKEY_SRCOVERLAY, &colorKey );

	return S_OK;
}

HRESULT SetAlphaChanel(TCHAR* texName, BYTE val)
{
	GEM_TEXTURE*		texture = FindTexture( texName );	
	DDSURFACEDESC2			ddsd;
	
	ZeroMemory( &ddsd, sizeof(ddsd) );
	ddsd.dwSize = sizeof(ddsd);
	ddsd.ddpfPixelFormat.dwSize = sizeof(ddsd.ddpfPixelFormat);

	if( !texture )
		return E_FAIL;

	if( !texture->pddsSurface )
		return E_FAIL;	

	if( !texture->bAlphaChanel )
		return E_FAIL;

	if( SUCCEEDED( texture->pddsSurface->Lock( NULL, &ddsd, DDLOCK_NOSYSLOCK, NULL ) ) )
	{
		BYTE* buff = (BYTE*)ddsd.lpSurface;
		for( LONG i = 0; i<(LONG)ddsd.dwHeight ; i++ )
			for( LONG j = 0 ; j<(LONG)ddsd.dwWidth ; j++ )				
				buff[i*ddsd.lPitch+j*4-1] =	val;				 

		texture->pddsSurface->Unlock( NULL );
	}

	return S_OK;
}

