
//
//	Vector.h - 3D vector template for realtime graphics
//	Written by Xavier Defrang <xdefrang@csi.com>
//

#ifndef _VECTOR_H_INCLUDED
#define _VECTOR_H_INCLUDED

#include <cmath>
#include <iostream>

// Check out typedef at the end of this file !!!

template <class coord> class Generic_Vector {

	inline coord sqr( coord k ) const { return k*k; };

public:
	union { float x; float r; };
	union { float y; float g; };
	union { float z; float b; };

	Generic_Vector<coord>() : x(0), y(0), z(0) {};
	Generic_Vector<coord>( coord nx, coord ny, coord nz ) : x(nx), y(ny), z(nz) {};
	Generic_Vector<coord>( const Generic_Vector<coord> &v ) : x(v.x), y(v.y), z(v.z) {};
	Generic_Vector<coord>( const Generic_Vector<coord> &a, const Generic_Vector<coord> &b ) : x(b.x-a.x), y(b.y-a.y), z(b.z-a.z) {};

	coord distance() const
	{
		coord tmp = sqr(x)+sqr(y)+sqr(z);
		return tmp ? sqrt(tmp) : 0;	
	}

	coord distance( const Generic_Vector<coord> &v ) const
	{
		coord tmp = sqr(v.x-x)+sqr(v.y-y)+sqr(v.z-z);
		return tmp ? sqrt(tmp) : 0;	
	}

	coord dotproduct( const Generic_Vector<coord> &v ) const
	{
		return x*v.x+y*v.y+z*v.z;
	}

	const Generic_Vector<coord> &normalize()
	{
		coord tmp = distance();
		if ( tmp ) {
			tmp = 1/tmp;
			x *= tmp;
			y *= tmp;
			z *= tmp;
		} else {
			x = 0;
			y = 0;
			z = 0;
		}
		return *this;
	}

	// instead of returing self-reference, we return the previous length of
	// the vector, this side-effect saves lots of time!
	coord normalize2()
	{
		coord tmp = distance();
		if ( tmp ) {
			tmp = 1/tmp;
			x *= tmp;
			y *= tmp;
			z *= tmp;
		} else {
			x = 0;
			y = 0;
			z = 0;
		}
		return tmp;
	}

	const Generic_Vector<coord> &combine( const Generic_Vector<coord> &v )
	{
		x *= v.x;
		y *= v.y;
		z *= v.z;
		return *this;
	}

	const Generic_Vector<coord> &operator - ()
	{
		x = -x;
		y = -y;
		z = -z;
		return *this;
	}

	const Generic_Vector<coord> &operator += ( const Generic_Vector<coord> &v )
	{
		x += v.x;
		y += v.y;
		z += v.z;
		return *this;
	}
	
	const Generic_Vector<coord> &operator -= ( const Generic_Vector<coord> &v )
	{
		x -= v.x;
		y -= v.y;
		z -= v.z;
		return *this;
	}
	
	const Generic_Vector<coord> &operator += ( coord k )
	{
		x += k;
		y += k;
		z += k;
		return *this;
	}
	
	const Generic_Vector<coord> &operator -= ( coord k )
	{
		x -= k;
		y -= k;
		z -= k;
		return *this;
	}
	
	const Generic_Vector<coord> &operator *= ( coord k )
	{
		x *= k;
		y *= k;
		z *= k;
		return *this;
	}
/*
	const Generic_Vector<coord> &operator *= ( const Generic_Matrix<coord> &m )
	{
		Generic_Vector<coord> tmp = *this;
		x = m(0,0)*tmp.x+m(1,0)*tmp.y+m(2,0)*tmp.z+m(3,0);	
		y = m(0,1)*tmp.x+m(1,1)*tmp.y+m(2,1)*tmp.z+m(3,1);	
		z = m(0,2)*tmp.x+m(1,2)*tmp.y+m(2,2)*tmp.z+m(3,2);	
		return *this;
	}
*/
	Generic_Vector<coord> operator + ( const Generic_Vector<coord> &v ) const
	{
		Generic_Vector<coord> tmp = *this;
		return tmp += v;
	}

	Generic_Vector<coord> operator - ( const Generic_Vector<coord> &v ) const
	{
		Generic_Vector<coord> tmp = *this;
		return tmp -= v;
	}

	Generic_Vector<coord> operator + ( coord k ) const
	{
		Generic_Vector<coord> tmp = *this;
		return tmp += k;
	}

	Generic_Vector<coord> operator - ( coord k ) const
	{
		Generic_Vector<coord> tmp = *this;
		return tmp -= k;
	}

	Generic_Vector<coord> operator * ( coord k ) const
	{
		Generic_Vector<coord> tmp = *this;
		return tmp *= k;
	}

	Generic_Vector<coord> operator * ( const Generic_Vector<coord> &v ) const
	{
		return Generic_Vector<coord>( y*v.z-z*v.y, z*v.x-x*v.z, x*v.y-y*v.x);
	}
	
	friend ostream &operator << ( ostream &os, const Generic_Vector<coord> &v )
	{
		return os << v.x << "\t" << v.y << "\t" << v.z;
	}

};

typedef Generic_Vector<float> Vector;

#endif // _VECTOR_H_INCLUDED


