///////////////////////////////////////////////////////////////////////////////
//	DemoNews Reader for Windows
//	-- Luc-Eric Rousseau (l.e.rousseau@usa.net), July 1996
//
//	Reader for Honet's DemoNews ascii file
//	Entry point : WinMain.
// TABS=3
//
//	Version 1.01
//
//	1.00	-	first release (obviously)
//	1.01	-	relaxed requierement on first line (.Start.Of.Demonews)
//			-	Fixed bugs with upload section showing up weird
//			-	freeze window update of text edit when filling it (avoids flicker)
//			-	multiple underlines work for explicit.nfo
///////////////////////////////////////////////////////////////////////////////

#define WIN32_LEAN_AND_MEAN	// removes unused stuff form Windows.h
#include <windows.h>				// to speed up compile
#include <objbase.h>
#include <richedit.h>
#include <commctrl.h>
#include <commdlg.h>
#include	<stdio.h>
#include <stdlib.h>

#include "resource.h"

#define WM_LOADALLNEWS WM_USER

const char *Ls(int nID);	// loads a string

///////////////////////////////////////////////////////////////////////////////
//	== Types
//	a linked-list for loading demo news
struct  CLineList
{
	CLineList()
	{	memset(this, 0, sizeof(CLineList));	}
	


	char			m_sz[100];	// the text

	CLineList	*pNext;
};


///////////////////////////////////////////////////////////////////////////////
//	== Global Variables
// window class names - class name is local to this app, no conflict are likely
char				g_szMainWindowClass[] = "dnr95Main";	

HINSTANCE		g_hInstance;
HWND				g_hMainWindow;
HWND				g_hTextEdit;
int				g_FontPointSize;
BOOL				g_fSplitHorizontal = FALSE;
double			g_dblSplitFraction = 0.25;
HWND				g_hTreeView;
HIMAGELIST		g_BookImages;




///////////////////////////////////////////////////////////////////////////////
//	== LoadDemoNews
//	-- Here's a big one.  Loads demonews in a linked list and sets it in a tree control
//		in a tree control

BOOL	LoadDemoNews(const char *pszFileName, HWND hTree, BOOL fByUser)
{
	TV_INSERTSTRUCT	is;
	HTREEITEM			RootNode;
	CLineList			*pListHead = 	new CLineList;
	CLineList			*pLine;
	char					szBuf[100];


	// read it all in memory
	FILE *f = fopen(pszFileName, "r");
	if (f==NULL)
		return FALSE;

	if (fgets(pListHead->m_sz, sizeof(pLine->m_sz), f) == NULL)
	{
		fclose(f);
		return FALSE;
	}
	
	// check for signature
	strupr(pListHead->m_sz);
	if (strstr(pListHead->m_sz, "START.OF") == 0)
	{
		if (!fByUser
					|| MessageBox(	g_hMainWindow, 
										Ls(IDS_MAYNOTBEGOOD), 
										NULL, 
										MB_APPLMODAL | MB_YESNO | MB_ICONQUESTION
										) == IDNO)
		{
			fclose(f);
			return FALSE;
		}
	}
	
	SendMessage(hTree, WM_SETREDRAW, FALSE, 0);

	for (pLine = pListHead; 
				fgets(pLine->m_sz, sizeof(pLine->m_sz), f) != NULL;
						pLine	= pLine->pNext)
	{
		OemToChar(pLine->m_sz, pLine->m_sz);

		pLine->pNext	= new CLineList;
	}

	fclose(f);

	int SetAllMask = TVIF_TEXT | TVIF_CHILDREN | TVIF_PARAM | TVIF_STATE | TVIF_IMAGE | TVIF_SELECTEDIMAGE;
	// add the root node
	is.hParent				= NULL;
	is.hInsertAfter		= TVI_LAST;
	is.item.mask			= SetAllMask;
	is.item.state			= TVIS_EXPANDED;
	is.item.stateMask		= 0;
	is.item.pszText		= strdup(pszFileName);
	is.item.cChildren		= 1;
	is.item.lParam			= (LPARAM)pListHead;
	is.item.iImage				= I_IMAGECALLBACK;
	is.item.iSelectedImage	= I_IMAGECALLBACK;

	RootNode = TreeView_InsertItem(hTree, &is);

	// add all sections
	HTREEITEM	FirstLevel = RootNode;
	is.hInsertAfter		= TVI_LAST;
	is.item.mask			= SetAllMask;
	is.item.state			= TVIS_EXPANDED;
	is.item.stateMask		= TVIS_EXPANDED;
	//	MAIN SECTIONS like
	//	======[content] or
	//	=-=-=-<content>
	for (pLine = pListHead; pLine != NULL;	pLine	= pLine->pNext)
	{
		if ((pLine->m_sz[0] == '=' && pLine->m_sz[1] == '=') ||
			strstr(pLine->m_sz, "=-=-") != NULL)
		{
			strncpy(szBuf, pLine->m_sz, 100);	szBuf[99] = '\0';
			
			is.hParent				= RootNode;
			is.item.pszText		= strtok(szBuf, "-=[]<>\n");
			if (is.item.pszText == NULL || strlen(is.item.pszText) == 0) is.item.pszText = "(no title)";
			//OemToChar(is.item.pszText, is.item.pszText);
			is.item.cChildren		= 0;
			is.item.lParam			= (LPARAM)pLine->pNext;
			is.item.mask			= SetAllMask;
			FirstLevel = TreeView_InsertItem(hTree, &is);
		}
		else
		// SUB-SECTION, like
		// =-------(alpha)-------=
		if (pLine->m_sz[0] == '=' && pLine->m_sz[1] == '-')
		{
			if (strpbrk(pLine->m_sz, "<([") != NULL)
			{
				strncpy(szBuf, pLine->m_sz, 100);	szBuf[99] = '\0';
				is.hParent				= FirstLevel;
				is.item.pszText		= strtok(strpbrk(szBuf, "<(["), "()<>[]");
				//OemToChar(is.item.pszText, is.item.pszText);
				is.item.cChildren		= 0;
				is.item.lParam			= (LPARAM)pLine;	// not on next line
				is.item.mask			= SetAllMask;
				TreeView_InsertItem(hTree, &is);
				is.item.hItem			= FirstLevel;
				is.item.cChildren		= 1;
				is.item.mask			= TVIF_CHILDREN;
				TreeView_SetItem(hTree, &is.item);
			}

		}
		else	
		// ARTICLES with autors names 
		//	::Title 
		//	::Name 
		//	::ect
		if (strncmp(pLine->m_sz, " :: \"", 5) == 0)
		{
			strncpy(szBuf, pLine->m_sz, 100);	szBuf[99] = '\0';

			is.hParent				= FirstLevel;
			is.item.pszText		= strtok(szBuf+4, "\"");
			//OemToChar(is.item.pszText, is.item.pszText);
			is.item.cChildren		= 0;
			is.item.lParam			= (LPARAM)pLine;
			is.item.mask			= SetAllMask;
			TreeView_InsertItem(hTree, &is);
			is.item.hItem			= FirstLevel;
			is.item.cChildren		= 1;
			is.item.mask			= TVIF_CHILDREN;
			TreeView_SetItem(hTree, &is.item);
		}

	}


	SendMessage(hTree, WM_SETREDRAW, TRUE, 0);

	return TRUE;
}

///////////////////////////////////////////////////////////////////////////////
//		 utility functions
//	==	Ls
//	--	loads a string in a static buffer, returns pointer to it
const char *Ls(int nID)
{
	static	char	aszBuf[5][512];
	static	BYTE	CurrentIndex = 4;

	if (++CurrentIndex > 4)	CurrentIndex = 0;
	LoadString(g_hInstance, nID, aszBuf[CurrentIndex], 512);
	return aszBuf[CurrentIndex];
}

///////////////////////////////////////////////////////////////////////////////
//	== SendWMSize 
//	-- sends a properly made WM_SIZE to a window
void SendWMSize(HWND hwnd)
{
	RECT rc;
	GetClientRect(hwnd, &rc);
	InvalidateRect(hwnd, NULL, TRUE);
	PostMessage(hwnd, WM_SIZE, SIZE_RESTORED, MAKELPARAM(rc.right, rc.bottom));

}

///////////////////////////////////////////////////////////////////////////////
//	== GetTreeItemInfo
//	-- get item status from a tree control
void	GetTreeItemInfo(HWND hTree, HTREEITEM hItem, 
							BOOL * pfIsExpanded, int * pcChildren, BOOL *pfIsRoot)
{
	TV_ITEM it;
	it.hItem = hItem;
	it.mask			= TVIF_CHILDREN | TVIF_STATE;
	it.stateMask	= TVIS_EXPANDED;

	TreeView_GetItem(hTree, &it);

	*pfIsExpanded  = it.state & TVIS_EXPANDED;

	*pcChildren = it.cChildren;

	*pfIsRoot = (TreeView_GetParent(hTree, hItem) == NULL);

}

///////////////////////////////////////////////////////////////////////////////
//	== CreateTextEdit
//	-- Creates a rich text edit control
//		You'll have to resize and show the window
HWND CreateTextEdit(HWND hwndParent, int nChildID)
{
	return CreateWindowEx(	WS_EX_CLIENTEDGE,
									"RichEdit", "",
									WS_CHILD | ES_MULTILINE | ES_READONLY | 
									ES_AUTOVSCROLL | WS_VSCROLL | WS_HSCROLL,
									0, 0, 100, 100,
									hwndParent,
									(HMENU)nChildID,
									g_hInstance, 0);

}

///////////////////////////////////////////////////////////////////////////////
//	== CreateTreeView
//	-- Creates a tree view control
//		You'll have to resize and show the window
HWND CreateTreeView(HWND hwndParent, int nChildID)
{
	return CreateWindowEx(	WS_EX_CLIENTEDGE,
									WC_TREEVIEW, "",
									WS_CHILD | TVS_HASBUTTONS |  TVS_HASLINES | TVS_LINESATROOT,
									0, 0, 100, 100,
									hwndParent,
									(HMENU)nChildID,
									g_hInstance, 0);

}

///////////////////////////////////////////////////////////////////////////////
//	== DrawResizeLine
//	-- draws and Xor'd line used when resizing with mouse
//		Used by MainWndProc
void	DrawResizeLine(HWND hwnd, int Pos, BOOL fHorizontal)
{
	RECT	rc;
	GetClientRect(hwnd, &rc);
	if (fHorizontal)
		{	rc.top = Pos;	rc.bottom = Pos+2; }
	else
		{	rc.left = Pos;	rc.right = Pos+2; }
	
	HDC hDC = GetDC(hwnd);
	DrawFocusRect(hDC, &rc);
	ReleaseDC(hwnd, hDC);
}


///////////////////////////////////////////////////////////////////////////////
//	== FillTextEdit
//	-- Puts demonews line into the text edit, replacing what's there
struct FILLETEXTCOOKIE
{
	char	*pBuf;
	DWORD	dwBufferSize;
	DWORD	dwTotalStreamed;
};

DWORD CALLBACK EditStreamCallback(DWORD dwCookie, 
    LPBYTE pbBuff, LONG cb, LONG FAR *pcb)
{
	FILLETEXTCOOKIE *ftc = (FILLETEXTCOOKIE *)dwCookie;

	if (ftc->dwTotalStreamed == ftc->dwBufferSize)
		*pcb = 0;
	else
	{
		DWORD dwHowMany = min((DWORD)cb, ftc->dwBufferSize - ftc->dwTotalStreamed);
		memcpy(pbBuff, ftc->pBuf+ftc->dwTotalStreamed, dwHowMany);
		ftc->dwTotalStreamed += dwHowMany;
		*pcb = dwHowMany;
	}
	return 0;
}


///////////////////////////////////////////////////////////////////////////////
//	FormatArticleToBuffer - transforms DemoNews to RTF!
void	FormatArticleToBuffer(FILLETEXTCOOKIE *ftc, CLineList *pLinesStart)
{
	char	*pDest = ftc->pBuf, *pSrc;
	char	*pBumber = ftc->pBuf + ftc->dwBufferSize - 50;
	DWORD	Total = 0;
	
	sprintf(pDest, "{\\ftf1\\pc{\\colortbl;\\red0\\green0\\blue0"
						"\\red128\\green0\\blue0;\\red0\\green0\\blue128;}\\fs%d "
						"{\\fonttbl{\\f1\\froman Times New Roman;}}",
						g_FontPointSize*2);
	pDest += strlen(pDest);

	BOOL	fFirstTitle = TRUE;
	BOOL	fIsSubSection;

	if (strncmp(pLinesStart->m_sz, "=--", 3) == 0)
	{
		pLinesStart = pLinesStart->pNext;
		fIsSubSection = TRUE;
	}
	else
		fIsSubSection = FALSE;


	for (CLineList * pLine = pLinesStart; 
					pLine != NULL && pDest < pBumber;
								pLine	= pLine->pNext)
	{
		if (strncmp(pLine->m_sz, "....", 4) == 0)
			continue;
		
		// two cases that stop a section/article
		if (pLine->m_sz[0] == '=' && pLine->m_sz[1] == '=')
			break;
	
		if (pLine->m_sz[0] == '=' && pLine->m_sz[1] == '-')
			if (!fIsSubSection || strpbrk(pLine->m_sz, "(<[") != NULL)
				break;

		
		pSrc = pLine->m_sz;

/*		if (strncmp(pLine->m_sz, " _____", 6) == 0)
		{
			pSrc += 6;
			strcpy(pDest, "{\\i\\b\\cf2       ");
			pDest += strlen(pDest);
		}
		else*/
		if (strncmp(pLine->m_sz, " :: ", 4) == 0)
		{
			sprintf(pDest, "{\\f1\\fs%d ", g_FontPointSize*3);
			pDest += strlen(pDest);
			if (fFirstTitle)
			{
				strcpy(pDest, "\\cf1\\qc ");
				fFirstTitle = FALSE;
			}
			else
				strcpy(pDest, "\\cf2\\qc ");

			pDest += strlen(pDest);
			pSrc += 4;
		}

		int	nUnderlineDepth = 0;
		while (*pSrc != '\n' && *pSrc != '\0' && pDest < pBumber)
		{
			// thanks a lot, snowman
			// so you want this in two column
			// in Explicit.nfo, hey?
			// here's a patch that'll do it
			if (strncmp(pSrc, " _____", 5) == 0)	
			{													
				strcpy(pDest, "{\\i\\b\\cf2       ");				
				pDest	+= strlen(pDest);					
				pSrc += 6;
				nUnderlineDepth++;
			}
			else
			{
				*pDest++ = *pSrc++;
				Total++;
			}
		}
		strcpy(pDest, "\\par ");
		pDest += strlen(pDest);

		while (nUnderlineDepth > 0)
		{
			strcpy(pDest, "}" );
			pDest++;
			nUnderlineDepth--;
		}
		
/*		if (strncmp(pLine->m_sz, " _____", 6) == 0)
		{
			strcpy(pDest, "}" );
			pDest++;
		}
		else*/
		if (strncmp(pLine->m_sz, " :: ", 4) == 0)
		{
			sprintf(pDest, "\\cf0 \\ql\\fs%d}", g_FontPointSize*2);
			pDest += strlen(pDest);
		}
	}

	strcpy(pDest, "}");
	pDest += strlen(pDest);

	ftc->dwBufferSize = pDest - ftc->pBuf;
	ftc->pBuf = (char *)realloc(ftc->pBuf, ftc->dwBufferSize);
}


void	FillTextEdit(HWND hTextEdit, CLineList *pLines)
{
	FILLETEXTCOOKIE	ftc;
	ftc.pBuf					=	(char *)malloc(40000);
	ftc.dwBufferSize		=	40000;
	ftc.dwTotalStreamed	=	0;

	FormatArticleToBuffer(&ftc, pLines);	

	EDITSTREAM es;
	es.dwCookie		=  (DWORD)&ftc;
	es.dwError		=  FALSE;
	es.pfnCallback	=	EditStreamCallback;

	SendMessage(hTextEdit, EM_STREAMIN, SF_RTF, (LPARAM)&es);

}

///////////////////////////////////////////////////////////////////////////////
//	== LoadAllDemonews
//	--	scan directory for DemoNews files
//
void	LoadAllDemonews()
{
	WIN32_FIND_DATA wfd;
	HANDLE hFind =	FindFirstFile("*.*", &wfd);
	if (hFind == INVALID_HANDLE_VALUE)
		return;

	do
	{
		LoadDemoNews(wfd.cFileName,g_hTreeView, FALSE);
		UpdateWindow(g_hTreeView);
	} while (FindNextFile(hFind, &wfd));
	
	FindClose(hFind);
}

///////////////////////////////////////////////////////////////////////////////
//	==	MainWndProc
//	-- event handler for the main window
LRESULT CALLBACK MainWndProc(HWND hwnd,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
	static	BOOL	fInSplitterResize = FALSE;
	static	int	nLastMousePos;	// when in SplitterResize

	switch (uMsg)
	{
		case WM_CREATE :
			g_hTextEdit = CreateTextEdit(hwnd, 2);
			ShowWindow(g_hTextEdit, SW_SHOW);
			g_hTreeView = CreateTreeView(hwnd, 1);
			TreeView_SetImageList(g_hTreeView, g_BookImages, TVSIL_NORMAL);
			ShowWindow(g_hTreeView, SW_SHOW);
			g_FontPointSize = 9;
			{
				CHARFORMAT cf;
				cf.cbSize		=	sizeof(cf);
				cf.dwMask		=	CFM_FACE | CFM_SIZE;
				cf.yHeight		=	g_FontPointSize*20;
				cf.yOffset		=	0;
				cf.bCharSet		=	OEM_CHARSET;
				cf.bPitchAndFamily	= FIXED_PITCH | FF_MODERN;
				strcpy(cf.szFaceName, "Terminal");
				SendMessage(g_hTextEdit, EM_SETCHARFORMAT, 0, (LPARAM)&cf);
			}
			PostMessage(hwnd, WM_COMMAND, ID_VIEW_SPLITVERTICAL, 0);
			PostMessage(hwnd, WM_LOADALLNEWS, 0, 0);
			break;
		case WM_LOADALLNEWS :
			LoadAllDemonews();
			break;
		case WM_GETMINMAXINFO :
		  // avoids use resizing the window too small
		  ((MINMAXINFO *)lParam)->ptMinTrackSize.x = 50;
		  ((MINMAXINFO *)lParam)->ptMinTrackSize.y = 50;
		  break;
		
		case WM_SIZE :
			// positions the splitter
			if (wParam != SIZE_MAXIMIZED)
			{
				int	nWidth = LOWORD(lParam);  // width of client area 
				int	nHeight = HIWORD(lParam); // height of client area 
				int	nTreeSize;
				if (g_fSplitHorizontal)
				{
					nTreeSize = (int)(nHeight*g_dblSplitFraction);
					MoveWindow(g_hTreeView, 0, 0,					nWidth, nTreeSize-1, TRUE);
					MoveWindow(g_hTextEdit, 0, nTreeSize+2,	nWidth, nHeight - nTreeSize, TRUE);
				}
				else
				{
					nTreeSize = (int)(nWidth*g_dblSplitFraction);
					MoveWindow(g_hTreeView, 0,					0,	nTreeSize-2,		nHeight, TRUE);
					MoveWindow(g_hTextEdit, nTreeSize+2,	0, nWidth-nTreeSize, nHeight, TRUE);
				}

			}
			break;

		case WM_NOTIFY :
			// child windows want to talk to mama
			switch (((NMHDR *)lParam)->code)
			{
				case TVN_GETDISPINFO :
				{
					TV_DISPINFO * pdi = (TV_DISPINFO *)lParam;
					BOOL fIsExpanded, fIsRoot; int cChildren;
					GetTreeItemInfo(g_hTreeView, pdi->item.hItem, 
										&fIsExpanded, &cChildren, &fIsRoot);
					
					int Image = 2;
					if (fIsRoot)
					{
						if (fIsExpanded)
							Image = 4;
						else
							Image = 3;
					}
					else
						if (cChildren == 0)
							Image = 2;
						else
							if (fIsExpanded)
								Image = 1;
							else
								Image = 0;
					if (pdi->item.mask & TVIF_IMAGE)
						pdi->item.iImage			 = Image;
					if (pdi->item.mask & TVIF_SELECTEDIMAGE)
						pdi->item.iSelectedImage = Image;
					
				}
				break;
				case TVN_SELCHANGED :
				{
					NM_TREEVIEW * tv = (NM_TREEVIEW *)lParam;
					SendMessage(g_hTextEdit, WM_SETREDRAW, FALSE, 0);
					FillTextEdit(g_hTextEdit, (CLineList *)tv->itemNew.lParam);
					// this stuff avoids flicker					
					SendMessage(g_hTextEdit, WM_SETREDRAW, TRUE, 0);
					InvalidateRect(g_hTextEdit, NULL, TRUE);
					UpdateWindow(g_hTextEdit);
				}
				break;
			}
			break;
		case WM_COMMAND :	
			// menu commands
			switch (LOWORD(wParam))
			{
				case ID_FILE_DNOPEN :
					{
						OPENFILENAME ofn;		memset(&ofn, 0, sizeof(ofn));
						ofn.lStructSize = sizeof(ofn);
						ofn.hwndOwner	= hwnd;
						ofn.hInstance	= g_hInstance;
						ofn.Flags		=  OFN_ALLOWMULTISELECT | OFN_FILEMUSTEXIST | OFN_EXPLORER | OFN_HIDEREADONLY;
						ofn.lpstrFile	= new char[1024];
						ofn.lpstrFile[0] = '\0';
						ofn.nMaxFile	= 1024;
						
						DWORD dwErr;
						if (!GetOpenFileName(&ofn))
							dwErr = CommDlgExtendedError();
						else
						{
							if (strlen(ofn.lpstrFile) < ofn.nFileOffset)
							{	// multiple selection
								char *pszWorkBuf = new char[255], *p, *pPath = ofn.lpstrFile;
								p = strchr(pPath, '\0') + 1;
								while (*p != '\0')
								{
									strcpy(pszWorkBuf, pPath);  strcat(pszWorkBuf, "\\");
									strcat(pszWorkBuf, p);
									LoadDemoNews(pszWorkBuf, g_hTreeView, TRUE);
									p = strchr(p, '\0') + 1;
								}
								delete [] pszWorkBuf;
							}
							else
								LoadDemoNews(ofn.lpstrFile, g_hTreeView, TRUE);
						}
						delete [] ofn.lpstrFile;
					}
					break;
				case ID_FILE_EXIT :
					PostMessage(hwnd, WM_CLOSE, 0, 0);
					break;
				case ID_EDIT_RTCOPY :
					SendMessage(g_hTextEdit, WM_COPY, 0, 0);
					break;
				case ID_EDIT_SELECTALL :
					SendMessage(g_hTextEdit, EM_SETSEL, 0, (LPARAM)-1);
					break;
				case ID_VIEW_SPLITHOZRIZONTAL :
					{
						HMENU hMenu = GetMenu(hwnd);
						g_fSplitHorizontal = TRUE;
						CheckMenuItem(hMenu, ID_VIEW_SPLITHOZRIZONTAL,	MF_CHECKED);
						CheckMenuItem(hMenu, ID_VIEW_SPLITVERTICAL,		MF_UNCHECKED);
						SendWMSize(hwnd);
					}
					break;
				case ID_VIEW_SPLITVERTICAL :
					{
						HMENU hMenu = GetMenu(hwnd);
						g_fSplitHorizontal = FALSE;
						CheckMenuItem(hMenu, ID_VIEW_SPLITVERTICAL,		MF_CHECKED);
						CheckMenuItem(hMenu, ID_VIEW_SPLITHOZRIZONTAL,	MF_UNCHECKED);
						SendWMSize(hwnd);
					}
					break;
				case ID_VIEW_FONT :
					{
						CHARFORMAT cf;
						cf.cbSize		=	sizeof(cf);
						SendMessage(g_hTextEdit, EM_GETCHARFORMAT, 0, (LPARAM)&cf);


						LOGFONT lf;
						memset(&lf, 0, sizeof(lf));
						lf.lfCharSet			=	cf.bCharSet;
						lf.lfPitchAndFamily	= cf.bPitchAndFamily;
						strcpy(lf.lfFaceName, cf.szFaceName);
						lf.lfHeight				=	cf.yHeight/-20;

						CHOOSEFONT chf;
						memset(&chf, 0, sizeof(chf));
						chf.lStructSize = sizeof(chf);
						chf.lpLogFont = &lf;
						chf.Flags	= CF_INITTOLOGFONTSTRUCT | CF_SCREENFONTS;
						ChooseFont(&chf);

						cf.dwMask		=	CFM_FACE | CFM_SIZE;
						cf.yHeight		=	180;
						cf.yOffset		=	0;
						cf.bCharSet		=	lf.lfCharSet;
						cf.bPitchAndFamily	= lf.lfPitchAndFamily;
						cf.yHeight		=	lf.lfHeight*-20;
						strcpy(cf.szFaceName, lf.lfFaceName);
						SendMessage(g_hTextEdit, EM_SETCHARFORMAT, 0, (LPARAM)&cf);
						g_FontPointSize = abs(lf.lfHeight);
					}
					break;
				case ID_HELP_ABOUT :
					MessageBox(hwnd, Ls(IDS_ABOUT), Ls(IDS_ABOUTTILE), 
							MB_APPLMODAL | MB_ICONINFORMATION | MB_OK);
					break;
			}
			break;
		case WM_LBUTTONDOWN :
			{
				int	Pos;
				if (g_fSplitHorizontal)
					Pos = HIWORD(lParam);
				else
					Pos = LOWORD(lParam);
				RECT	rc;
				GetClientRect(hwnd, &rc);
				int	nSoftSpot = (int)(rc.right * g_dblSplitFraction);
				fInSplitterResize = TRUE;
				SetCapture(hwnd);
				nLastMousePos = Pos;
				DrawResizeLine(hwnd, Pos, g_fSplitHorizontal);
			}
			break;
		case WM_MOUSEMOVE :
			if (fInSplitterResize)
			{
				RECT	rc;
				GetClientRect(hwnd, &rc);
				int	Pos;
				if (g_fSplitHorizontal)
					Pos = HIWORD(lParam);
				else
					Pos = LOWORD(lParam);
				if (Pos != nLastMousePos )
				{
					// erases the old resize line
					DrawResizeLine(hwnd, nLastMousePos, g_fSplitHorizontal);
					// draws a new one
					DrawResizeLine(hwnd, Pos, g_fSplitHorizontal);
					nLastMousePos = Pos;
				}
			}
			break;
			
		case WM_LBUTTONUP :
			if (fInSplitterResize)
			{
				fInSplitterResize = FALSE;
				ReleaseCapture();
				
				RECT	rc;
				GetClientRect(hwnd, &rc);
				int	Pos;
				if (g_fSplitHorizontal)
				{
					Pos = HIWORD(lParam);
					g_dblSplitFraction = (double)Pos / rc.bottom;
				}
				else
				{
					Pos = LOWORD(lParam);
					g_dblSplitFraction = (double)Pos / rc.right;
				}
				
				if (g_dblSplitFraction < 0.05) g_dblSplitFraction = 0.05;
				if (g_dblSplitFraction > 0.95) g_dblSplitFraction = 0.95;

				// erases the resize line
				DrawResizeLine(hwnd, nLastMousePos, g_fSplitHorizontal);

				SendWMSize(hwnd);
			}
			break;
		case WM_SETCURSOR :
			if ((HWND)wParam == hwnd && LOWORD(lParam) == HTCLIENT)
				SetCursor(LoadCursor(NULL, MAKEINTRESOURCE(g_fSplitHorizontal ? IDC_SIZENS : IDC_SIZEWE)));
			else
				return DefWindowProc(hwnd, uMsg, wParam, lParam);
			break;
		case WM_DESTROY :
			// Main window is being destroyed, tell windows we're quitting
			PostQuitMessage(0);
			break;
		default :
			return DefWindowProc(hwnd, uMsg, wParam, lParam);
	};
	return 0;
}

///////////////////////////////////////////////////////////////////////////////
//	==	CreateMainWindow
//		Creates the main window, places handle into g_hMainWindow
HWND CreateMainWindow()
{
	// registers the main window classes
	WNDCLASS	wc;
	wc.style				=	CS_DBLCLKS | CS_HREDRAW | CS_VREDRAW;
	wc.lpfnWndProc		=	MainWndProc;
	wc.cbClsExtra		=	0;
	wc.cbWndExtra		=	0;
	wc.hInstance		=	g_hInstance;
	wc.hIcon				=	LoadIcon(g_hInstance, MAKEINTRESOURCE(IDI_MAINWINICON));
	wc.hCursor			=	LoadCursor(NULL, MAKEINTRESOURCE(IDC_SIZEWE));
	wc.hbrBackground	=	(HBRUSH)(COLOR_WINDOW+1);
	wc.lpszMenuName	=	MAKEINTRESOURCE(IDR_MAINWINMENU);
	wc.lpszClassName	=	g_szMainWindowClass;

	RegisterClass(&wc);

	return			CreateWindow(	g_szMainWindowClass,
											Ls(IDS_MAINWINTITLE),
											WS_OVERLAPPEDWINDOW | WS_VISIBLE,
											CW_USEDEFAULT, CW_USEDEFAULT,
											CW_USEDEFAULT,	CW_USEDEFAULT,
											NULL, NULL,	g_hInstance, 0);
}



///////////////////////////////////////////////////////////////////////////////
//	== Main
//	-- gotta start somewhere
int WINAPI WinMain(	HINSTANCE hInstance,	
							HINSTANCE hPrevInstance,
							LPSTR lpCmdLine,
							int nCmdShow)
{
	g_hInstance = hInstance;


	// makes sure the rich edit library is loaded
	HINSTANCE hRichEdLib = LoadLibrary("RICHED32.DLL");
	if (hRichEdLib == NULL)
		return -1;

	// we'll be using the many common control stuff
	InitCommonControls();
	// build a image list for book icons
	g_BookImages = ImageList_LoadBitmap(g_hInstance, MAKEINTRESOURCE(IDB_BOOKS),
													16, 3, RGB(255, 0, 0));



	g_hMainWindow = CreateMainWindow();
	if (g_hMainWindow == NULL)		// unlikely!
		return -1;




	// The Main Message Loop
	//	Gets all messages for this thread and let window send
	//	them to the appropriate window
	MSG	msg;
	while (GetMessage(&msg, NULL, 0, 0))	// returns FALSE when user quits
	{
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}


	// free the book image list
	ImageList_Destroy(g_BookImages);


	// frees the rich edit library
	FreeLibrary(hRichEdLib);

	return 0;
}
