;
;
;                        32-bit FLAT mode GIF loader
;                       by TAD ( tad_uk@bigfoot.com )
;
;
;
; compile:
;       TASM32  /m gif32
;       TLINK32    gif32
;       STUBIT     gif32.exe            ; STUBIT from Michael Tippach's WDOSX
; or,
;       ML /c      gif32.asm            ; MASM
;       DLINK      gif32                ; DLINK  from Adam Seychell's DOS32
;       STUBIT     gif32.exe            ; STUBIT from Michael Tippach's WDOSX
;
;
; notes:
;       This was written to demonstrate how to load & decompress a
;       single image from a GIF file. It does NOT deal with the
;       various extension blocks (it should skip past these blocks)
;       and only decompresses the first TIB image.
;
;       It was written & tested using Michael Tippach's excellent
;       WDOSX dos extender and is intended for 32-bit PM FLAT MODE.
;
;       As it stands the routine is reasonably quick, but could be
;       quicker. The reason for a 'slower than it could be' depack
;       is due to the large '@@Output' section of code which allows
;       a small, rectangular part of a large GIF to be loaded.
;       It also allows a small GIF to be decompressed directly onto
;       the screen WITHOUT the need to use a temporary buffer to
;       depack the entire image. 
;
;                               enjoy..
;
;

TASM    = 1                     ; use: 0=MASM or 1=TASM

IF TASM
        LOCALS
ENDIF

        .386
        .MODEL FLAT
        .STACK 4096

VRAM_BASE       equ     000A0000h       ; Video-mem 32-bit linear base

MAX_GIF_WIDTH   equ     1024

;
; Name of the GIF
;
        .DATA
gif_filename    db      'TINYHUGI.GIF',0

        include gif.inc                 ; All the GIF structures & equates

;
; workspace structure 
;

gif STRUC
  ImageBits     db      ?               ; Bits-per-pixel in image (BPP)

  CodeBits      db      ?               ; = (ImageBits + 1) ... 12 bits
  FlushCode     dd      ?               ; = 1 << ImageBits
  EndCode       dd      ?               ; = FlushCode + 1
  TopSlot       dd      ?               ; = 2 << ImageBits
  Slot          dd      ?               ; next free slot
  FutureCode    dd      ?
  OldCode       dd      ?

  DestPnt       dd      ?
  DestLeft      dd      ?
  ClipX         dd      ?               ; \
  ClipY         dd      ?               ;
  ClipWidth     dd      ?               ;   used to clip output
  ClipHeight    dd      ?               ; 
  ClipPerLine   dd      ?               ; /

  BitRack       db      ?
  BitPos        db      ?

  BytePos       db      ?
  BytesLeft     db      ?
  bytes         db      256 dup (?)

  phraze        db      4097 dup (?)
  suffixes      db      4097 dup (?)
  prefixes      dw      4097 dup (?)

  palette       db      768 dup (?)

  LineBuffer    db      MAX_GIF_WIDTH dup (?)

  header        gifhdr  <>
  lsd           giflsd  <>
  tid           giftid  <>

gif ENDS

;
; all the GIF loader variables & LZW decompression buffers
;
        .DATA?
workspace       gif     <>


;
; Demonstrate that it actually works ;)
;
        .CODE
start32:

        ;; init the ES segment-selector (WDOSX) ;;

        push    DS
        pop     ES                              ; ES = DS

        ;; switch to 320 x 200 x 256 mode ;;

        mov     ax, 0013h
        int     10h

        ;; setup some GIF clipping values ;;

        lea     ebp, workspace
        mov     [ebp+gif.ClipX], 0              ; = + X pixels to skip
        mov     [ebp+gif.ClipY], -0             ; = - Y lines to skip
        mov     [ebp+gif.ClipWidth], 320        ; = clipping width
        mov     [ebp+gif.ClipHeight], 200       ; = clipping height
        mov     [ebp+gif.ClipPerLine], 320      ; = Dest. bytes per line

        ;; load the GIF file ;;

        lea     edx, gif_filename       ; [DS:EDX] --> filename
        mov     edi, VRAM_BASE          ; [ES:EDI] --> dest. buffer
        lea     ebp, workspace          ; [SS:EBP] --> workspace
        call    LoadGIF
        jc      Quit

        ;; set the palette ;;

        lea     esi, workspace.palette
        call    SetPalette

        ;; wait for key ;;

        mov     ah, 00h
        int     16h

        ;; text mode ;;

        mov     ax, 0003h
        int     10h

        ;; quit
Quit:
        mov     ax, 4C00h
        int     21h


;
; CF=LoadGIF[DS:EDX] in[ES:EDI] workspace[SS:EBP]
;
LoadGIF PROC
        mov     [ebp+gif.DestPnt], edi

        ;; open the file ;;

        mov     ax, 3D00h
        int     21h
        jc      @@OpenError
        xchg    eax, ebx

        ;; read GIF header & check ID signature ;;

        lea     edx, [ebp+gif.header]
        mov     ecx, SIZE gifhdr
        call    ReadFromFile
        jc      @@LoadError

        mov     eax, dword ptr [edx+gifhdr.Signature]
        shl     eax, 8
        cmp     eax, "FIG" SHL 8        ; remember it's intel low..high
        jnz     @@LoadError     

        ;; read LSD (Logical Screen Desciptor) ;;

        lea     edx, [ebp+gif.lsd]
        mov     ecx, SIZE giflsd
        call    ReadFromFile
        jc      @@LoadError

        ;; read GCT Global-Color-Table (palette) if any ;;

        mov     al, [edx+giflsd.LsdFlags]
        lea     edx, [ebp+gif.palette]
        call    ReadColorTable
        jc      @@LoadError

        ;; skip any extension blocks (AEX,CEX,PTE) ;;

@@ExtLoop:
        call    ReadFileByte
        jc      @@LoadError
        cmp     al, 02Ch
        jz      @@ImageID               ; TID ?
        cmp     al, 021h
        jnz     @@LoadError             ; not 021 xx extension block ?

        call    ReadFileByte
        jc      @@LoadError
        cmp     al, 0F9h
        jz      short @@SkipExt         ; 021 0F9 hex = GCE ?
        cmp     al, 0FEh
        jz      short @@SkipExt         ; 021 0FE hex = CEX ?
        cmp     al, 001h
        jz      short @@SkipExt         ; 021 001 hex = PTX ?
        cmp     al, 0FFh
        jz      short @@SkipExt         ; 021 0FF hex = AEX ?

        stc
        jmp     @@LoadError             ; else unknown extension

        ;; skip extension sub-block(s) ;;

@@SkipBlock:
        movzx   ecx, al
        lea     edx, [ebp+gif.bytes]
        call    ReadFromFile
        jc      @@LoadError
@@SkipExt:
        call    ReadFileByte
        jc      @@LoadError
        cmp     al, 0
        jnz     short @@SkipBlock

        jmp     short @@ExtLoop


        ;; read TID (The Image Descriptor) ;;

@@ImageID:
        lea     edx, [ebp+gif.tid+1]    ; the 02C hex Separator was
        mov     ecx, SIZE giftid - 1    ; read in the above code..
        call    ReadFromFile
        jc      @@LoadError

        movzx   eax, [ebp+gif.tid.ImageWidth]
        mov     [ebp+gif.DestLeft], eax

        ;; read LCT Local-Color-Table (if any) ;;

        mov     al, [ebp+gif.tid.TidFlags]
        lea     edx, [ebp+gif.palette]
        call    ReadColorTable
        jc      @@LoadError

        ;; now finally read & depack the GIF LZW image ;;

        call    ReadFileByte
        jc      @@LoadError
        mov     [ebp+gif.ImageBits], al
        xchg    eax, ecx

        mov     [ebp+gif.BitPos], 8             ; BitPos > 7 !!

        sub     eax, eax
        mov     [ebp+gif.FutureCode], eax       ; FutureCode = 0
        mov     [ebp+gif.OldCode], eax          ; OldCode = 0
        mov     [ebp+gif.BytesLeft], al         ; BytesLeft, 0
        inc     eax
        shl     eax, cl
        mov     [ebp+gif.FlushCode], eax        ; FlushCode = 1 << ImageBits
        inc     cl
        mov     [ebp+gif.CodeBits], cl          ; CodeBits = ImageBits + 1
        lea     edx, [eax+eax]
        mov     [ebp+gif.TopSlot], edx          ; TopSlot = 1 << CodeBits
        inc     eax
        mov     [ebp+gif.EndCode], eax          ; EndCode = FlushCode + 1
        inc     eax
        mov     [ebp+gif.Slot], eax             ; Slot = FlushCode + 2

        ;; the main decompression loop ;;

        lea     edi, [ebp+gif.LineBuffer]

@@Depack:
        lea     esi, [ebp+gif.phraze+4097]      ; init stack as empty

        call    ReadGIFCode
        jc      @@LoadError                     ; Token = ReadGIFCode()
        mov     edx, eax                        ; Code = Token
        cmp     eax, [ebp+gif.EndCode]
        jz      @@Done                          ; Token = EndCode ?

        cmp     eax, [ebp+gif.FlushCode]
        jz      @@Flush                         ; Token = FlushCode ?

        cmp     edx, [ebp+gif.Slot]
        jb      short @@Reverse                 ; if Code >= Slot then ...
        mov     edx, [ebp+gif.OldCode]          ; Code = OldCode
        mov     ecx, [ebp+gif.FutureCode]       ;
        dec     esi
        mov     [esi], cl                       ; StackByte(FutureCode)

        ;; extract LZW phraze by following prefix chain and ;;
        ;; reverse the suffix bytes into the correct order. ;;

@@Reverse:
        cmp     edx, [ebp+gif.EndCode]
        jbe     short @@First                   ; while Code > EndCode ...
        mov     cl, [ebp+edx+gif.suffixes]
        dec     esi
        mov     [esi], cl                       ; StackByte( suffix[Code] )
        movzx   edx, [ebp+edx*2+gif.prefixes]   ; Code = prefix[Code]
        jmp     short @@Reverse                 ; wend

@@First:
        dec     esi
        mov     [esi], dl                       ; StackByte(Code)

        ;; construct a 1-byte longer phraze ;;

        mov     ecx, [ebp+gif.Slot]
        cmp     ecx, [ebp+gif.TopSlot]
        jae     short @@Full                    ; if Slot < TopSlot ...
        mov     [ebp+gif.FutureCode], edx       ; FutureCode = Code
        mov     [ebp+ecx+gif.suffixes], dl      ; suffix[Slot] = Code
        mov     edx, [ebp+gif.OldCode]          ; 
        mov     [ebp+ecx*2+gif.prefixes], dx    ; prefix[Slot] = OldCode
        inc     [ebp+gif.Slot]                  ; Slot ++
        mov     [ebp+gif.OldCode], eax          ; OldCode = Code
@@Full:

        ;; do we need to bump CodeBits ? ;;

        mov     ecx, [ebp+gif.Slot]
        cmp     ecx, [ebp+gif.TopSlot]
        jb      short @@Fits            ; if Slot >= TopSlot and ...
        cmp     [ebp+gif.CodeBits], 12
        jae     short @@Fits            ; CodeBits < 12 then ...
        shl     [ebp+gif.TopSlot], 1    ; TopSlot << 1
        inc     [ebp+gif.CodeBits]      ; CodeBits ++
@@Fits:

        ;; output the already reversed phraze ;;
        ;; (this deals with wrapping a phraze beyond the right-edge ;;
        ;;  and also with clipping/positioning of an image) ;;

@@Output:
        lea     edx, [ebp+gif.phraze+4097]
        sub     edx, esi                        ; length of phraze

@@Blap:
        mov     ecx, [ebp+gif.DestLeft]
        cmp     ecx, edx
        jbe     short @@Wrap
        mov     ecx, edx
@@Wrap:
        sub     edx, ecx
        sub     [ebp+gif.DestLeft], ecx
        rep     movsb
        jg      @@Depack                        ; not finished line ?

        movzx   ecx, [ebp+gif.tid.ImageWidth]
        mov     [ebp+gif.DestLeft], ecx         ; reload X counter

        mov     eax, [ebp+gif.ClipY]
        cmp     eax, [ebp+gif.ClipHeight]
        jae     short @@Clip                    ; beyond dest Y limits ?

        lea     esi, [ebp+gif.LineBuffer]       ; [DS:ESI] --> GIF line
        mov     edi, [ebp+gif.DestPnt]          ; [ES:EDI] --> dest line

        mov     eax, [ebp+gif.ClipWidth]
        cmp     ecx, eax
        jbe     short @@Right
        mov     ecx, eax                        ; Clip right
@@Right:
        mov     eax, [ebp+gif.ClipX]
        add     esi, eax
        sub     ecx, eax                        ; Clip left
        jle     short @@Clip
        push    esi
        rep     movsb                           ; copy line --> dest
        pop     esi
        mov     eax, [ebp+gif.ClipPerLine]
        add     [ebp+gif.DestPnt], eax
@@Clip:
        inc     [ebp+gif.ClipY]
        lea     edi, [ebp+gif.LineBuffer]
        test    edx, edx
        jnz     short @@Blap                    ; remainder ?

        jmp     @@Depack

        ;; flush LZW dictionary & reinit stuff ;;

@@Flush:
        mov     cl, [ebp+gif.ImageBits]
        inc     cl
        mov     eax, 1
        shl     eax, cl
        mov     [ebp+gif.CodeBits], cl  ; CodeBits = Image Bits + 1
        mov     [ebp+gif.TopSlot], eax  ; TopSlot = 1 << CodeBits
        mov     eax, [ebp+gif.EndCode]
        inc     eax
        mov     [ebp+gif.Slot], eax     ; Slot = EndCode + 1

        ;; fetch the next non-FlushCode token ;;

@@Reinit:
        call    ReadGifCode
        jc      @@LoadError
        cmp     eax, [ebp+gif.FlushCode]
        jz      short @@Reinit

        cmp     eax, [ebp+gif.EndCode]
        jz      @@Done                  ; Token = EndCode ?

        cmp     eax, [ebp+gif.Slot]
        jb      @@Beyond
        sub     eax, eax                ; if Token > Slot then Token = 0
@@Beyond:
        mov     [ebp+gif.OldCode], eax          ; OldCode = Token
        mov     [ebp+gif.FutureCode], eax       ; FutureCode = Token
        dec     esi
        mov     [esi], al                       ; OutputByte(Token)
        jmp     @@Output

        ;; the usual okay/fail stuff (CF=error) ;;
@@LoadError:
        stc
@@Done:
        pushfd
        mov     ah, 3Eh
        int     21h
        popfd
@@OpenError:
        ret
LoadGIF ENDP

;
; CF=Read (AL)=Byte from file(BX)
;
ReadFileByte PROC
        pushad
        lea     edx, [esp+01Ch]
        mov     ecx, 1
        mov     ah, 3Fh
        call    ReadFromFile
        popad
        ret
ReadFileByte ENDP

;
; CF=Read from file(BX) into[DS:EDX] length(ECX)
;
ReadFromFile PROC
        mov     ah, 3Fh
        int     21h
        jc      @@ReadError
        cmp     eax, ecx
@@ReadError:
        ret
ReadFromFile ENDP

;
; CF=Read GCT or LCT color table(AL) into[DS:EDX]
;
ReadColorTable PROC
        test    al, GCT_FLAG            ; = LCT_FLAG
        jz      short @@NoColorTable
        mov     cl, GCT_SIZE_MASK       ; = LCT_SIZE_MASK
        and     cl, al
        mov     eax, 6
        shl     eax, cl
        xchg    eax, ecx                ; gif_Bytes = 6 << GCT/LCT size
        call    ReadFromFile
        jc      short @@NoColorTable

        ;; convert 8-bit --> 6-bit RGB ;;
@@Convert:
        shr     byte ptr [edx], 2
        inc     edx
        loop    @@Convert
        clc
@@NoColorTable:
        ret
ReadColorTable ENDP

;
; CF=Read (EAX)=GIF LZW token from file(BX) GIF Bitstream
;
ReadGIFCode PROC
        mov     cl, [ebp+gif.BitPos]

        cmp     cl, 7
        jbe     short @@NotEmpty        ; if BitPos > 7 then ...
        call    ReadGIFByte
        jc      short @@Failed
        mov     [ebp+gif.BitRack], al   ; BitRack = ReadFileByte()
        mov     cl, 0                   ; BitPos = 0
@@NotEmpty:
        movzx   edx, [ebp+gif.BitRack]
        shr     edx, cl                 ; Token = BitRack >> BitPos
        neg     cl                      ; GotBits = -BitPos + ...
@@Build:
        add     cl, 8                   ; GotBits + 8
        cmp     cl, [ebp+gif.CodeBits]
        jae     short @@GotToken        ; is GotBits >= CodeBits ?
        sub     eax, eax                ; (have we got a complete token ?)
        call    ReadGIFByte
        jc      short @@Failed
        mov     [ebp+gif.BitRack], al   ; else BitRack = ReadFileByte()
        shl     eax, cl
        or      edx, eax                ; Token OR (BitRack << GotBits)
        jmp     short @@Build
@@GotToken:
        sub     cl, [ebp+gif.CodeBits]  ; LeftBits = GotBits - CodeBits
        neg     cl
        add     cl, 8
        mov     [ebp+gif.BitPos], cl    ; BitPos = 8 - LeftBits

        mov     eax, [ebp+gif.TopSlot]
        dec     eax
        and     eax, edx                ; = Token  AND ((1 << CodeBits) - 1)
@@Failed:
        ret
ReadGIFCode ENDP

;
; CF=Read (AL)=Byte from GIF file(BX) or from the 255 byte cache
;
ReadGIFByte PROC
        cmp     [ebp+gif.BytesLeft], 0
        jnz     short @@Cached          ; BytesLeft <> 0 ?
        call    ReadFileByte
        jc      short @@Failed
        mov     [ebp+gif.BytesLeft], al ; else BytesLeft = ReadFileByte()

        pushad
        lea     edx, [ebp+gif.bytes]
        movzx   ecx, [ebp+gif.BytesLeft]
        call    ReadFromFile            ; read TBI bitstream block
        popad
        jc      short @@Failed
        mov     [ebp+gif.BytePos], 0
@@Cached:
        push    edx
        movzx   edx, [ebp+gif.BytePos]
        mov     al, [ebp+edx+gif.bytes] ; (AL) = bitstream byte cache[n]
        pop     edx
        dec     [ebp+gif.BytesLeft]
        inc     [ebp+gif.BytePos]
@@Failed:
        ret
ReadGIFByte ENDP

;
; Set entire 256-color palette[DS:ESI]
;
SetPalette PROC
        mov     dx, 03C8h
        mov     al, 0
        out     dx, al
        inc     edx
        mov     ecx, 768
        rep     outsb
        ret
SetPalette ENDP


        end     start32
