;;; dict.el --- frontend to dict client
     
;; Copyright (C) 1998 Alexander Vorobiev
     
;; Author: Alexander Vorobiev <sparrow@sparrow.diamant.ru>
;; Maintainer: Alexander Vorobiev <sparrow@sparrow.diamant.ru>
;; Created: 24 Sep 1998
;; Version: 1.1
;; Keywords: dict
;; Requires: dict client

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.    
;;
;; This file is neither part of XEmacs nor GNU Emacs. 

;;; Commentary:
;;
;; Make sure that your dict client is properly installed. It can be
;; obtained from http://www.dict.org
;; Then you may setq several variables. These are default values:
;;
;; (setq dict-pathname "dict")
;; This should be pathname of your dict executable
;;
;; (setq dict-server-host nil)
;; Hostname or ip address of dictd server, if 'nil' dict client will be
;; called without -h option which means that it will try to connect to
;; dict.org servers
;;
;; (setq dict-server-port nil)
;; Tcp port where dictd listens
;;
;; (setq dict-db-name nil)
;; Database name. Default (nil) is to lookup in all databases. List of
;; available databases could be received using M-x dict-list-databases
;;
;; (setq dict-strategy nil)
;; Lookup strategy to use. List of available strategies could be received
;; using M-x dict-list-strategies
;;
;; Now you can lookup words using M-x dict-lookup-words. Default is to
;; lookup the word under cursor. You can also bind this function to some
;; key sequence, for example, C-c d
;;
;; (global-set-key [(control ?c) d] 'dict-lookup-words)
;;

;;; LCD Archive Entry:
;; dict|Alexander Vorobiev|sparrow@sparrow.diamant.ru|
;; frontend for dict client|
;; $Date: 1998/10/02 14:36:23 $|$Revision: 1.6 $||

;;; RCS $Id: dict.el,v 1.6 1998/10/02 14:36:23 root Exp $
;; The revision number does NOT coincide with the version number.

;;; Change log:
;; $Log: dict.el,v $
;; Revision 1.6  1998/10/02 14:36:23  root
;; Fixed a bug in dict-lookup-words-region: re-search-forward
;; emitted an error on regions with 'empty' endings.
;;
;; Revision 1.5  1998/10/02 12:21:51  root
;; Added new functions dict-lookup-words-region and
;;  dict-lookup-words-buffer
;; Buffer name changed to *dict* (Navindra Umanee)
;; Added (require 'cl) (Jim Diamond, Michael A. Patton)
;; Default value for dict-server-host changed to nil (Ben Hyde)
;; Added '*' at the beginning of some variables documentation
;;  strings to make them user settable (Michael A. Patton)
;; Version number changed to 1.1
;;
;; Revision 1.4  1998/10/01 09:13:22  root
;; The code was slightly rearranged to avoid double compilation,
;; added some comments and LCD Archive Entry.
;;
;; Revision 1.3  1998/09/30 10:48:33  root
;; Added some installation commentaries.
;;
;; Revision 1.2  1998/09/29 15:39:59  root
;; Added ability to call dict-lookup-words interactively using the word
;; under cursor as a default.
;;
;; Revision 1.1  1998/09/25 17:45:31  root
;; Initial revision
;;
;; 

;;; Code:

(require 'cl)

(defvar dict-pathname "dict"
  "*Pathname of dict executable")

;;; Variables correspondent to dict client parameters
;;
(defvar dict-server-host nil
  "*dictd server host (string); if nil, try to connect to dict.org")

(defvar dict-server-port nil
  "*dictd server port (string); if nil, use default port")

(defvar dict-db-name nil
  "*Dictionary database name - default is to browse all available databases.
Databases could be listed using M-x dict-list-databases")

(defvar dict-strategy nil
  "*Use specific matching strategy for dictionary lookup. 
Default is to search for exact matches. Which strategies are available 
depends on your dictd server. They could be listed using 
M-x dict-list-strategies")

;; User configurable variables end here.

(defvar dict-buffer-name "*dict*"
  "Name of the buffer to display dictionary lookup results.")

(defconst dict-options-alist
  '((dict-server-host    "-h" t)
    (dict-server-port    "-p" t)
    (dict-db-name        "-d" t)
    (dict-db-list        "-D" nil)
    (dict-strategy       "-s" t)
    (dict-strategy-list  "-S" nil)
    (dict-server-info    "-I" nil))
  "Dict client options. It is processed as an alist keyed by command name, 
third value in each row shows if correspondent option requires additional 
argument (which is the value of a variable named the same as the command).")

(defun dict-make-options-list (options)
  "Make list of options for dict"
  (let ((list))
    (dolist (cmd options (nreverse list))
      (let ((opt (assoc cmd dict-options-alist)))
	(if (not (caddr opt))       ; Option doesn't require additonal
	    (push (cadr opt) list)  ;  parameter
	  (when (symbol-value cmd)  ; Only if the additional parameter's value 
	    (push (cadr opt) list)  ;  is not nil
	    (push (symbol-value cmd) list)))))))

(defmacro dict-client-call (options &rest args)
  "Construct a body of dict- function using required options and arguments"
  `(with-output-to-temp-buffer dict-buffer-name
     (apply #'call-process (nconc (list dict-pathname nil dict-buffer-name nil)
				  (dict-make-options-list ,options)
				  ,@args))))

(defun dict-list-databases ()
  "List dictionary databases available on default server"
  (interactive)
  (dict-client-call '(dict-server-host dict-server-port dict-db-list) '()))

(defun dict-list-strategies ()
  "List dictionary lookup strategies available on default server"
  (interactive)
  (dict-client-call '(dict-server-host dict-server-port dict-strategy-list) '()))

(defun dict-server-information ()
  "Show information about dictd server"
  (interactive)
  (dict-client-call '(dict-server-host dict-server-port dict-server-info) '()))

(defun dict-lookup-words (&rest words)
  "Look up the words in a dictionary using DICT client-server protocol."
  (interactive 
   (let ((w (current-word)))
     (list (read-string
	    "Word to look up: "
	    (unless (string-equal w "")
	      w)))))
  (dict-client-call '(dict-server-host 
		      dict-server-port 
		      dict-db-name 
		      dict-strategy)
		     words)
  words)
 

(defun dict-lookup-words-region (start end)
  "Lookup all words in region"
  (interactive "r")
  (let ((region-words-list))
    (save-excursion
      (goto-char start)
      (while (< (point) end)
	(if (re-search-forward "[a-zA-Z]+" end t)
	    (push (match-string 0) region-words-list)
	  (goto-char end))))
    
    (apply #'dict-lookup-words 
	   (nreverse (delete-duplicates region-words-list
					:test #'string-equal)))))

(defun dict-lookup-words-buffer ()
  "Lookup all words in current buffer"
  (interactive)
  (dict-lookup-words-region (point-min) (point-max)))
	    
(provide 'dict)

;;; dict.el ends here


