// -----------------------------------------------------------------------------
//
//      Mimic Engine
//      Copyright (C) 1997-1999 by Maciej Sinilo
//
//      MODULE  : VL_MIXED.H - Mixed matrix/vector operations
//      CREATED : 24-03-99
//
// -----------------------------------------------------------------------------

#ifndef _VL_MIXED_H_
#define _VL_MIXED_H_



// --- PROTOTYPES --------------------------------------------------------------

// Row-order transforms (vector * matrix)
VL_INL vlPnt4 operator * (const vlPnt4&,      const vlMatrix4&);
VL_INL vlVec3 operator * (const vlVec3&,      const vlMatrix4&);
VL_INL vlPnt3 operator * (const vlPnt3&,      const vlMatrix4&);

// Column-order transforms (matrix * vector)
VL_INL vlPnt4 operator * (const vlMatrix4&, const vlPnt4&);
VL_INL vlVec3 operator * (const vlMatrix4&, const vlVec3&);
VL_INL vlPnt3 operator * (const vlMatrix4&, const vlPnt3&);

// Assignments
VL_INL vlPnt4& operator *= (vlPnt4&, const vlMatrix4&);
VL_INL vlVec3& operator *= (vlVec3&, const vlMatrix4&);
VL_INL vlPnt3& operator *= (vlPnt3&, const vlMatrix4&);




// ----------------------------------
//
//  Implementation
//
// ----------------------------------

// 4D point * 4D matrix
VL_INL vlPnt4 operator * (const vlPnt4& a, const vlMatrix4& b)
{
        return vlVec4(
                a[0]*b(0,0) + a[1]*b(1,0) + a[2]*b(2,0) + a[3]*b(3,0),
                a[0]*b(0,1) + a[1]*b(1,1) + a[2]*b(2,1) + a[3]*b(3,1),
                a[0]*b(0,2) + a[1]*b(1,2) + a[2]*b(2,2) + a[3]*b(3,2),
                a[0]*b(0,3) + a[1]*b(1,3) + a[2]*b(2,3) + a[3]*b(3,3));
}

// 3D vector * 3x3 submatrix (rotation only)
VL_INL vlVec3 operator * (const vlVec3& a, const vlMatrix4& b)
{
        return vlVec3(
                a[0]*b(0,0) + a[1]*b(1,0) + a[2]*b(2,0),
                a[0]*b(0,1) + a[1]*b(1,1) + a[2]*b(2,1),
                a[0]*b(0,2) + a[1]*b(1,2) + a[2]*b(2,2));
}

// 3D point * 3x3 submatrix + translation
VL_INL vlPnt3 operator * (const vlPnt3 &a, const vlMatrix4 &b)
{
        return vlVec3(
                a[0]*b(0,0) + a[1]*b(1,0) + a[2]*b(2,0) + b(3,0),
                a[0]*b(0,1) + a[1]*b(1,1) + a[2]*b(2,1) + b(3,1),
                a[0]*b(0,2) + a[1]*b(1,2) + a[2]*b(2,2) + b(3,2));
}

// 4D matrix * 4D point
VL_INL vlPnt4 operator * (const vlMatrix4& b, const vlPnt4& a)
{
        return vlVec4(
                a[0]*b(0,0) + a[1]*b(0,1) + a[2]*b(0,2) + a[3]*b(0,3),
                a[0]*b(1,0) + a[1]*b(1,1) + a[2]*b(1,2) + a[3]*b(1,3),
                a[0]*b(2,0) + a[1]*b(2,1) + a[2]*b(2,2) + a[3]*b(2,3),
                a[0]*b(3,0) + a[1]*b(3,1) + a[2]*b(3,2) + a[3]*b(3,3));
}

// 3x3 submatrix * 3D vector (rotation only)
VL_INL vlVec3 operator * (const vlMatrix4& b, const vlVec3& a)
{
        return vlVec3(
                a[0]*b(0,0) + a[1]*b(0,1) + a[2]*b(0,2),
                a[0]*b(1,0) + a[1]*b(1,1) + a[2]*b(1,2),
                a[0]*b(2,0) + a[1]*b(2,1) + a[2]*b(2,2));
}

// 3x3 submatrix * 3D point + translation
VL_INL vlPnt3 operator * (const vlMatrix4& b, const vlPnt3& a)
{
        return vlPnt3(
                a[0]*b(0,0) + a[1]*b(0,1) + a[2]*b(0,2) + b(0,3),
                a[0]*b(1,0) + a[1]*b(1,1) + a[2]*b(1,2) + b(1,3),
                a[0]*b(2,0) + a[1]*b(2,1) + a[2]*b(2,2) + b(2,3));
}


VL_INL vlPnt4& operator *= (vlPnt4& a, const vlMatrix4& b)
{
        float v0, v1, v2;

        v0 = a[0]*b(0,0) + a[1]*b(1,0) + a[2]*b(2,0) + a[3]*b(3,0);
        v1 = a[0]*b(0,1) + a[1]*b(1,1) + a[2]*b(2,1) + a[3]*b(3,1);
        v2 = a[0]*b(0,2) + a[1]*b(1,2) + a[2]*b(2,2) + a[3]*b(3,2);
        a[3] = a[0]*b(0,3) + a[1]*b(1,3) + a[2]*b(2,3) + a[3]*b(3,3);
        a[0] = v0;
        a[1] = v1;
        a[2] = v2;

        return a;
}

VL_INL vlVec3& operator *= (vlVec3& a, const vlMatrix4& b)
{
        float v0, v1;

        v0 = a[0]*b(0,0) + a[1]*b(1,0) + a[2]*b(2,0);
        v1 = a[0]*b(0,1) + a[1]*b(1,1) + a[2]*b(2,1);
        a[2] = a[0]*b(0,2) + a[1]*b(1,2) + a[2]*b(2,2);
        a[0] = v0;
        a[1] = v1;

        return a;
}

VL_INL vlPnt3& operator *= (vlPnt3& a, const vlMatrix4& b)
{
        float v0, v1;

        v0 = a[0]*b(0,0) + a[1]*b(1,0) + a[2]*b(2,0) + b(3,0);
        v1 = a[0]*b(0,1) + a[1]*b(1,1) + a[2]*b(2,1) + b(3,1);
        a[2] = a[0]*b(0,2) + a[1]*b(1,2) + a[2]*b(2,2) + b(3,2);
        a[0] = v0;
        a[1] = v1;

        return a;
}

// -----------------------------------------------------------------------------
//      VL_MIXED.H - HEADER END
// -----------------------------------------------------------------------------
#endif  // !_VL_MIXED_H_
