// -----------------------------------------------------------------------------
//
//      Mimic Engine
//      Copyright (C) 1997-1999 by Maciej Sinilo
//
//      MODULE  : VL_QUAT.H - Quaternions
//      CREATED : 24-03-99
//
// -----------------------------------------------------------------------------

#include <iomanip.h>

#include "vl_mat4.h"
#include "vl_quat.h"



// --- CLASS BODY --------------------------------------------------------------


// Convert quaternion to matrix
vlMatrix4 vlQuaternion::ToMatrix() const
{
        vlMatrix4 m;
        vl_real x(v[0]), y(v[1]), z(v[2]),
              x2(x*VL_TWO), y2(y*VL_TWO), z2(z*VL_TWO), w2(w*VL_TWO),
              xx2(x*x2), yy2(y*y2), zz2(z*z2), ww2(w*w2),
              xy2(x*y2), xz2(x*z2), xw2(x*w2),
              yz2(y*z2), yw2(y*w2),
              zw2(z*w2);

#ifdef VL_ROW_ORIENT
        m[0] = vlVec4(VL_ONE-yy2-zz2, xy2+zw2, xz2-yw2);
        m[1] = vlVec4(xy2-zw2, VL_ONE-xx2-zz2, yz2+xw2);
        m[2] = vlVec4(xz2+yw2, yz2-xw2, VL_ONE-xx2-yy2);
#else
        m[0] = vlVec4(VL_ONE-yy2-zz2, xy2-zw2, xz2+yw2);
        m[1] = vlVec4(xy2+zw2, VL_ONE-xx2-zz2, yz2-xw2);
        m[2] = vlVec4(xz2-yw2, yz2+xw2, VL_ONE-xx2-yy2);
#endif
        return m;
}

// -----------------------------------------------------------------------------
// Slerp - spherical interpolation between two quaternions
// INPUT:  a, b - quaternions to interpolate between (a -> b)
//         t    - time parameter (0.0 - 1.0)
// OUTPUT: interpolated quaternion for this frame
vlQuaternion Slerp(const vlQuaternion& a, const vlQuaternion& b, const vl_real t)
{
        vlQuaternion result;
        vl_real        cosOm, sinOm, omega;
        vl_real        sclA, sclB;

        cosOm = a.v * b.v + a.w * b.w;

        if (cosOm + VL_ONE > VL_EPSILON)
        {
                if ((VL_ONE - cosOm) > VL_EPSILON)
                {
                        omega = (vl_real)acos(cosOm);
                        sinOm = sin(omega);

                        sclA = sin((VL_ONE - t) * omega) / sinOm;
                        sclB = sin(t * omega) / sinOm;
                }
                else
                {
                        // Very close keys (standard linear interpolation)
                        sclA = VL_ONE - t;
                        sclB = t;
                }

                result.v = (a.v * sclA) + (b.v * sclB);
                result.w = (a.w * sclA) + (b.w * sclB);
        }
        else
        {
                // Swap
                result.v[0] = -a.v[1];
                result.v[1] =  a.v[0];
                result.v[2] = -a.w;
                result.w    =  a.v[2];

                sclA = sin((VL_ONE - t) * VL_PI * VL_HALF);
                sclB = sin(t * VL_PI * VL_HALF);

                result.v = (sclA * a.v) + (sclB * b.v);
        }

        return result;
}

// -----------------------------------------------------------------------------

ostream& operator << (ostream& s, const vlQuaternion& q)
{
        int w = s.width();

        s << '[' << setw(w) << q.v << " " << setw(w) << q.w << setw(w) << ']';
        return s;
}

// -----------------------------------------------------------------------------
//      VL_QUAT.CPP - MODULE END
// -----------------------------------------------------------------------------
