#ifndef _VL_VEC4_H_
#define _VL_VEC4_H_
// -----------------------------------------------------------------------------
//
//      Mimic Engine
//      Copyright (C) 1997-1999 by Maciej Sinilo
//
//      MODULE  : VL_VEC4.H - 4D Vector/Point
//      CREATED : 24-03-99
//
// -----------------------------------------------------------------------------

#include "vl_main.h"
#include "vl_vec3.h"

// --- vlBase4 DEFINITION ------------------------------------------------------
//
// Base class for 4D points/vectors. User don't have right to create
// instances of vlBase4 class, so all constructors are protected.
// Class incorporates REAL homogeneous stuff - it performs all
// calculations on all components.
//
class vlBase4
{
public:

        // Set coordinates
        vlBase4& operator () (const vl_real x, const vl_real y, const vl_real z,
                const vl_real w);
        vlBase4& Set(const vl_real x, const vl_real y, const vl_real z,
                const vl_real w);

        // Get/set element (coordinate)
        vl_real& operator [] (const int index);
        const vl_real& operator [] (const int index) const;

        // Conversion
        operator vl_real* ()            { return (vl_real *)p;        }
        operator const vl_real*() const { return (const vl_real*)p;   }

        // Get/set element (coordinate)
        // Other fashion, more convenient when using pointers
        vl_real&            rX()             { return p[0]; }
        const vl_real&      rX() const       { return p[0]; }
        vl_real&            rY()             { return p[1]; }
        const vl_real&      rY() const       { return p[1]; }
        vl_real&            rZ()             { return p[2]; }
        const vl_real&      rZ() const       { return p[2]; }
        vl_real&            rW()             { return p[3]; }
        const vl_real&      rW() const       { return p[3]; }

        // Operations
        void MakeZero();
        vl_real* GetPtr()                 { return (vl_real*)p;       }
        const vl_real* GetReadPtr() const { return (const vl_real*)p; }

        void FromArray(const vl_real* pArray);

protected:

        vlBase4();
        vlBase4(const vl_real x, const vl_real y, const vl_real z,
                const vl_real w);
        vlBase4(const vlBase4& other);
        vlBase4(const vl_real* pArray);

        vlBase4& operator = (const vlBase4& other);

        vl_real   p[4];                   // Coordinates
};
// -------------------------------------------------
//
//  Base operators declared outside the class
//
// -------------------------------------------------
// Comparisons
VL_INL bool operator > (const vlBase4&, const vlBase4&);
VL_INL bool operator < (const vlBase4&, const vlBase4&);
VL_INL bool operator >= (const vlBase4&, const vlBase4&);
VL_INL bool operator <= (const vlBase4&, const vlBase4&);
VL_INL bool operator == (const vlBase4&, const vlBase4&);
VL_INL bool operator != (const vlBase4&, const vlBase4&);
// Return true if two vectors are ALMOST equal (== operator return true only
// when vectors are EXACTLY equal). The difference between each coordinate
// should be less than tol.
VL_INL bool AlmostEqual(const vlBase4&, const vlBase4&, vl_real tol);

// Streams
ostream& operator << (ostream& s, const vlBase4& a);


// --- vlPnt4 DEFINITION -------------------------------------------------------

class vlPnt4 : public vlBase4
{
public:
        vlPnt4();
        vlPnt4(const vl_real x, const vl_real y, const vl_real z,
                const vl_real w = VL_ONE);
        vlPnt4(const vlPnt3& p, const vl_real w = VL_ONE);
        vlPnt4(const vl_real* pArray);

        vlPnt4& operator += (const vlVec4& v);
        vlPnt4& operator -= (const vlVec4& v);

        operator vlVec4();

// OPERATIONS
        // 4D -> 3D
        void Homogenize();
};
// -------------------------------------------------
//
//  Point operators declared outside the class
//
// -------------------------------------------------
VL_INL vlPnt4 operator + (const vlPnt4&, const vlVec4&);
VL_INL vlPnt4 operator + (const vlVec4&, const vlPnt4&);
VL_INL vlPnt4 operator - (const vlPnt4&, const vlVec4&);

// Distance between two points in 4D
VL_INL vl_real Dist(const vlPnt4& begin, const vlPnt4& end);
// Distance between two points in 4D
// (Use this one if possible because it's faster (no sqrt))
VL_INL vl_real Dist2(const vlPnt4& begin, const vlPnt4& end);






// --- vlVec4 DEFINITION -------------------------------------------------------

class vlVec4 : public vlBase4
{
public:

        vlVec4();
        vlVec4(const vl_real x, const vl_real y, const vl_real z,
                const vl_real w = VL_ZERO);
        vlVec4(const vlVec3 &v, const vl_real w = VL_ZERO);
        vlVec4(vlAxis axis);
        vlVec4(const vl_real* pArray);

        vlVec4& operator = (vlAxis axis);
        vlVec4& operator += (const vlVec4& v);
        vlVec4& operator -= (const vlVec4& v);

        vlVec4& operator *= (const vl_real s);
        vlVec4& operator /= (const vl_real s);

        // Type casting (to point)
        operator vlPnt4 ();

        // Negation (unary operator).
        // Declared as class member to avoid automatic compiler conversions
        vlVec4 operator - () const;

// OPERATIONS

        vl_real Mag() const;              // Magnitude
        vl_real Mag2() const;             // Squared magnitude (FASTER!)
        int Dominant() const;           // Dominant axis (index)

        vlVec4& Normalize(const vl_real value = VL_ONE);
};
// -------------------------------------------------
//
//  Vector operators declared outside the class
//
// -------------------------------------------------
//VL_INL vlVec4 operator - (const vlVec4&);      // Negation
VL_INL vlVec4 operator + (const vlVec4&, const vlVec4 &);
VL_INL vlVec4 operator - (const vlVec4&, const vlVec4 &);
// Make vector from two points
VL_INL vlVec4 operator - (const vlPnt4& end, const vlPnt4& begin);
// Scale vector by scalar
VL_INL vlVec4 operator * (const vlVec4&, const vl_real);
VL_INL vlVec4 operator * (const vl_real, const vlVec4&);
VL_INL vlVec4 operator / (const vlVec4&, const vl_real);
// Dot product of two vectors
VL_INL vl_real operator * (const vlVec4&, const vlVec4&);
// Cross product (three vectors)
// I prefer clear function than some fancy operators... How am I supposed
// to know that ^ for example stands for cross product?
// [Defined in VL_VEC4.CPP]
vlVec4 Cross(const vlVec4&, const vlVec4&, const vlVec4 &);
// Component mul/div Result vector:
//  result[0] = a[0] *(/) b[0]
//  result[1] = a[1] *(/) b[1]
//  result[2] = a[2] *(/) b[2]
//  result[3] = a[3] *(/) b[3]
// Convenient when using vectors as colors for example.
VL_INL vlVec4 BlockMul(const vlVec4& a, const vlVec4& b);
VL_INL vlVec4 BlockDiv(const vlVec4& a, const vlVec4& b);






// --- vlBase4 IMPLEMENTATION --------------------------------------------------

VL_INL vlBase4::vlBase4() { }
VL_INL vlBase4::vlBase4(const vl_real x, const vl_real y, const vl_real z, const vl_real w) { p[0]=x; p[1]=y; p[2]=z; p[3]=w; }
VL_INL vlBase4::vlBase4(const vlBase4& other) { p[0]=other[0]; p[1]=other[1]; p[2]=other[2]; p[3]=other[3]; }
VL_INL vlBase4::vlBase4(const vl_real* pArray) { FromArray(pArray); }
VL_INL vlBase4& vlBase4::operator = (const vlBase4& other) { p[0]=other[0]; p[1]=other[1]; p[2]=other[2]; p[3]=other[3]; return SELF; }
VL_INL vlBase4& vlBase4::operator () (const vl_real x, const vl_real y, const vl_real z,
        const vl_real w) { p[0] = x; p[1] = y; p[2] = z; p[3] = w; return SELF; }
VL_INL vlBase4& vlBase4::Set(const vl_real x, const vl_real y, const vl_real z,
        const vl_real w) { p[0] = x; p[1] = y; p[2] = z; p[3] = w; return SELF; }
VL_INL vl_real& vlBase4::operator [] (const int i) { return p[i]; }
VL_INL const vl_real& vlBase4::operator [] (const int i) const { return p[i]; }
VL_INL void vlBase4::MakeZero() { p[0] = VL_ZERO; p[1] = VL_ZERO; p[2] = VL_ZERO; p[3] = VL_ZERO; }
VL_INL void vlBase4::FromArray(const vl_real* pArray) { p[0] = pArray[0]; p[1] = pArray[1]; p[2] = pArray[2]; p[3] = pArray[3]; }

VL_INL bool operator > (const vlBase4& a, const vlBase4& b) { return ((a[0]>b[0]) && (a[1]>b[1]) && (a[2]>b[2]) && (a[3]>b[3])); }
VL_INL bool operator >= (const vlBase4& a, const vlBase4& b) { return ((a[0]>=b[0]) && (a[1]>=b[1]) && (a[2]>=b[2]) && (a[3]>=b[3])); }
VL_INL bool operator < (const vlBase4& a, const vlBase4& b) { return ((a[0]<b[0]) && (a[1]<b[1]) && (a[2]<b[2]) && (a[3]<b[3])); }
VL_INL bool operator <= (const vlBase4& a, const vlBase4& b) { return ((a[0]<=b[0]) && (a[1]<=b[1])) && (a[2]<=b[2] && (a[3]<=b[3])); }
VL_INL bool operator == (const vlBase4& a, const vlBase4& b) { return ((a[0]==b[0]) && (a[1]==b[1]) && (a[2]==b[2]) && (a[3]==b[3])); }
VL_INL bool operator != (const vlBase4& a, const vlBase4& b) { return ((a[0]!=b[0]) || (a[1]!=b[1]) || (a[2]!=b[2]) || (a[3]!=b[3])); }
VL_INL bool AlmostEqual(const vlVec4& a, const vlVec4& b, vl_real tol)
{
        return ((vlTolComp(a[0], b[0], tol) == 0) &&
                (vlTolComp(a[1], b[1], tol) == 0) &&
                (vlTolComp(a[2], b[2], tol) == 0) &&
                (vlTolComp(a[3], b[3], tol) == 0));
}





// --- vlPnt4 IMPLEMENTATION ---------------------------------------------------

VL_INL vlPnt4::vlPnt4() : vlBase4() { }
VL_INL vlPnt4::vlPnt4(const vl_real x, const vl_real y, const vl_real z, const vl_real w) : vlBase4(x, y, z, w) { }
VL_INL vlPnt4::vlPnt4(const vlPnt3& v, const vl_real w) : vlBase4(v[0], v[1], v[2], w) { }
VL_INL vlPnt4::vlPnt4(const vl_real* pArray) : vlBase4(pArray) { }
VL_INL vlPnt4::operator vlVec4 () { return vlVec4(p[0], p[1], p[2], p[3]); }
VL_INL vlPnt4& vlPnt4::operator += (const vlVec4& v) { p[0] += v[0]; p[1] += v[1]; p[2] += v[2]; p[3] += v[3]; return SELF; }
VL_INL vlPnt4& vlPnt4::operator -= (const vlVec4& v) { p[0] -= v[0]; p[1] -= v[1]; p[2] -= v[2]; p[3] += v[3]; return SELF; }
VL_INL void vlPnt4::Homogenize()
{
        if (p[3] == VL_ZERO)
                return;
        vl_real recip = VL_ONE / p[3];
        p[0] *= recip;
        p[1] *= recip;
//        p[2] *= recip;  // not really needed...
}

VL_INL vlPnt4 operator + (const vlPnt4& a, const vlVec4& b) { return vlPnt4(a[0]+b[0], a[1]+b[1], a[2]+b[2], a[3]+b[3]); }
VL_INL vlPnt4 operator + (const vlVec4& a, const vlPnt4& b) { return vlPnt4(a[0]+b[0], a[1]+b[1], a[2]+b[2], a[3]+b[3]); }
VL_INL vlPnt4 operator - (const vlPnt4& a, const vlVec4& b) { return vlPnt4(a[0]-b[0], a[1]-b[1], a[2]-b[2], a[3]-b[3]); }

VL_INL vl_real Dist(const vlPnt4& a, const vlPnt4& b) { return vlVec4(b-a).Mag(); }
VL_INL vl_real Dist2(const vlPnt4& a, const vlPnt4& b) { return vlVec4(b-a).Mag2(); }




// --- vlVec4 IMPLEMENTATION ---------------------------------------------------

VL_INL vlVec4::vlVec4() : vlBase4() { }
VL_INL vlVec4::vlVec4(const vl_real x, const vl_real y, const vl_real z, const vl_real w) : vlBase4(x, y, z, w) { }
VL_INL vlVec4::vlVec4(const vlVec3& v, const vl_real w) : vlBase4(v[0],v[1],v[2],w) { }
VL_INL vlVec4::vlVec4(vlAxis axis) { SELF = axis; }
VL_INL vlVec4::vlVec4(const vl_real* pArray) : vlBase4(pArray) { }

VL_INL vlVec4& vlVec4::operator = (vlAxis axis)
{
        if (axis == VL_AXIS_X)          Set(VL_ONE,  VL_ZERO, VL_ZERO, VL_ZERO);
        else if (axis == VL_AXIS_Y)     Set(VL_ZERO, VL_ONE,  VL_ZERO, VL_ZERO);
        else if (axis == VL_AXIS_Z)     Set(VL_ZERO, VL_ZERO, VL_ONE,  VL_ZERO);
        else                            Set(VL_ZERO, VL_ZERO, VL_ZERO, VL_ONE );

        return SELF;
}
VL_INL vlVec4& vlVec4::operator += (const vlVec4& v) { p[0] += v[0]; p[1] += v[1]; p[2] += v[2]; p[3] += v[3]; return SELF; }
VL_INL vlVec4& vlVec4::operator -= (const vlVec4& v) { p[0] -= v[0]; p[1] -= v[1]; p[2] -= v[2]; p[3] -= v[3]; return SELF; }
VL_INL vlVec4& vlVec4::operator *= (const vl_real s) { p[0] *= s; p[1] *= s; p[2] *= s; p[3] *= s; return SELF; }
// WARNING: divides are slow, use multiplying by reciprocal if possible!
VL_INL vlVec4& vlVec4::operator /= (const vl_real s) { p[0] /= s; p[1] /= s; p[2] /= s; p[3] /= s; return SELF; }
VL_INL vlVec4::operator vlPnt4 () { return vlPnt4(p[0], p[1], p[2], p[3]); }
VL_INL vlVec4 vlVec4::operator - () const { return vlVec4(-p[0], -p[1], -p[2], -p[3]); }

VL_INL vl_real vlVec4::Mag() const { return vlSqrt(vlSquare(p[0])+vlSquare(p[1])+vlSquare(p[2])+vlSquare(p[3])); }
VL_INL vl_real vlVec4::Mag2() const { return (vlSquare(p[0])+vlSquare(p[1])+vlSquare(p[2])+vlSquare(p[3])); }
VL_INL int vlVec4::Dominant() const
{
        int d = vlAbs(p[0]) < vlAbs(p[1]);
        if (d)
        {
                d = vlAbs(p[1]) > vlAbs(p[2]);
                if (d)
                        d = vlAbs(p[1]) > vlAbs(p[3]);
                else
                        d = (vlAbs(p[2]) > vlAbs(p[3]) ? 2 : 3);
        }
        else
        {
                d = (vlAbs(p[0]) > vlAbs(p[2]) ? 0 : 2);
                if (d)
                        d = (vlAbs(p[2]) > vlAbs(p[3]) ? 2 : 3);
                else
                        d = (vlAbs(p[0]) > vlAbs(p[3]) ? 0 : 3);
        }
        return d;
}
VL_INL vlVec4& vlVec4::Normalize(const vl_real val)
{
        vl_real mag = Mag();

        // How to handle this case? No idea... so return
        if (mag == VL_ZERO)
                return SELF;

        vl_real recip = val / mag;
        p[0] *= recip;
        p[1] *= recip;
        p[2] *= recip;
        p[3] *= recip;
        return SELF;
}

VL_INL vlVec4 operator + (const vlVec4& a, const vlVec4& b) { return vlVec4(a[0]+b[0], a[1]+b[1], a[2]+b[2], a[3]+b[3]); }
VL_INL vlVec4 operator - (const vlVec4& a, const vlVec4& b) { return vlVec4(a[0]-b[0], a[1]-b[1], a[2]-b[2], a[3]-b[3]); }
VL_INL vlVec4 operator - (const vlPnt4& a, const vlPnt4& b) { return vlVec4(a[0]-b[0], a[1]-b[1], a[2]-b[2], a[3]-b[3]); }
VL_INL vlVec4 operator * (const vlVec4& v, const vl_real s) { return vlVec4(v[0]*s, v[1]*s, v[2]*s, v[3]*s); }
VL_INL vlVec4 operator * (const vl_real s, const vlVec4& v) { return vlVec4(v[0]*s, v[1]*s, v[2]*s, v[3]*s); }
// WARNING: divides are slow, use multiplying by reciprocal if possible!
VL_INL vlVec4 operator / (const vlVec4& v, const vl_real s) { return vlVec4(v[0]/s, v[1]/s, v[2]/s, v[3]/s); }
VL_INL vl_real operator * (const vlVec4& a, const vlVec4& b) { return (a[0]*b[0] + a[1]*b[1] + a[2]*b[2] + a[3]*b[3]); }
VL_INL vlVec4 BlockMul(const vlVec4& a, const vlVec4& b) { return vlVec4(a[0]*b[0], a[1]*b[1], a[2]*b[2], a[3]*b[3]); }
VL_INL vlVec4 BlockDiv(const vlVec4& a, const vlVec4& b) { return vlVec4(a[0]/b[0], a[1]/b[1], a[2]/b[2], a[3]/b[3]); }

// -----------------------------------------------------------------------------
//      VL_VEC4.H - HEADER END
// -----------------------------------------------------------------------------
#endif  // !_VL_VEC4_H_

