#include "..\h\PartDispatcher.h"

#define PROJECT_VERSION "1.00"

bool firstLog=true;
FILE *logFile;

void errorExit(const char *s, ...)
{
	char dummy[512];
	va_list marker;

	va_start(marker, s);							// initialize the variable argument list marker
	_vsnprintf(dummy, 512, s, marker);		// write the message in our buffer (max. 512 chars)
	va_end(marker);								// and release the marker

	printf("Error: %s\n", dummy);
	exit(-1);				// and exit with error code
}

void verbose(const char *s, ...)
{
	va_list marker;

	if (firstLog)
	{
		firstLog=false;
		logFile=fopen("Demo.log", "w");
	} else
		logFile=fopen("Demo.log", "a");

	if (logFile!=NULL)
	{
		va_start(marker, s);
		vfprintf(logFile, s, marker);
		va_end(marker);
		fclose(logFile);
	}
}

PartDispatcher::PartDispatcher()
{
	scenes=(Scene**)calloc(MAX_SCENES, sizeof(Scene*));
	sCount=0;
	actualFirstPart=0;
}

PartDispatcher::~PartDispatcher()
{
	int i;

	verbose("destroying part dispatcher\n");

	for (i=0; i<sCount; i++)
		delete scenes[i];
	free(scenes);

	if (parts!=NULL)
	{
		for (i=0; parts[i]!=NULL; i++)
			delete parts[i];
		free(parts);
	}
}

void PartDispatcher::registerScene(Scene *scene)
{
	if (scenes==NULL)
		errorExit("scenes==NULL in PartDispatcher. This can't be happening!");

	if (sCount<MAX_SCENES)	// there is space left
	{
		scenes[sCount]=scene;
		for (int i=0; i<sCount; i++)
			if (strcmp(scenes[i]->name, scenes[sCount]->name)==0)
				errorExit("Duplicated scene name %s", scenes[sCount]->name);
		sCount++;
	} else
		errorExit("sCount exceeds 40 in PartDispatcher!");
}

void (*PartDispatcher::getFunctionByName(Scene *scene, char *name))(void *tthis, float time, void **p)
{
	if (scene==NULL)
		errorExit("getFunctionByName scene==NULL");

	if (scene->getFunctionByName(name)==NULL)
		errorExit("function %s not found", name);

	return scene->getFunctionByName(name);
}

void *PartDispatcher::getThisByName(char *name)
{
	Scene *sTemp=NULL;

	for (int i=0; i<sCount; i++)
		if (strcmp(scenes[i]->name, name)==0)
			sTemp=scenes[i];

	if (sTemp==NULL)
		errorExit("Scene %s not found", name);

	return sTemp;
}

float PartDispatcher::readFloat()
{
	double t=0.0;
	int s;
	if (s=fread(&t, 8, 1, in)!=1)
		errorExit("double != 8 bytes (%d read)\n", s);
	return (float)t;
}

int PartDispatcher::readInt()
{
	int t;
	int s;
	if (s=fread(&t, 4, 1, in)!=1)
		errorExit("int != 4 bytes (%d read)\n", s);
	return t;
}

byte PartDispatcher::readByte()
{
	byte t;
	int s;
	if (s=fread(&t, 1, 1, in)!=1)
		errorExit("byte != 1 byte (%d read)\n", s);
	return t;
}

bool PartDispatcher::readBool()
{
	byte t;
	int s;
	if (s=fread(&t, 1, 1, in)!=1)
		errorExit("byte != 1 byte (%d read)\n", s);
	switch(t)
	{
		case 0:	return false; break;
		case 1:	return true; break;
		default:	errorExit("unknown boolean values\n");
	}
	return false;
}

char* PartDispatcher::readString()
{
	char *t;
	char c=0;
	int i=0;
	int s;

	t=(char*)malloc(128);
	while ((c!=0x0a)&&(i<127))
	{
		if (s=fread(&c, 1, 1, in)!=1)
			errorExit("char != 1 byte (%d read)\n", s);
		if ((c!=0x0d)&&(c!=0x0a))
			t[i++]=c;
	}
	t[i]=0;		// end of string
	return t;
}

void PartDispatcher::readPart(int num)
{
	int i, j, k;
	int pCount, fCount;
	Function *fTemp;

	if (parts==NULL)
		errorExit("parts==NULL in PartDispatcher. This can't be happening!");

	parts[num]=new Part();
	strncpy(parts[num]->name, readString(), 127);
	verbose("    name %s\n", parts[num]->name);
	readInt();			// part color (ignored)
	parts[num]->position=readFloat();		// start time of the part
	verbose("    pos %f\n", parts[num]->position);
	parts[num]->start=readFloat();			// additional start time (this is added to the actual time, used for reversed speed)
	verbose("    start %f\n", parts[num]->start);
	parts[num]->length=readFloat();
	verbose("    length %f\n", parts[num]->length);
	parts[num]->track=readByte();
	verbose("    track %d\n", parts[num]->track);
	readBool();			// selected (ignored)
	if (!readBool())
		errorExit("inactive part found");
	parts[num]->speed=readFloat();		// speed of the part (negative=reversed)
	verbose("    speed %f\n", parts[num]->speed);

	fCount=readByte();				// count of functions
	parts[num]->functions=(Function**)calloc(fCount+1, sizeof(Function*));

	for (i=0; i<fCount; i++)
	{
		fTemp=parts[num]->functions[i]=new Function();
		fTemp->tthis=getThisByName(readString());
		strncpy(fTemp->name, readString(), 127);
		fTemp->f=getFunctionByName((Scene*)(fTemp->tthis), fTemp->name);

		pCount=readByte();		// count of parameters
		fTemp->parameters=(Parameter**)calloc(pCount+1, sizeof(Parameter*));
		for (j=0; j<pCount; j++)
		{
			fTemp->parameters[j]=new Parameter();
			verbose("      parameter\n");
			strncpy(fTemp->parameters[j]->name, readString(), 127);		// copy the parameter name
			verbose("        name %s\n", fTemp->parameters[j]->name);
			fTemp->parameters[j]->type=readByte();		// and set the type
			verbose("        type %d\n", fTemp->parameters[j]->type);
			if ((fTemp->parameters[j]->type<0)||(fTemp->parameters[j]->type>4))
				errorExit("unknown parameter type at %s.%s", fTemp->name, fTemp->parameters[j]->name);

			fTemp->parameters[j]->pConst=readFloat();
			verbose("        const %f\n", fTemp->parameters[j]->pConst);
			fTemp->parameters[j]->pSinPeriod=readFloat();
			verbose("        sin period %f\n", fTemp->parameters[j]->pSinPeriod);
			fTemp->parameters[j]->pSinAmplitude=readFloat();
			verbose("        sin amplitude %f\n", fTemp->parameters[j]->pSinAmplitude);
			fTemp->parameters[j]->pSinOffset=readFloat();
			verbose("        sin offset %f\n", fTemp->parameters[j]->pSinOffset);
			fTemp->parameters[j]->pSinPhase=readFloat();
			verbose("        sin phase %f\n", fTemp->parameters[j]->pSinPhase);

			fTemp->parameters[j]->dotCount=readByte();		// count of dots
			verbose("        %d dots\n", fTemp->parameters[j]->dotCount);
			fTemp->parameters[j]->x=(float*)calloc(fTemp->parameters[j]->dotCount, sizeof(float));
			fTemp->parameters[j]->y=(float*)calloc(fTemp->parameters[j]->dotCount, sizeof(float));
			for (k=0; k<fTemp->parameters[j]->dotCount; k++)
			{
				fTemp->parameters[j]->x[k]=readFloat();
				fTemp->parameters[j]->y[k]=readFloat();
				verbose("          %f %f\n", fTemp->parameters[j]->x[k], fTemp->parameters[j]->y[k]);
			}
			fTemp->parameters[j]->min=readFloat();
			verbose("        min %f\n", fTemp->parameters[j]->min);
			fTemp->parameters[j]->max=readFloat();
			verbose("        max %f\n", fTemp->parameters[j]->max);
			strcpy(fTemp->parameters[j]->pString, readString());
			verbose("        string %s\n", fTemp->parameters[j]->pString);
		}
		fTemp->parameters[pCount]=NULL;
	}
	parts[num]->functions[fCount]=NULL;
}

void PartDispatcher::loadProject(char *filename)
{
	int pCount;

	in=fopen(filename, "rb");
	if (in==NULL)
		errorExit("Project file %s not found!", filename);
	if (strcmp(readString(), PROJECT_VERSION)!=0)
		errorExit("Wrong file version. Must be %s\n", PROJECT_VERSION);

	verbose("start reading %s\n", filename);
	verbose("executable file name %s\n", readString());		// executable file name (ignored)
	startTime=readFloat();
	verbose("start time %f\n", startTime);
	verbose("functions file name %s\n", readString());		// functions file name (ignored)
	verbose("left time %f\n", readFloat());		// left time (ignored)
	totalTime=readFloat();								// total demo time
	verbose("total time %f\n", totalTime);
	verbose("zoom factor %f\n", readFloat());		// zoom factor (ignored)
	char *s=readString();
	if (strlen(s)>1)
		verbose("sound file name %s\n", s);
	else
		verbose("no sound file defined\n");

	verbose("ignored fields successfully read\n");

	int count=readInt();	// number of ticks (needed to proceed)
	verbose("reading %d ticks\n", count);
	for (int i=0; i<count; i++)
		verbose("   %f\n", readFloat());		// tick time (ignored)

	verbose("ticks successfully read\n");

	pCount=readInt();	// number of parts
	if (pCount>=MAX_PARTS)
		errorExit("pCount exceeds maximum of %d in PartDispatcher!", MAX_PARTS);
	parts=(Part**)calloc(pCount+1, sizeof(Part*));
	verbose("reading %d parts\n", pCount);
	for (i=0; i<pCount; i++)
	{
		verbose("  Part %d\n  =======\n", i+1);
		readPart(i);		// read the part
	}
	parts[pCount]=NULL;

	fclose(in);
	verbose("project file successfully parsed\n");
}

void PartDispatcher::execute(float time)
{
	int i, j, k;
	int calledParts[16];			// maximum number of 16 parts simultaneously
	int dummy;

	if (time>=totalTime)
	{
		printf("STOP\n");
		exit(0);
	}

	printf("time %1.2f   ", time);

	// determine, which is the first actual part
	i=actualFirstPart;
	while ((parts[i]!=NULL)&&(parts[i]->position+parts[i]->length<=time))
		i++;
	actualFirstPart=i;

	k=0;			// determine, which parts must be executed
	for (i=actualFirstPart; (parts[i]!=NULL)&&(parts[i]->position<=time); i++)
		if (parts[i]->position+parts[i]->length>time)	// part must be executed
			calledParts[k++]=i;

	// sort the parts depending on their track
	for (i=k-1; i>0; i--)
		for (j=0; j<i; j++)
			if (parts[calledParts[j]]->track>parts[calledParts[j+1]]->track)
			{
				dummy=calledParts[j];
				calledParts[j]=calledParts[j+1];
				calledParts[j+1]=dummy;
			}

	for (i=0; i<k; i++)
		parts[calledParts[i]]->execute(time);
}
