;
;Initialise hardware, alloc ram, load data etc.

align 32
proc    startup n

        ;Set interrupt flag for pmode
        sti
        ;Initialise fpu
        finit

        ;Do we give help?
        mov     al, 'h'
        call    checkswitchnc
        jnc     @@help

        ;Retrace sync?
        mov     al, 'v'
        call    checkswitchnc
        sbb     [sync], 0

        ;Parallax?
        mov     al, 'p'
        call    checkswitchnc
        sbb     [parallax], 0

        ;Turtle mode?
        mov     al, 't'
        call    checkswitchnc
        jc      @@no_turtle

        ;Halve ang_vel (scale be 2^-1 which is 1/2)
        fld1
        fchs
        fld     [ang_vel]
        fscale
        fstp    [ang_vel]
        fpop    1

@@no_turtle:
        ;Alloc the filebuffer, etc.
        call    init_file

        ;Alloc and generate sincos tables
        call    f_gen_sincos
        jc      @@error

        ;Setup the video mode
        call    init_video
        jc      @@error

        ;Load the shapes and maps
        call    load_data
        jc      @@error

        ;Setup spheremapping coords
        call    init_spheremap
        jc      @@error

        ;Alloc ram for sorting, setup morphing, etc.
        call    init_3d
        jc      @@error

@@done:
        ret

@@error:
        ;Something went wrong
        ;Make sure we clean up after ourselves
        call    reset_video
        call    reset_file
        mode3
        msg     "An error occured"
        endl
        msg     "Sorry"
        endl
        jmp     _exit

@@help:
        ;The help screen
        endl
        cmsg    "3D Example"
        endl
        cmsg    "Copyright (c) 1997 Andrew Harvey, c9607776@engmail.newcastle.edu.au"
        endl
        endl
        cmsg    "- 3D morphed animation -"
        endl
        cmsg    "- Linear texture mapping -"
        endl
        cmsg    "- Motion blurred spherical mapping -"
        endl
        cmsg    "- pmode courtesy of Tran, a.k.a. Tomas Pytel -"
        endl
        cmsg    "- Texture mapping docs courtesy of Chris Hecker -"
        endl
        cmsg    "- Spherical mapping doc courtesy of John De Goes -"
        endl
        endl
        msg     "Command-line switches:"
        endl
        msg     "    -h : This help screen."
        endl
        msg     "    -p : Parallax spheremapping."
        endl
        msg     "    -t : Turtle mode - half speed animation."
        endl
        msg     "    -v : Vertical retrace syncing (Slow, but less shearing)."
        endl
        msg     "    -s : SVGA mode. Requires a VBE2 640x480 bios.  Minimal error checking."
        endl
        endl
        cmsg    "Please credit me if you use this"
        endl

        jmp     _exit

endp

;
;Alloc ram for sorting, setup morphing, etc.

align 32
proc    init_3d n

        ;Get the number of points & polys in the shape file
        mov     esi, [shape_mem]
        mov     eax, [esi]
        mov     ebx, [esi + 4]
        mov     [shape_numpts], eax
        mov     [shape_numpolys], ebx

        ;Alloc ram for morphed points
        mov     ecx, [shape_numpts]
        shl     ecx, 4
        mov     eax, [_himembase]
        add     [_himembase], ecx
        mov     [temp_pts], eax

        ;Alloc ram for rotated points (needed during transformation)
        mov     ecx, [shape_numpts]
        shl     ecx, 4
        mov     eax, [_himembase]
        add     [_himembase], ecx
        mov     [rot_pts], eax

        ;Alloc ram for rotated & projected points
        ;(needed during transformation)
        mov     ecx, [shape_numpts]
        shl     ecx, 4
        mov     eax, [_himembase]
        add     [_himembase], ecx
        mov     [proj_pts], eax

        ;Alloc ram for sorting tree
        mov     ecx, [shape_numpolys]
        imul    ecx, size btree_node
        mov     eax, [_himembase]
        add     [_himembase], ecx
        mov     [btree_start], eax

        ;Alloc ram for morphing data
        mov     ecx, [shape_numpts]
        shl     ecx, 4
        mov     [morph_size], ecx
        imul    ecx, [num_shape_frames]
        mov     eax, [_himembase]
        add     [_himembase], ecx
        mov     [morph_mem], eax

        ;Setup morphing data for each frame
        mov     ebp, [num_shape_frames]
        xor     ebx, ebx
        mov     esi, 1  
        mov     edi, [morph_mem]

@@make_morph:
        push    ebx esi edi
        mov     ecx, [shape_numpts]
        mov     ebx, [ebx*4 + o shape_pts]
        mov     esi, [esi*4 + o shape_pts]
        call    setup_morph
        pop     edi esi ebx

        inc     ebx
        inc     esi
        cmp     ebp, 2
        jne     @@no_wrap

        xor     esi, esi

@@no_wrap:
        add     edi, [morph_size]
        dec     ebp
        jnz     @@make_morph

        mov     eax, [proj_pts]
        mov     ebx, [buffer_ptr]
        mov     [scan_pts], eax
        mov     [scan_dest], ebx

@@done:
        clc
        ret

@@error:
        stc
        ret

endp

;
;In:
;       ecx: number of points in frame
;       ebx: points to morph from
;       esi: points to morph to
;       edi: morphing data
;
;The linear interpolation of two points with parameter t is:
;       p = (1-t) * p1 + t * p2
;rearranging:
;       p = p1 + t * (p2 - p1)
;We can precalculate (p2 - p1) so morphing only requires one add & one mul.
;This is, incidentally, the vector equation for a line.

align 32
proc    setup_morph n

@@morph:
        ;(p2 - p1)
        fld     [d esi]
        fsub    [d ebx]
        fld     [d esi + 4]
        fsub    [d ebx + 4]
        fld     [d esi + 8]
        fsub    [d ebx + 8]
        fxch    st(2)
        fstp    [d edi]
        fstp    [d edi + 4]
        fstp    [d edi + 8]

        add     ebx, 16
        add     esi, 16
        add     edi, 16
        dec     ecx
        jnz     @@morph

        ret

endp

;
;Load the shapes and maps

align 32
proc    load_data n

        ;Load the maps
        call    load_tmaps
        jc      @@error

        ;Load the shapes
        call    load_shapes
        jc      @@error

@@done:
        clc
        ret

@@error:
        stc
        ret

endp

;
;Load the shapes

align 32
proc    load_shapes n

        ;Shapes are named shape1, shape2, shape3, ...
        ;Count how any there are by going until load shapen
        ;Each shape is one animation frame
        mov     edx, o shape_file
        mov     [b edx + 8], '1'

        xor     eax, eax
        ;There can be at most 9 frames
        mov     ecx, 9

@@count_files:
        call    openfile
        jc      @@got_count
        call    closefile
        jc      @@error

        inc     eax
        inc     [b edx + 8]
        dec     ecx
        jnz     @@count_files

@@got_count:
        test    eax, eax
        jz      @@error
        mov     [num_shape_frames], eax

        ;Get the filesize for the first frame
        mov     edx, o shape_file
        mov     [b edx + 8], '1'
        call    openfile
        jc      @@error
        call    filesize
        jc      @@error
        call    closefile
        jc      @@error
        mov     [shape_size], eax

        ;Alloc ram for all frames
        mov     ecx, [shape_size]
        imul    ecx, [num_shape_frames]
        mov     eax, [_himembase]
        add     [_himembase], ecx
        mov     [shape_mem], eax

        ;Load all the frames
        mov     ecx, [shape_size]
        mov     edx, o shape_file
        mov     [b edx + 8], '1'
        mov     edi, [shape_mem]
        mov     ebp, [num_shape_frames]

@@read_shapes:
        ;All frames must be the same size
        call    openfile
        jc      @@error
        call    filesize
        jc      @@error
        cmp     eax, ecx
        jne     @@error

        ;Load one frame
        push    edx
        mov     edx, edi
        call    readfile
        pop     edx
        jc      @@error
        inc     [b edx + 8]
        add     edi, ecx
        dec     ebp
        jnz     @@read_shapes

        ;Find pointers to each frame's points and polys
        mov     ecx, [num_shape_frames]
        mov     ebx, [shape_mem]
        mov     esi, o shape_pts
        mov     edi, o shape_polys

@@make_ptrs:
        lea     eax, [ebx + 8]
        mov     [esi], eax
        mov     eax, [ebx]
        shl     eax, 4
        lea     eax, [eax + ebx + 8]
        mov     [edi], eax

        add     ebx, [shape_size]
        add     esi, 4
        add     edi, 4
        dec     ecx
        jnz     @@make_ptrs

@@done:
        clc
        ret

@@error:
        call    closefile
        stc
        ret

endp

;
;Load the maps

align 32
proc    load_tmaps n

        ;Alloc ram for two texture maps, plus extra for alignment
if HIGH_COLOUR
        mov     ecx, 65536*2 * 3
else
        mov     ecx, 65536 * 3
endif
        mov     eax, [_himembase]
        add     [_himembase], ecx
        mov     [texture_mem], eax

        ;Align maps to 64k boundary
        ;They must be this way for spheremapper and texturemapper
if HIGH_COLOUR
        add     eax, 65536*2 - 1
        and     eax, not (65536*2 - 1)
        mov     [tex_ptr], eax
        add     eax, 65536*2
        mov     [sky_ptr], eax
else
        add     eax, 65535
        and     eax, not 65535
        mov     [tex_ptr], eax
        add     eax, 65536
        mov     [sky_ptr], eax
endif

if HIGH_COLOUR
        mov     edx, o tmap_file
        call    openfile
        jc      @@error
        mov     ecx, 256 * 256 * 3
        mov     edx, [_himembase]
        call    readfile
        jc      @@error
        call    closefile
        jc      @@error
        mov     esi, [_himembase]
        mov     edi, [tex_ptr]
        call    true_2_hi

        mov     edx, o sky_file
        call    openfile
        jc      @@error
        mov     ecx, 256 * 256 * 3
        mov     edx, [_himembase]
        call    readfile
        jc      @@error
        call    closefile
        jc      @@error
        mov     esi, [_himembase]
        mov     edi, [sky_ptr]
        call    true_2_hi
else
        ;Load texturemap for shape, must be 256x256
        mov     edx, o tmap_file
        mov     edi, [tex_ptr]
        mov     [pcx_palptr], o palette
        call    decompress_pcxfile
        jc      @@error

        ;Load texturemap for sky, must be 256x256
        mov     edx, o sky_file
        mov     edi, [sky_ptr]
        mov     [pcx_palptr], o palette
        call    decompress_pcxfile
        jc      @@error

        ;Set the palette, each map must have the same palette
        xor     eax, eax
        mov     ecx, 256
        mov     esi, o palette
        writepalette
endif

@@done:
        clc
        ret

@@error:
        stc
        ret

endp

;

if HIGH_COLOUR
align 32
proc    true_2_hi n

        mov     ecx, 65536

@@convert_pixel:
        xor     eax, eax
        mov     al, [esi + 2]
        shr     eax, 3
        mov     ebx, eax

        mov     al, [esi + 1]
        shr     eax, 2
        shl     eax, 5
        or      ebx, eax

        xor     eax, eax
        mov     al, [esi]
        shr     eax, 3
        shl     eax, 11
        or      ebx, eax

        mov     [edi], bx

        add     esi, 3
        add     edi, 2
        dec     ecx
        jnz     @@convert_pixel

        ret

endp
endif

;
