//**************************************************
// Texture Mapping Tutorial Demo Program
// by Tumblin / Bodies In Motion  (Terry Sznober)
//
// See TMAPTUT.TXT for tutorial
// See BIMINFO.TXT for info about Bodies In Motion
//**************************************************

// set your tab spacing to 2

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>

#include "fixed32.h"

char *screen=(char *)0x0a0000; // mode 13h screen

int ledge_x[200];  // destination x coordinate for left edge
int ledge_sx[200]; // source x coordinate for left edge
int ledge_sy[200]; // source y coordinate for left edge

int redge_x[200]; // destination x coordinate for right edge
int redge_sx[200]; // source x coordinate for right edge
int redge_sy[200]; // source y coordinate for right edge

int ytable[200]; // so u can do ytable[y]+x (fast) instead of y*320+x (slow)

// structure for a 3 sided texture mapped polygon (triangle)
typedef struct
{
	int sx1,sy1,sx2,sy2,sx3,sy3; // source texture coordinates (c-clockwise)
	int dx1,dy1,dx2,dy2,dx3,dy3; // distination screen coordinates (c-clockwise)
} TPolygonTYPE;

TPolygonTYPE poly;

char bitmap[128*128];
char dummy_buffer[64000]; // load the graphics into here

char palette[768];

// clipping area
int ClipTop=0;
int ClipBottom=199;
int ClipLeft=0;
int ClipRight=319;


//---------------------- function prototypes ------------------------
int main(void);
void InitYTable(void);
void SetUpScreen(void);
void SetPalette(char *palettebuffer);
void LoadGraphics(void);
void DemoLoop(void);
void TextureMap(TPolygonTYPE *poly);
void TScanEdge(int xd1,int yd1,int xd2,int yd2,
							 int xs1,int ys1,int xs2,int ys2);

//---------------------------------------------------------------------

void SetMode13h(void);
#pragma aux SetMode13h = \
		"mov eax,013h " \
		"int 10h" \
		modify [eax];

//---------------------------------------------------------------------

void SetMode03h(void);
#pragma aux SetMode03h = \
		"mov eax,003h " \
		"int 10h" \
		modify [eax];

//---------------------------------------------------------------------

int main(void)
{
	printf("\n\nTexture Mapping Tutorial demo program\n");
	printf("coded by Tumblin / Bodies In Motion  (Terry Sznober)\n");
	printf("\n\nPress any key to see my picture,\n");
	printf("then press any key again to continue.\n");
	getch();

	InitYTable();
	SetMode13h();
	LoadGraphics();
	getch();

	DemoLoop();

	SetMode03h();
	return(0);
}



//---------------------------------------------------------------------

void InitYTable(void)
{
	int i;
	for(i=0;i<200;i++)
	{
		ytable[i]=i*320;
	}
}

//------------------------------- palette functions ---------------
void SetPalette(char *palettebuffer)
{
	int i;

	for(i=0;i<256;i++)
	{
		outp(0x3c8,i);  // color number to set
		outp(0x3c9,palettebuffer[i*3]);   // red
		outp(0x3c9,palettebuffer[i*3+1]); // green
		outp(0x3c9,palettebuffer[i*3+2]); // blue
	}
}


//---------------------------------------------------------------------

void DemoLoop(void)
{
	// initialize the source texture coordinates
	// assume that bitmap is 64x64 pixels in size
	poly.sx1=0;  poly.sy1=0;  // top left corner
	poly.sx2=0;  poly.sy2=127; // bottom left corner
	poly.sx3=127; poly.sy3=127; // bottom right corner

	do
	{
		// initialize the screen coordinates of the polygon
		poly.dx1 = (rand()%360) - 20; // between -20 and +340 to show clipping
		poly.dy1 = (rand()%240) - 20; // between -20 and +220 to show clipping
		poly.dx2 = (rand()%360) - 20; // between -20 and +340 to show clipping
		poly.dy2 = (rand()%240) - 20; // between -20 and +220 to show clipping
		poly.dx3 = (rand()%360) - 20; // between -20 and +340 to show clipping
		poly.dy3 = (rand()%240) - 20; // between -20 and +220 to show clipping

		// draw the texture mapped triangle
		TextureMap(&poly);

	} while(!kbhit());
	getch();

}

//-----------------------------------------------------------------

void LoadGraphics(void)
{
	int i,j;
	FILE *fp;

	// This is a picture of me on my first day at university at
	// the University of New Brunswick, Saint John campus (UNBSJ)

	// load in the bitmap we will use for texture mapping
	if( (fp=fopen("texture.raw","rb")) == NULL)
	{
		printf("ERROR: Couldn't load bitmap data\n");
		exit(1);
	}
	fread(dummy_buffer,sizeof(char),64000,fp);
	fclose(fp);

	// copy bitmap to the bitmap buffer (128x128 pixels)
	for(i=0;i<128;i++)
	{
		for(j=0; j<128; j++)
		{
			bitmap[j*128+i]=dummy_buffer[j*320+i];
		}
	}

	// load in the palette
	if( (fp=fopen("texture.pal","rb")) == NULL)
	{
		printf("ERROR: Couldn't load palette data\n");
		exit(1);
	}
	fread(palette,sizeof(char),768,fp);
	fclose(fp);

	SetPalette(palette);

	// show picture on the screen
	for(i=0; i<64000; i++)
	{
		screen[i] = dummy_buffer[i];
	}
}

/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////

//---------------------- texture mapping stuff -----------------------

void TextureMap(TPolygonTYPE *poly)
{
	int i;
	char *temp_ptr;
	int temp;

	int sx,sy; // source texture coordinates
	int x,y; // screen coordinates
	int msx,msy; // slopes of source edge
	int ymin = ClipBottom; // set it to its extreme
	int ymax = ClipTop; // set it to its extreme

	// initialize the edge buffers to their extremes
	for(i=ClipTop;i<=ClipBottom;i++)
	{
		ledge_x[i]=ClipRight;
		redge_x[i]=ClipLeft;
	}

	// scan each edge of triangle
	TScanEdge(poly->dx1,poly->dy1,poly->dx2,poly->dy2,
						poly->sx1,poly->sy1,poly->sx2,poly->sy2);
	TScanEdge(poly->dx2,poly->dy2,poly->dx3,poly->dy3,
						poly->sx2,poly->sy2,poly->sx3,poly->sy3);
	TScanEdge(poly->dx3,poly->dy3,poly->dx1,poly->dy1,
						poly->sx3,poly->sy3,poly->sx1,poly->sy1);

	// find minimum and maximum y coordinates
	if(poly->dy1 < ymin) ymin = poly->dy1;
	if(poly->dy1 > ymax) ymax = poly->dy1;
	if(poly->dy2 < ymin) ymin = poly->dy2;
	if(poly->dy2 > ymax) ymax = poly->dy2;
	if(poly->dy3 < ymin) ymin = poly->dy3;
	if(poly->dy3 > ymax) ymax = poly->dy3;

	// make sure ymin and ymax are between ClipTop and ClipBottom
	if(ymin<ClipTop) ymin=ClipTop;
	if(ymax>ClipBottom) ymax=ClipBottom;

	// texture fill each horizontal scanline
	for(y=ymin;y<=ymax;y++)
	{
		// if the scanline belongs to the polygon... (scanline has pos width)
		if(ledge_x[y] <= redge_x[y])
		{
			// find the slopes of the source edges

			// are we able to divide?
			if( (redge_x[y] - ledge_x[y]) != 0)
			{
				// divide is okay
				msx = ((redge_sx[y] - ledge_sx[y]) << 16) /(redge_x[y] - ledge_x[y]);
				msy = ((redge_sy[y] - ledge_sy[y]) << 16) /(redge_x[y] - ledge_x[y]);
			}
			else
			{
				// we can't divide by zero
				msx = (redge_sx[y] - ledge_sx[y])<<16;
				msy = (redge_sy[y] - ledge_sy[y])<<16;
			}

			// initialize source texture coordinates
			sx=ledge_sx[y]<<16;
			sy=ledge_sy[y]<<16;

			// clip the left edge if neccessary
			if(ledge_x[y] < ClipLeft)
			{
				temp = ClipLeft - ledge_x[y]; // find the amount to skip
				if(temp<0) temp=-temp; // make sure its positive

				ledge_x[y]=ClipLeft; // set it to left edge of clipping area
				sx+=FixedMul(msx,temp<<16);
				sy+=FixedMul(msy,temp<<16);
			}

			// clip right edge if necessary
			if(redge_x[y] > ClipRight)
			{
				redge_x[y] = ClipRight; // hehehe this is too easy!
			}

			// draw the horizontal scanline

			// get the initial screen memory address
			temp_ptr=&screen[ytable[y]+ledge_x[y]];

			// for each pixel across the scanline...
			for(x=ledge_x[y];x<redge_x[y];x++)
			{
				// pixel = bitmap[ y*128 + x]
				*temp_ptr = bitmap[ ((sy>>16)<<7) + (sx>>16) ];
				// increment source texture coordinates to trace edge
				sx+=msx;
				sy+=msy;
				temp_ptr++; // increment screen pointer too
			}
		}
	}
}


void TScanEdge(int xd1,int yd1,int xd2,int yd2,
							 int xs1,int ys1,int xs2,int ys2)
{
	int msx,msy; // slopes of source x and y
	int mdx,mdy; // slopes of destination x and y
	int temp; // for swapping
	int ssx,ssy; // source x and y screen coordinates
	int dsx,dsy; // destination x and y screen coordinates

	// make sure that source edge goes from top to bottom
	if(yd1 > yd2)
	{
		// we need to swap the coordinates around
		temp=xd1;
		xd1=xd2;
		xd2=temp;

		temp=yd1;
		yd1=yd2;
		yd2=temp;

		temp=xs1;
		xs1=xs2;
		xs2=temp;

		temp=ys1;
		ys1=ys2;
		ys2=temp;
	}

	// initialize the slopes for stepping the edges
	if((yd2-yd1) != 0)
	{
		mdx = ((xd2-xd1) << 16) / (yd2-yd1); // dx_dest/dy_dest
		msx = ((xs2-xs1) << 16) / (yd2-yd1); // dx_src/dy_dest
		msy = ((ys2-ys1) << 16) / (yd2-yd1); // dy_src/dy_dest
	}
	else
	{
		mdx = ((xd2-xd1) << 16); // dx_dest
		msx = ((xs2-xs1) << 16); // dx_src
		msy = ((ys2-ys1) << 16); // dy_src
	}

	// initialize first coordinates
	ssx=xs1<<16;
	ssy=ys1<<16;
	dsx=xd1<<16;

	// step through edge and record coordinates along the way
	for(dsy=yd1;dsy<yd2;dsy++)
	{
		// as long as we are between the vertical clipping area... trace it
		if( (dsy>=ClipTop) && (dsy<=ClipBottom) )
		{
			if( (dsx>>16) < ledge_x[dsy])
			{
				// update left edge information
				ledge_x[dsy]=(dsx>>16);
				ledge_sx[dsy]=(ssx>>16);
				ledge_sy[dsy]=(ssy>>16);
			}

			if( (dsx>>16) > redge_x[dsy])
			{
				// update right edge information
				redge_x[dsy]=(dsx>>16);
				redge_sx[dsy]=(ssx>>16);
				redge_sy[dsy]=(ssy>>16);
			}
		}

		// increment the coordinates by their respective slopes
		dsx+=mdx;
		ssx+=msx;
		ssy+=msy;
	}
}


/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////

