{----------------------------------------------------------}
{ LOWRES unit ver. 2.0  1/1993                             }
{ Geoffrey Silverton                                       }
{                                                          }
{ Known to work with Turbo Pascal 4 and Turbo Pascal 6.    }
{ Supports CGA, EGA, MCGA, and VGA video boards.           }
{ Does not include "snow checking" to avoid interference   }
{   on early CGA boards.  This can be easily added.        }
{----------------------------------------------------------}

unit lowres;

{$V-}

{*****}  interface  {*****}
uses crt, dos, graph;

const MONO = -2;       { This is not really supported and so does nothing... }
      ON   = TRUE;
      OFF  = FALSE;

var card:        integer;
    screen_seg:  word;

procedure enter_lowres_mode;
procedure exit_lowres_mode;
procedure prepare_low(x1, y1, x2, y2: integer; fill, c1, c2: byte);

function  video_card_detected: integer;

procedure lplot(x, y: integer; c: byte);
function  lgetdotcolor(x, y: integer): byte;
procedure lline(x1, y1, x2, y2: integer; c: byte);
procedure lbox(x1, y1, x2, y2: integer; c: byte);
procedure lfill_box(x1, y1, x2, y2: integer; c: byte);

procedure lgotoxy(x, y: integer);
procedure ltextcolor(c: byte);
procedure ltextbackground(c: byte);
procedure lwrite(st: string);
procedure lwrite_chr(ch: char);
procedure lwrite_num(x: integer);

{*****}  implementation  {*****}

{ This is a little 5x6 pixel font.  Not too pretty, but small. }
{$I LOWFONT.INC}

const
    ldummy: boolean =  FALSE;      { To initialize lowres_installed to FALSE }

var cursorx_, cursory_:  integer;  { Globals for text position }
    color_, color_back_: byte;     { Text color globals        }
    blinking:            boolean;
    lowres_installed:    boolean absolute ldummy;


{**************************** LOCAL ROUTINES *****************************}

{ Switch two bytes }
{ Switch two integers }
procedure switch(var a, b: integer);
  var dummy: integer;
  begin
    dummy := a;
    a     := b;
    b     := dummy;
  end;

{ HIDE_TEXT_CURSOR hides the textmode cursor by positioning it off screen }
procedure hide_text_cursor;
  var reg: registers;
  begin
    reg.ah := 2;                { Function 2: set cursor location }
    reg.bh := 0;
    reg.dx := 25 shl 8;
    intr($10, reg);
  end;

{ Set video register }
procedure set_video_reg(n, val: byte);
  begin
    port[$3d4] := n;            { Write to CGA/EGA/MCGA/VGA register }
    port[$3d5] := val;
  end;

{ Draw vertical line quicker than Bresenham }
procedure lline_quick_v(x, y1, y2: integer; c: byte);
var
  y: integer;
  m: word;
begin
  if y1>y2 then
    switch(y1, y2);                 { Go downwards on screen }

  m := (x or 1) + y1*160;           { Get first memory address }

  if odd(x) then                    { Different for odd or even column }
    for y:=y1 to y2 do
      begin                           { Set background colors }
        mem[screen_seg:m] := mem[screen_seg:m] and 15 or (c shl 4);
        inc(m, 160);
      end
  else
    for y:=y1 to y2 do
      begin                           { Set foreground colors }
        mem[screen_seg:m] := mem[screen_seg:m] and 240 or c;
        inc(m, 160);                  
      end;
end;


{**************************** GLOBAL ROUTINES *****************************}

{ Read color value at pixel position }
function lgetdotcolor;
    var m: word;
    begin
      m := (x or 1) + y*160;      { Get memory address }

      if odd(x)                   { Check background if odd, else foreground }
        then
          lgetdotcolor := (mem[screen_seg:m] and 240) shr 4
        else
          lgetdotcolor := mem[screen_seg:m] and 15;
    end;

{ Plot color value at given pixel coordinates }
procedure lplot;
    var m: word;
    begin
      m := (x or 1) + y*160;      { Get memory address }
      if odd(x)
        then                        { Set background color if odd column }
          mem[screen_seg:m] := mem[screen_seg:m] and 15 or (c shl 4)
        else                        { Set foreground color if even column }
          mem[screen_seg:m] := mem[screen_seg:m] and 240 or c;
    end;


{******* Text functions... ********}

{ Move down one "line" and to extreme left of screen }
procedure new_line;
  begin
    cursorx_ := 0;
    cursory_ := cursory_+6;
  end;

{ Write character at cursor position }
procedure lwrite_chr;
  var i, j, b, len: byte;
  begin
    if ch = #13 then                  { Carriage-return in text? }
      new_line
    else
      begin
        len := lofont[ord(ch)][5];    { Get width of character }
        if cursorx_+len>159 then
          new_line;

        for i := 0 to len-1 do        { Go through each column of character }
          begin
            b := 128;                 { Start with bit 7 (high bit) }
            for j := 0 to 5 do        { Go through 6 bits (6 rows of pixels) }
              begin
                if (lofont[ord(ch)][i] and b) <> 0 then
                  lplot(i+cursorx_, j+cursory_-5, color_)        { foregrnd }
                else
                  lplot(i+cursorx_, j+cursory_-5, color_back_);  { backgrnd }

                b := b shr 1;         { Move to next lower bit }
              end;
          end;
        { Advance to next character position }
        cursorx_ := cursorx_+len+1;

        { Fill in space between characters (remove this if causes problems) }
        lline_quick_v(cursorx_-1, cursory_, cursory_-5, color_back_);
      end;
  end;

{ Set a new text color }
procedure ltextcolor;
  begin
    color_ := c;
  end;

{ Set a new text color }
procedure ltextbackground;
  begin
    color_back_ := c;
  end;

{ Change cursor position }
procedure lgotoxy;
  begin
    cursorx_ := x;
    cursory_ := y;
  end;

{ Write text string }
procedure lwrite;
  var i: byte;
  begin
    for i := 1 to length(st) do lwrite_chr(st[i]);
  end;

{ Write number value string }
procedure lwrite_num;
  var st: string[6];
  begin
    str(x, st);                     { integer --> string }
    lwrite(st);                     { Write that string }
  end;


{********* Miscellaneous stuff... **********}

{ Which video card? }
function video_card_detected;
  var detected, dummy: integer;
  begin
    detectgraph(detected, dummy);     { Use Turbo graphics unit function }
    video_card_detected := detected;
  end;

{ Change value of port $3?8 }
procedure video(request: boolean);
  begin
    case card of
      CGA, MCGA, VGA:
        port[$3d8] := 1 or (ord(request) shl 3) or (ord(blinking) shl 5);
      MONO, HERCMONO:
        port[$3b8] := 1 or (ord(request) shl 3) or (ord(blinking) shl 5);
    end;
  end;

{ LORES_MODE enters (if not already in) the 160x100 lowres 16 color mode. }
{ This undocumented graphics mode can be achieved on most color cards.    }

{ CGA and similar cards: use "2 pixel high" text  }
{   characters. On CGA-type cards that use more   }
{   than 8 pixels for text characters, like many  }
{   non-IBM CGA cards and MCGA, this will work    }
{   anyway.  Blink is suppressed to get 16        }
{   background colors.  This is done by making    }
{   bit 5 of port $3d8 zero.                      }

{ EGA, VGA: use 3 pixel or 4 pixel high text      }
{   respectively.  Blink is suppressed by using   }
{   function $10 of interrupt $10, subfunction 3, }
{   set BL = 0.  On the EGA card, 100 rows of     }
{   "text" do not evenly fit on the screen.  In   }
{   order to display all 100 rows, 3 pixel high   }
{   characters are used and a fair amount of the  }
{   screen is left blank at the bottom.  You      }
{   really have a 160x116 pixel mode.             }

procedure enter_lowres_mode;
  var reg: registers;
  begin
    screen_seg := $b800;                   { Assume we have a color board }
    textmode(co80);                        { Enter color text mode }
    fillchar(mem[screen_seg:0], 24000, 0); { FILL ALL 16K NEEDED+EXTRA FOR EGA }
    lowres_installed := TRUE;
    blinking := OFF;

    card := video_card_detected;

    case card of
      CGA, MCGA, ATT400:
          begin
            video(off);
            set_video_reg(4, 127);  { REG. 4 IS VERTICAL TOTAL ROWS }
            set_video_reg(6, 100);  { REG. 6 IS VERTICAL DISPLAYED ROWS }
            set_video_reg(7, 112);  { REG. 7 IS "VERTICAL SYNC. POSITION" }
            set_video_reg(9, 1);    { REG. 9 IS PIXEL HEIGHT OF TEXT         }
                                    {   (SET TO 2 PIXELS HERE: 0 WOULD BE 1) }
            video(on);              { Blink turned off by this routine on CGA }
          end;
      VGA:
          begin
            video(off);
            set_video_reg(9, 3);    { SET PIXEL HEIGHT = 4 }
            reg.AX := $1003;         
            reg.BL := 0;
            intr($10, reg);         { Turn blink off }
            video(on);
          end;
      EGA, EGA64:
          begin
            video(off);
            set_video_reg(9, 2);    { SET PIXEL HEIGHT = 3 }
            reg.AX := $1003;
            reg.BL := 0;
            intr($10, reg);         { Turn blink off }
            video(on);
          end;
{                                     Hercules not supported at this time...
       HERCMONO:
          begin
            set_video_reg(4, 100);
            set_video_reg(6, 100);
            set_video_reg(7, 101);
            set_video_reg(9, 3);
          end;
}
    end;
    hide_text_cursor;               { Don't want cursor in "graphics" mode! }

    cursorx_ := 0;
    cursory_ := 5;
    color_   := 14;
  end;

procedure exit_lowres_mode;
  var reg: registers;
  begin
    if lowres_installed then
      begin
        reg.ax := $0004;          { Restore normal text mode characteristics }
        intr($10, reg);           {   (Turbo Pascal may be unaware of our    }
                                  {    changes.  So use BIOS.)               }

        textmode(lastmode);       { Tell Turbo Pascal to use user's old mode }

        lowres_installed := FALSE;
      end;
  end;


{ Fill video screen region with "fill" character.                       }
{ Often used with "fill" = chr(221) so can plot pixels in left or right }
{ half of each character using foreground/background colors.            }
procedure prepare_low;
  var xa, xb:    integer;
      x, y:      integer;
      filler, m: word;
  begin
    if x1 > x2 then switch(x1, x2);  { Go left to right }
    if y1 > y2 then switch(y1, y2);  { Top to bottom }

    { Two pixel columns for each byte address: }
    xa := x1 shr 1;                  { x1 / 2 to get byte address }
    xb := x2 shr 1;                  { x2 / 2 to get byte address}

    m := y1 * 160;                   { Memory address of start of pixel row }

    filler := fill                   { Character with          }
            + c1 shl 8               {   foreground color      }
            + c2 shl 12;             {   and background color. }

    for y := y1 to y2 do
      begin
        for x := xa to xb do
          memw[screen_seg : (x shl 1) + m] := filler;  { Set word of memory }
        inc(m, 160);                                   { Down one row }
      end;
  end;


{ Draw Bresenham line }
procedure lline;
var
  dx, dy:                     integer;
  incrStraight, incrDiagonal: integer;
  d:                          integer;
  x, y:                       integer;
  incX, incY:                 integer;

begin
  if x1 = x2 then
    lline_quick_v(x1, y1, y2, c);    { Vertical line?  Do it quicker... }

  if y1 = y2 then
    lfill_box(x1, y1, x2, y2, c);    { Horizontal line?  Do it quicker... }

  if x2 > x1 then incX := 1          { Right }
             else incX := -1;        { Left }
  if y2 > y1 then incY := 1          { Down }
             else incY := -1;        { Up }

  dx := abs(x2 - x1);                { Horizontal distance }
  dy := abs(y2 - y1);                { Vertical distance }

  if dx > dy then                    { Not steep? }
    begin
      d := (dy shl 1) - dx;               { Step less than 1 in vert. dir. }
      incrStraight := dy shl 1;
      incrDiagonal := (dy-dx) shl 1;
    end
  else                               { Steep. }
    begin
      d := (dx shl 1) - dy;               { Step less than 1 in horiz. dir. }
      incrStraight := dx shl 1;
      incrDiagonal := (dx-dy) shl 1;
    end;

  x := x1;                                { Start out... }
  y := y1;
  lplot(x, y, c);                         { Plot first pixel }

  while (x <> x2) or (y <> y2) do         { Until we reach end of the line... }
    begin
      if (d <= 0) then                    { Still go straigt? }
        begin
          inc(d, incrStraight);              { Yes, go straight }
          if (dx>dy) then
            inc(x, incX)                     { Move in horizontal direction }
          else
            inc(y, incY);                    { Move in vertical direction }
        end
      else
        begin
          inc(d, incrDiagonal);              { Move diagonally now. }
          inc(x, incX);
          inc(y, incY);
        end;

      lplot(x, y, c);                        { Plot this pixel }
    end;
end;


{ Draw box }
procedure lbox;
  begin
    lfill_box(x1, y1, x2, y1, c);    { Horizontal lines }
    lfill_box(x1, y2, x2, y2, c);

    lline_quick_v(x1, y1, y2, c);    { Vertical lines }
    lline_quick_v(x2, y1, y2, c);
  end;

{ Draw filled box }
procedure lfill_box;
  var x, y:   integer;
      nbytes: integer;
      m:      word;
      cbyte:  byte;
  begin
    if x1>x2 then                     { Go left to right }
      switch(x1, x2);
    if y1>y2 then                     { Go top to bottom }
      switch(y1, y2);

    if odd(x1) then                   { Not entire byte on left edge? }
      begin
        lline_quick_v(x1, y1, y2, c);    { Draw single pixel column }
        inc(x1);
      end;

    if not odd(x2) then               { Not entire byte on right edge? }
      begin
        lline_quick_v(x2, y1, y2, c);    { Draw single pixel column }
        dec(x2);
      end;

    cbyte := c or (c shl 4);          { Color byte: both fore. and background }
    nbytes := (x2-x1+1) shr 1;        { Number of bytes to fill }
    m := (x1 or 1) + y1*160;          { Starting memory address }

    for y:=y1 to y2 do                { For each row of pixels... }
      begin
        for x:=1 to nbytes do           { For each byte in row... }
          begin
            mem[screen_seg:m] := cbyte;   { Set byte }
            inc(m,2);                     { Move two bytes ahead }
          end;
        inc(m, 160 - (nbytes shl 1));   { Go back to left and down a line }
      end;
  end;


end.

