 /*
   ** Copyright (C) 1995  Jonathan Paul Griffiths.  All rights reserved.
   **
   ** You may do anything with this code you wish EXCEPT sell it. You may sell
   ** any software you create using this code,  but you MUST NOT charge for
   ** the code itself.  See the file "readme.jon" for detailed terms. 
 */

/* jmemcopy.h - Memory Operations */

/*
   This module provides memory routines. Every routine takes len as an
   unsigned int representing the number of BYTES to copy,  while source and
   dest are void pointers. Only get_best_copy_method returns a value,  all
   other routines return void.  Any of these routines may be implemented as
   macros.  The store functions take UBYTES to store.

   The following routines must be provided by this module.

   JINLINE int get_best_copy_method(source,destination,length)
   This routine returns a value representing the alignment state of the
   addresses and the length given.  This information is used to optimise
   particular cases on targets such as the 80x86 that have fast string
   copy or store instructions.  The routine returns a value from
   ALIGNMENT_STATE_1 to ALIGNMENT_STATE_8.  There are 8 different copy
   and store routines that each correspond to an alignment state.  These 
   routines are called MEM_COPY_ALIGN_1 to MEM_COPY_ALIGN_8 and
   MEM_STORE_ALIGN_1 to MEM_STORE_ALIGN_8.

   other routines that must be provided by this module:

   FAST_SHORT_COPY - for short copies.
   FAST_LONG_COPY  - for long copies.
   FAST_VERY_LONG_COPY - for very long copies.
   BYTE_COPY     - ideally copies UBYTES.  
   WORD_COPY     - ideally copies USHORTS.  **len must be divisible by 2**
   LONG_COPY     - ideally copies 32 bits at a time. **len must divide by 4**

   FAST_SHORT_STORE - for short stores.
   FAST_LONG_STORE  - for long stores.
   FAST_VERY_LONG_STORE - for very long stores.
   BYTE_STORE     - ideally stores using UBYTES.
   WORD_STORE     - ideally stores using USHORTS.**len must be divisible by 2**
   LONG_STORE     - ideally stores using 32 bits. **len must divide by 4**
 */

#include <jconfig.h>

#ifndef __JMEMCOPY_H__          /* make sure file is only included once */
#define __JMEMCOPY_H__

#ifdef __cplusplus
extern "C" {
#endif 


/****************************************************************************/
#ifdef LINUX_SVGALIB            /* not very optimised at the moment.    */
/****************************************************************************/

/****************************************************************************/
#ifdef MODE_13H
/****************************************************************************/

#include <string.h>

#define get_best_copy_method(s,d,l) ALIGNMENT_STATE_1

#define ALIGNMENT_STATE_1   0
#define ALIGNMENT_STATE_2   1
#define ALIGNMENT_STATE_3   2
#define ALIGNMENT_STATE_4   3
#define ALIGNMENT_STATE_5   4
#define ALIGNMENT_STATE_6   5
#define ALIGNMENT_STATE_7   6
#define ALIGNMENT_STATE_8   7

/* every memory copy is done using memcpy */
#define MEM_COPY_ALIGN_1(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_2(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_3(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_4(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_5(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_6(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_7(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_8(x,y,z)  memcpy( (y) , (x) , (z) )

#define FAST_SHORT_COPY(s,d,len)     memcpy( (d) , (s) , (len) )
#define FAST_LONG_COPY(s,d,len)      memcpy( (d) , (s) , (len) )
#define FAST_VERY_LONG_COPY(s,d,len) memcpy( (d) , (s) , (len) )
#define BYTE_COPY(s,d,len)           memcpy( (d) , (s) , (len) )
#define WORD_COPY(s,d,len)           memcpy( (d) , (s) , (len) )
#define LONG_COPY(s,d,len)           memcpy( (d) , (s) , (len) )

/* every memory store is done using memset */
#define MEM_STORE_ALIGN_1(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_2(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_3(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_4(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_5(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_6(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_7(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_8(d,val,len)  memset( (d) , (val) , (len) )

#define FAST_SHORT_STORE(d,val,len)  memset( (d) , (val) , (len) )
#define FAST_LONG_STORE(d,val,len)   memset( (d) , (val) , (len) )
#define FAST_VERY_LONG_STORE(d,val,len) memset( (d) , (val) , (len) )
#define BYTE_STORE(d,val,len)        memset( (d) , (val) , (len) )
#define WORD_STORE(d,val,len)        memset( (d) , (val) , (len) )
#define LONG_STORE(d,val,len)        memset( (d) , (val) , (len) )

#endif /* 13h */

#endif /* linux svgalib */

/****************************************************************************/
#ifdef LINUX_GRX2
/****************************************************************************/

#include <string.h>

#define get_best_copy_method(s,d,l) ALIGNMENT_STATE_1

#define ALIGNMENT_STATE_1   0
#define ALIGNMENT_STATE_2   1
#define ALIGNMENT_STATE_3   2
#define ALIGNMENT_STATE_4   3
#define ALIGNMENT_STATE_5   4
#define ALIGNMENT_STATE_6   5
#define ALIGNMENT_STATE_7   6
#define ALIGNMENT_STATE_8   7

/* every memory copy is done using memcpy */
#define MEM_COPY_ALIGN_1(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_2(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_3(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_4(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_5(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_6(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_7(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_8(x,y,z)  memcpy( (y) , (x) , (z) )

#define FAST_SHORT_COPY(s,d,len)     memcpy( (d) , (s) , (len) )
#define FAST_LONG_COPY(s,d,len)      memcpy( (d) , (s) , (len) )
#define FAST_VERY_LONG_COPY(s,d,len) memcpy( (d) , (s) , (len) )
#define BYTE_COPY(s,d,len)           memcpy( (d) , (s) , (len) )
#define WORD_COPY(s,d,len)           memcpy( (d) , (s) , (len) )
#define LONG_COPY(s,d,len)           memcpy( (d) , (s) , (len) )

/* every memory store is done using memset */
#define MEM_STORE_ALIGN_1(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_2(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_3(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_4(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_5(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_6(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_7(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_8(d,val,len)  memset( (d) , (val) , (len) )

#define FAST_SHORT_STORE(d,val,len)  memset( (d) , (val) , (len) )
#define FAST_LONG_STORE(d,val,len)   memset( (d) , (val) , (len) )
#define FAST_VERY_LONG_STORE(d,val,len) memset( (d) , (val) , (len) )
#define BYTE_STORE(d,val,len)        memset( (d) , (val) , (len) )
#define WORD_STORE(d,val,len)        memset( (d) , (val) , (len) )
#define LONG_STORE(d,val,len)        memset( (d) , (val) , (len) )

#endif /* linux grx 2 */

/****************************************************************************/
#ifdef DJGPP_DPMI_COMPLIANT     /* DJGPP with DOS,  OS/2,  WIN */
/****************************************************************************/

/* these routines apply to both 13h and grx variants , so are listed together */

/* return the best copy method for two addresses and a length */
JINLINE int get_best_copy_method (void *addr1, void *addr2, unsigned int len);

#define ALIGNMENT_STATE_1   0
#define ALIGNMENT_STATE_2   1
#define ALIGNMENT_STATE_3   2
#define ALIGNMENT_STATE_4   3
#define ALIGNMENT_STATE_5   4
#define ALIGNMENT_STATE_6   5
#define ALIGNMENT_STATE_7   6
#define ALIGNMENT_STATE_8   7

/* some copies are optimised according to alignment */

#define MEM_COPY_ALIGN_1(x,y,z)  _copy_sb( (x) , (y) , (z) )
#define MEM_COPY_ALIGN_2(x,y,z)  _copy_inc_sw( (x) , (y) , (z) )
#define MEM_COPY_ALIGN_3(x,y,z)  _copy_sw( (x) , (y) , (z) )
#define MEM_COPY_ALIGN_4(x,y,z)  _copy_sw_inc( (x) , (y) , (z) )
#define MEM_COPY_ALIGN_5(x,y,z)  _copy_inc_sw_inc( (x) , (y) , (z) )
#define MEM_COPY_ALIGN_6(x,y,z)  _copy_sb( (x) , (y) , (z) )
#define MEM_COPY_ALIGN_7(x,y,z)  _copy_sb( (x) , (y) , (z) )
#define MEM_COPY_ALIGN_8(x,y,z)  _copy_sb( (x) , (y) , (z) )

#define FAST_SHORT_COPY(a,b,len) _copy_sb((a),(b),(len))
#define FAST_LONG_COPY(a,b,len)  _blast((a),(b),(len))
#define FAST_VERY_LONG_COPY(a,b,len)  _blast((a),(b),(len))
#define BYTE_COPY(a,b,len)       _bcopy((a),(b),(len))
#define WORD_COPY(a,b,len)       _copy_sw((a),(b),(len))
#define LONG_COPY(a,b,len)       _blast((a),(b),(len))

/* every memory store is done using memset */
#define MEM_STORE_ALIGN_1(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_2(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_3(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_4(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_5(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_6(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_7(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_8(d,val,len)  memset( (d) , (val) , (len) )

#define FAST_SHORT_STORE(d,val,len)  memset( (d) , (val) , (len) )
#define FAST_LONG_STORE(d,val,len)   memset( (d) , (val) , (len) )
#define FAST_VERY_LONG_STORE(d,val,len) memset( (d) , (val) , (len) )
#define BYTE_STORE(d,val,len)        memset( (d) , (val) , (len) )
#define WORD_STORE(d,val,len)        memset( (d) , (val) , (len) )
#define LONG_STORE(d,val,len)        memset( (d) , (val) , (len) )


/* other copying routines */
void _blast (void *src, void *dest, unsigned int len);


/* ========================================================================
   Try to ignore this mess,  I had to this because the compiler wouldn't   
   inline the assembly.                                                            
   ========================================================================
 */

/* all of these copies are source, dest, len */
#define _copy_sb(src,dest,len) asm volatile(                              "\n\
push %%ecx \n \
movl %0, %%esi \n \
movl %1, %%edi \n \
movl %2, %%ecx \n \
rep \n \
movsb \n \
pop %%ecx "\
:                               /* NOTHING */ \
:" g " (src), " g " (dest), " g " (len) \
:"di", "si", "cx")

#define _copy_sw(src,dest,len) asm volatile(                              "\n\
push %%ecx \n \
movl %0, %%esi \n \
movl %1, %%edi \n \
movl %2, %%ecx \n \
shrl $1, %%ecx \n \
rep \n \
movsw \n \
pop %%ecx "\
:                               /* NOTHING */ \
:" g " (src), " g " (dest), " g " (len) \
:"di", "si", "cx")

#define _copy_sw_inc(src,dest,len) asm volatile(                         "\n\
push %%ecx \n \
movl %0, %%esi \n \
movl %1, %%edi \n \
movl %2, %%ecx \n \
shrl $1, %%ecx \n \
rep \n \
movsw \n \
movl $1, %%ecx \n \
rep \n \
movsb \n \
pop %%ecx "\
:                               /* NOTHING */ \
:" g " (src), " g " (dest), " g " (len) \
:"di", "si", "cx")

#define _copy_inc_sw(src,dest,len) asm volatile(                         "\n\
push %%ecx \n \
movl %0, %%esi \n \
movl %1, %%edi \n \
movl $1, %%ecx \n \
rep \n \
movsb \n \
movl %2, %%ecx \n \
shrl $1, %%ecx \n \
rep \n \
movsw \n \
pop %%ecx "\
:                               /* NOTHING */ \
:" g " (src), " g " (dest), " g " (len) \
:"di", "si", "cx")

#define _copy_inc_sw_inc(src,dest,len) asm volatile(                     "\n\
push %%ecx \n \
movl %0, %%esi \n \
movl %1, %%edi \n \
movl $1, %%ecx \n \
rep \n \
movsb \n \
movl %2, %%ecx \n \
shrl $1, %%ecx \n \
rep \n \
movsw \n \
movl $1, %%ecx \n \
rep \n \
movsb \n \
pop %%ecx "\
:                               /* NOTHING */ \
:" g " (src), " g " (dest), " g " (len) \
:"di", "si", "cx")

/* all of this stuff deals with *dos* memory copies */
#define DOS_MEMCOPY_ALIGN_1(x,y,z)  _dos_copy_sb( (x) , (y) , (z) )
#define DOS_MEMCOPY_ALIGN_2(x,y,z)  _dos_copy_inc_sw( (x) , (y) , (z) )
#define DOS_MEMCOPY_ALIGN_3(x,y,z)  _dos_copy_sw( (x) , (y) , (z) )
#define DOS_MEMCOPY_ALIGN_4(x,y,z)  _dos_copy_sw_inc( (x) , (y) , (z) )
#define DOS_MEMCOPY_ALIGN_5(x,y,z)  _dos_copy_inc_sw_inc( (x) , (y) , (z) )
#define DOS_MEMCOPY_ALIGN_6(x,y,z)  _dos_copy_sb( (x) , (y) , (z) )
#define DOS_MEMCOPY_ALIGN_7(x,y,z)  _dos_copy_sb( (x) , (y) , (z) )
#define DOS_MEMCOPY_ALIGN_8(x,y,z)  _dos_copy_sb( (x) , (y) , (z) )

#define _dos_copy_sb(src,dest,len) asm volatile(                         "\n\
pushw %%es \n \
movw __jlib_sel, %%es \n \
 push %%ecx \n \
 movl %0, %%esi \n \
 addl __jlib_off, %%esi \n \
 movl %1, %%edi \n \
 movl %2, %%ecx \n \
 rep \n \
 movsb \n \
 pop %%ecx \n \
 popw %%es "\
:                               /* NOTHING */ \
:" g " (src), " g " (dest), " g " (len) \
:"di", "si", "cx")

#define _dos_copy_sw(src,dest,len) asm volatile(                         "\n\
pushw %%es \n \
movw __jlib_sel, %%es \n \
 push %%ecx \n \
 movl %0, %%esi \n \
 addl __jlib_off, %%esi \n \
 movl %1, %%edi \n \
 movl %2, %%ecx \n \
 shrl $1, %%ecx \n \
 rep \n \
 movsw \n \
 pop %%ecx \n \
 popw %%es "\
:                               /* NOTHING */ \
:" g " (src), " g " (dest), " g " (len) \
:"di", "si", "cx")

#define _dos_copy_sw_inc(src,dest,len) asm volatile(                     "\n\
pushw %%es \n \
movw __jlib_sel, %%es \n \
 push %%ecx \n \
 movl %0, %%esi \n \
 addl __jlib_off, %%esi \n \
 movl %1, %%edi \n \
 movl %2, %%ecx \n \
 shrl $1, %%ecx \n \
 rep \n \
 movsw \n \
 movl $1, %%ecx \n \
 rep \n \
 movsb \n \
 pop %%ecx \n \
 popw %%es "\
:                               /* NOTHING */ \
:" g " (src), " g " (dest), " g " (len) \
:"di", "si", "cx")

#define _dos_copy_inc_sw(src,dest,len) asm volatile(                     "\n\
pushw %%es \n \
movw __jlib_sel, %%es \n \
 push %%ecx \n \
 movl %0, %%esi \n \
 addl __jlib_off, %%esi \n \
 movl %1, %%edi \n \
 movl $1, %%ecx \n \
 rep \n \
 movsb \n \
 movl %2, %%ecx \n \
 shrl $1, %%ecx \n \
 rep \n \
 movsw \n \
 pop %%ecx \n \
 popw %%es "\
:                               /* NOTHING */ \
:" g " (src), " g " (dest), " g " (len) \
:"di", "si", "cx")

#define _dos_copy_inc_sw_inc(src,dest,len) asm volatile(                     "\n\
pushw %%es \n \
movw __jlib_sel, %%es \n \
 push %%ecx \n \
 movl %0, %%esi \n \
 addl __jlib_off, %%esi \n \
 movl %1, %%edi \n \
 movl $1, %%ecx \n \
 rep \n \
 movsb \n \
 movl %2, %%ecx \n \
 shrl $1, %%ecx \n \
 rep \n \
 movsw \n \
 movl $1, %%ecx \n \
 rep \n \
 movsb \n \
 pop %%ecx \n \
 popw %%es "\
:                               /* NOTHING */ \
:" g " (src), " g " (dest), " g " (len) \
:"di", "si", "cx")

#endif /* djgpp */

/****************************************************************************/
#ifdef X_WINDOWS
/****************************************************************************/

#include <string.h>

#define get_best_copy_method(s,d,l) ALIGNMENT_STATE_1

#define ALIGNMENT_STATE_1   0
#define ALIGNMENT_STATE_2   1
#define ALIGNMENT_STATE_3   2
#define ALIGNMENT_STATE_4   3
#define ALIGNMENT_STATE_5   4
#define ALIGNMENT_STATE_6   5
#define ALIGNMENT_STATE_7   6
#define ALIGNMENT_STATE_8   7

/* every memory copy is done using memcpy */
#define MEM_COPY_ALIGN_1(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_2(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_3(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_4(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_5(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_6(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_7(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_8(x,y,z)  memcpy( (y) , (x) , (z) )

#define FAST_SHORT_COPY(s,d,len)     memcpy( (d) , (s) , (len) )
#define FAST_LONG_COPY(s,d,len)      memcpy( (d) , (s) , (len) )
#define FAST_VERY_LONG_COPY(s,d,len) memcpy( (d) , (s) , (len) )
#define BYTE_COPY(s,d,len)           memcpy( (d) , (s) , (len) )
#define WORD_COPY(s,d,len)           memcpy( (d) , (s) , (len) )
#define LONG_COPY(s,d,len)           memcpy( (d) , (s) , (len) )

/* every memory store is done using memset */
#define MEM_STORE_ALIGN_1(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_2(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_3(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_4(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_5(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_6(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_7(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_8(d,val,len)  memset( (d) , (val) , (len) )

#define FAST_SHORT_STORE(d,val,len)  memset( (d) , (val) , (len) )
#define FAST_LONG_STORE(d,val,len)   memset( (d) , (val) , (len) )
#define FAST_VERY_LONG_STORE(d,val,len) memset( (d) , (val) , (len) )
#define BYTE_STORE(d,val,len)        memset( (d) , (val) , (len) )
#define WORD_STORE(d,val,len)        memset( (d) , (val) , (len) )
#define LONG_STORE(d,val,len)        memset( (d) , (val) , (len) )

#endif /* X windows */

/****************************************************************************/
#ifdef GENERIC                  /* hopefully portable memory operations */
/****************************************************************************/
#include <string.h>

#define get_best_copy_method(s,d,l) ALIGNMENT_STATE_1

#define ALIGNMENT_STATE_1   0
#define ALIGNMENT_STATE_2   1
#define ALIGNMENT_STATE_3   2
#define ALIGNMENT_STATE_4   3
#define ALIGNMENT_STATE_5   4
#define ALIGNMENT_STATE_6   5
#define ALIGNMENT_STATE_7   6
#define ALIGNMENT_STATE_8   7

/* every memory copy is done using memcpy */

#define MEM_COPY_ALIGN_1(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_2(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_3(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_4(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_5(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_6(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_7(x,y,z)  memcpy( (y) , (x) , (z) )
#define MEM_COPY_ALIGN_8(x,y,z)  memcpy( (y) , (x) , (z) )

#define FAST_SHORT_COPY(s,d,len)     memcpy( (d) , (s) , (len) )
#define FAST_LONG_COPY(s,d,len)      memcpy( (d) , (s) , (len) )
#define FAST_VERY_LONG_COPY(s,d,len) memcpy( (d) , (s) , (len) )
#define BYTE_COPY(s,d,len)           memcpy( (d) , (s) , (len) )
#define WORD_COPY(s,d,len)           memcpy( (d) , (s) , (len) )
#define LONG_COPY(s,d,len)           memcpy( (d) , (s) , (len) )

/* every memory store is done using memset */
#define MEM_STORE_ALIGN_1(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_2(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_3(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_4(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_5(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_6(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_7(d,val,len)  memset( (d) , (val) , (len) )
#define MEM_STORE_ALIGN_8(d,val,len)  memset( (d) , (val) , (len) )

#define FAST_SHORT_STORE(d,val,len)  memset( (d) , (val) , (len) )
#define FAST_LONG_STORE(d,val,len)   memset( (d) , (val) , (len) )
#define FAST_VERY_LONG_STORE(d,val,len) memset( (d) , (val) , (len) )
#define BYTE_STORE(d,val,len)        memset( (d) , (val) , (len) )
#define WORD_STORE(d,val,len)        memset( (d) , (val) , (len) )
#define LONG_STORE(d,val,len)        memset( (d) , (val) , (len) )

#endif /* generic */

#ifdef __cplusplus
}
#endif 

#endif /* ifndef __JMEMCOPY_H__ */
