/* mixcmd.c */
/*
PLAY_ITW.EXE v0.02b : Player for Impulse Tracker modules files
Copyright (C) 1997  Olivier AUMAGE
E-mail : Olivier.Aumage@ens-lyon.fr
Web : http://www.ens-lyon.fr/~oaumage/

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  any later version.
  
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	  You should have received a copy of the GNU General Public License
	  along with this program; if not, write to the Free Software
	  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


/* main header files */
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

/* project header files */
#include "types.h"
#include "mix_fi0.h"
#include "mix_fi1.h"
#include "mix_fi2.h"
#include "mixtypes.h"
#include "mixvars.h"
#include "mixcmd.h"

/* functions */
double convert_to_amiga (double value)
{
	if (value < 1)
	{
		value = 1.0 ;
	}
	return (double) (1795162112/value) ;  /* 1795162112 = 16384 * 256 * 428, 428 = amiga val for middle C*/
}

double convert_from_amiga (double amiga_value)
{
	if (amiga_value < 1)
	{
		amiga_value = 1.0 ;
	}
	return (double) (1795162112/amiga_value) ;
}

void set_speed_and_tempo (unsigned char speed, unsigned char tempo)
{
	unsigned long frames_per_minutes ;
	
	beats_per_minute = tempo ;
	frames_per_minutes = frames_per_beats * beats_per_minute ;
	samples_per_frame = samples_per_minutes / frames_per_minutes ;
	frames_per_line = speed ;
}

/* new pattern steps to the next pattern in the orders, at the given line */
void new_pattern (unsigned char line)
{
	pointer_on_current_pattern = pointer_on_current_module->patterns[current_pattern] ;
	pointer_on_current_pattern_byte = pointer_on_current_pattern->packed_pattern_data ;
	current_order ++ ;
	
	while (
		(current_order < pointer_on_current_module->number_of_orders)
		&& (pointer_on_current_module->orders[current_order] == 254)
        )
	{
		current_order ++ ;
	}
	
	if ((current_order >= pointer_on_current_module->number_of_orders) || (pointer_on_current_module->orders[current_order] == 255)) /* end of song */
	{
		if (loop_allowed)
		{
			current_order = 0 ;
		}
		else
		{
			last_pattern = 1 ;
		}
	}
	
	if (!last_pattern)
	{
		current_pattern = pointer_on_current_module->orders[current_order] ;
		
		if (current_pattern >= pointer_on_current_module->number_of_patterns)
		{
			last_pattern = 1 ;
		}
	}
	
	if (line <= pointer_on_current_pattern->number_of_rows)
	{
		unsigned char local_line_counter ;

		for (local_line_counter = 0 ; local_line_counter < line ; local_line_counter++)
		{
			unsigned char channel_variable ;
			unsigned char mask_variable ;
			unsigned char channel_counter ;

			while ((channel_variable = *(pointer_on_current_pattern_byte++)) != 0)  /* 0 = end of row */
			{
				channel_counter = (channel_variable - 1) & 63 ;

				if (channel_variable & 128)
				{
					real_channels[channel_counter].previous_mask_variable = mask_variable = *(pointer_on_current_pattern_byte++) ;
				}
				else
				{
					mask_variable = real_channels[channel_counter].previous_mask_variable ;
				}

				if (mask_variable & 1)
				{
					real_channels[channel_counter].previous_note_value = *(pointer_on_current_pattern_byte++) ;
				}

				if (mask_variable & 2)
				{
					real_channels[channel_counter].previous_instrument = *(pointer_on_current_pattern_byte++) ;
				}

				if (mask_variable & 4)
				{
					real_channels[channel_counter].previous_volume_panning = *(pointer_on_current_pattern_byte++) ;
				}

				if (mask_variable & 8)
				{
					real_channels[channel_counter].previous_command = *(pointer_on_current_pattern_byte++) ;
					real_channels[channel_counter].previous_command_value = *(pointer_on_current_pattern_byte++) ;
				}
			} /* end while */
		} /* end local_line_counter loop */
	} /* end if */
}

/* new_line reads the next pattern line and updates the channels */
void new_line (void)
{
	/* decodes the new line */
	unsigned char channel_variable ;
	unsigned char new_note ;
	unsigned char note_value ;
	unsigned char new_instrument ;
	unsigned char instrument_value ;
	unsigned char sample_value ;
	unsigned char sample_note ;
	unsigned char new_volume_panning ;
	unsigned char volume_panning ;
	unsigned char new_command ;
	unsigned char command ;
	unsigned char command_value ;
	
	/* global command flags */
	unsigned char pattern_break_requested ;
	unsigned char pattern_break_line ;
	unsigned char new_global_volume_requested ;
	unsigned char new_global_volume ;
	unsigned char new_speed_requested ;
	unsigned char new_speed ;
	unsigned char new_tempo_requested ;
	unsigned char new_tempo ;
	unsigned char global_volume_slide_requested ;
	
	pattern_break_requested = 0 ;
	pattern_break_line = 0 ; /* default line to jump to */
	new_global_volume_requested = 0 ;
	new_global_volume = pointer_on_current_module->global_volume ;
	new_speed_requested = 0 ;
	new_speed = frames_per_line ; /* default speed */
	new_tempo_requested = 0 ;
	new_tempo = beats_per_minute ;
	global_volume_slide_requested = 0 ;
	global_volume_slide_on = 0 ;
	global_volume_slide_value = 0 ;

    while ((channel_variable = *(pointer_on_current_pattern_byte++)) != 0)  /* 0 = end of row */
    {
		p_real_channel pointer_on_channel ;
		p_virtual_channel pointer_on_virtual_channel ;
		unsigned char mask_variable ;
		unsigned char duplicate_check_result ;
		unsigned char new_note_action ;
		unsigned char channel_value ;
		
		note_value = 0 ;
		volume_panning = 0 ;
		command = 0 ;
		command_value = 0 ;
		new_note = new_instrument = new_volume_panning = new_command = 0;
		pointer_on_channel = &(real_channels[(channel_variable - 1) & 63]) ; /* channel is 0 based */
		channel_value = (channel_variable - 1) & 63 ;
		instrument_value = pointer_on_channel->previous_instrument ;
		pointer_on_virtual_channel = pointer_on_channel->foreground_virtual_channel ;
		pointer_on_channel->slide_to_note_direction = 0 ;
		pointer_on_channel->effect_update = 0 ;
		
		if (channel_variable & 128)
		{ /* read byte into mask variable */
			pointer_on_channel->previous_mask_variable = mask_variable = *(pointer_on_current_pattern_byte++) ;
		}
		else
		{ /* use the previous mask variable */
			mask_variable = pointer_on_channel->previous_mask_variable ;
		}
		
		if (mask_variable & 1) /* read note (byte value 0->119, 254 = note cut, 255 = note off) */
		{
			pointer_on_channel->previous_note_value = note_value = *(pointer_on_current_pattern_byte++) ;
			new_note = 1 ;
		}
		
		if (mask_variable & 2) /* read instrument (byte value 0->128) */
		{
			instrument_value = *(pointer_on_current_pattern_byte++) ;
			if (instrument_value == 0)
			{
				instrument_value = pointer_on_channel->previous_instrument ;
			}
			else
			{
				instrument_value-- ;
				pointer_on_channel->previous_instrument = instrument_value ;
			}
			if ((instrument_value >= pointer_on_current_module->number_of_instruments) || (pointer_on_current_module->instruments[instrument_value] == NULL))
			{
				new_note = 0 ;
				new_instrument = 0 ;
				if (pointer_on_channel->playing)
				{
					pointer_on_virtual_channel->channel_playing = 0 ;
					pointer_on_channel->playing = 0 ;
					
					number_of_virtual_channels_allocated-- ;
					
					{
						p_virtual_channel temp_virtual_channel ;
						
						temp_virtual_channel = virtual_channels[number_of_virtual_channels_allocated] ;
						virtual_channels[number_of_virtual_channels_allocated] = virtual_channels[pointer_on_channel->foreground_virtual_channel_number] ;
						virtual_channels[pointer_on_channel->foreground_virtual_channel_number] = temp_virtual_channel ;
						real_channels[temp_virtual_channel->corresponding_real_channel].foreground_virtual_channel_number = pointer_on_channel->foreground_virtual_channel_number ;
					}
				}
			}
			else
			{
				new_instrument = 1 ;
			}
		}
		
		if (mask_variable & 4) /* read volume / panning (byte value) */
		{
			pointer_on_channel->previous_volume_panning = volume_panning = *(pointer_on_current_pattern_byte++) ;
			new_volume_panning = 1 ;
		}
		
		if (mask_variable & 8) /* read command (byte value) and command value (byte value) */
		{
			command = *(pointer_on_current_pattern_byte++) ;
			command_value= *(pointer_on_current_pattern_byte++) ;
			pointer_on_channel->previous_command = command ;
			pointer_on_channel->previous_command_value = command_value ;
			if (command == 1) /* set song speed : A */
			{
				new_speed_requested = 1;
				new_speed = command_value ;
			}
			else if (command == 2) /* jump to order : B */
			{
				if ((command_value < pointer_on_current_module->number_of_orders) && (pointer_on_current_module->orders[command_value] < pointer_on_current_module->number_of_patterns))
				{
					if ((command_value >= current_order)||(loop_allowed))
					{
						pattern_break_requested = 1 ;
						current_order = command_value ;
						current_pattern = pointer_on_current_module->orders[current_order] ;
						pattern_break_line = 0 ;
						last_pattern = 0 ;
					}
					else
					{ /* prevent looping */
						last_pattern = 1 ;
					}
				}
			}
			else if (command == 3) /* pattern break : C */
			{
				pattern_break_requested = 1 ;
				pattern_break_line = command_value ;
			}
			else if (command == 20) /* set song tempo : T */
			{
				new_tempo_requested = 1;
				if (command_value >= 0x20 )
				{
					new_tempo = command_value ;
				}
				else
				{
					if (command_value < 0x10 )
					{
						signed short tempo ;
						tempo = ((signed short)new_tempo) - ((signed short)command_value) ;
						if (tempo < 0x20)
						{
							tempo = 0x20 ;
						}
						new_tempo = (unsigned char) tempo ;
					}
					else
					{
						unsigned short tempo ;
						tempo = ((unsigned short)new_tempo) + ((unsigned short)command_value - 0x10) ;
						if (tempo > 0xFF)
						{
							tempo = 0xFF ;
						}
						new_tempo = (unsigned char)tempo ;
					}
				}
			}
			else if ((command == 22) && (command_value<=0x80)) /* set global volume : V */
			{
				new_global_volume_requested = 1;
				new_global_volume = command_value ;
			}
			else if (command == 23) /* global volume slide : w */
			{
				global_volume_slide_requested = 1;
				global_volume_slide_value = command_value ;
			}
			else
			{ /* new command is set only for channels commands */
				new_command = 1 ;
			}
		}
		
		if (mask_variable & 16) /* use previous note value */
		{
			note_value = pointer_on_channel->previous_note_value ;
			new_note = 1 ;
		}
		
		if (mask_variable & 32) /* use previous instrument */
		{
			instrument_value = pointer_on_channel->previous_instrument ;
			if ((instrument_value >= pointer_on_current_module->number_of_instruments) || (pointer_on_current_module->instruments[instrument_value] == NULL))
			{
				new_note = 0 ;
				new_instrument = 0 ;
				if (pointer_on_channel->playing)
				{
					pointer_on_virtual_channel->channel_playing = 0 ;
					pointer_on_channel->playing = 0 ;
					
					number_of_virtual_channels_allocated-- ;
					
					{
						p_virtual_channel temp_virtual_channel ;
						
						temp_virtual_channel = virtual_channels[number_of_virtual_channels_allocated] ;
						virtual_channels[number_of_virtual_channels_allocated] = virtual_channels[pointer_on_channel->foreground_virtual_channel_number] ;
						virtual_channels[pointer_on_channel->foreground_virtual_channel_number] = temp_virtual_channel ;
						real_channels[temp_virtual_channel->corresponding_real_channel].foreground_virtual_channel_number = pointer_on_channel->foreground_virtual_channel_number ;
					}
				}
			}
			else
			{
				new_instrument = 1 ;
			}
		}
		
		if (mask_variable & 64) /* use previous volume / panning */
		{
			volume_panning = pointer_on_channel->previous_volume_panning ;
			new_volume_panning = 1 ;
		}
		
		if (mask_variable & 128) /* use previous command and command value */
		{
			command = pointer_on_channel->previous_command ;
			command_value = pointer_on_channel->previous_command_value ;
			if (command == 1) /* set song speed : A */
			{
				new_speed_requested = 1;
				new_speed = command_value ;
			}
			else if (command == 2) /* jump to order : B */
			{
				if ((command_value < pointer_on_current_module->number_of_orders) && (pointer_on_current_module->orders[command_value] < pointer_on_current_module->number_of_patterns))
				{
					pattern_break_requested = 1 ;
					current_order = command_value ;
					current_pattern = pointer_on_current_module->orders[current_order] ;
					pattern_break_line = 0 ;
					last_pattern = 0 ;
				}
			}
			else if (command == 3) /* pattern break : C */
			{
				pattern_break_requested = 1 ;
				pattern_break_line = command_value ;
			}
			else if (command == 23) /* global volume slide : w */
			{
				global_volume_slide_requested = 1;
				global_volume_slide_value = command_value ;
			}
			else if (command == 20) /* set song tempo : T */
			{
				new_tempo_requested = 1;
				if (command_value >= 0x20 )
				{
					new_tempo = command_value ;
				}
				else
				{
					if (command_value < 0x10 )
					{
						signed short tempo ;
						tempo = ((signed short)new_tempo) - ((signed short)command_value) ;
						if (tempo < 0x20)
						{
							tempo = 0x20 ;
						}
						new_tempo = (unsigned char)tempo ;
					}
					else
					{
						unsigned short tempo ;
						tempo = ((unsigned short)new_tempo) + ((unsigned short)command_value - 0x10) ;
						if (tempo > 0xFF)
						{
							tempo = 0xFF ;
						}
						new_tempo = (unsigned char)tempo ;
					}
				}
			}
			else
			{
				new_command = 1 ;
			}
		}
		
		if (pointer_on_channel->muted_channel)   /* muted channels handling */
		{
			continue ;
		}
		
		if ((new_command == 1))
		{
			if ((pointer_on_channel->playing == 1)&&((command == 7)||(command == 12))) /* Command G, and dual command L */
			{
				if (new_note == 1)
				{
					pointer_on_channel->last_slide_to_note_note = note_value ;
				}
				
				if ((command != 12) && (command_value != 0))
				{
					pointer_on_channel->last_slide_to_note_speed = command_value ;
				}
				
				if (new_instrument == 1)
				{
					p_instrument current_instrument ;
					p_sample current_sample ;
					
					if (instrument_value != pointer_on_channel->last_instrument_used)
					{
						note_value = pointer_on_channel->last_note_used ;
						
						sample_value = pointer_on_current_module->instruments[instrument_value]->note_sample_keyboard_table[note_value].sample - 1 ;
						sample_note = pointer_on_current_module->instruments[instrument_value]->note_sample_keyboard_table[note_value].note ;
						
						pointer_on_virtual_channel->instrument_of_channel = instrument_value ;
						pointer_on_virtual_channel->sample_of_channel = sample_value ;
						current_instrument = pointer_on_virtual_channel->pointer_on_instrument = pointer_on_current_module->instruments[instrument_value] ;
						current_sample = pointer_on_virtual_channel->pointer_on_sample = pointer_on_current_module->samples[sample_value] ;
						
						pointer_on_channel->current_duplicate_check_type = current_instrument->duplicate_check_type ;
						pointer_on_channel->current_duplicate_check_action = current_instrument->duplicate_check_action ;
						pointer_on_virtual_channel->pointer_in_sample_of_channel = pointer_on_current_module->samples[sample_value]->sample_data ;
						pointer_on_virtual_channel->pointer_on_sample_beginning = pointer_on_current_module->samples[sample_value]->sample_data ;
						pointer_on_virtual_channel->pointer_on_sample_end = pointer_on_current_module->samples[sample_value]->sample_data + (pointer_on_current_module->samples[sample_value]->length - 1);
						pointer_on_virtual_channel->direction = 1 ;
						pointer_on_virtual_channel->frac_position_in_sample_of_channel = 0 ;
						pointer_on_virtual_channel->note_table_value = (unsigned long)precalculated_notes_table[sample_note * 64l] ;
						pointer_on_virtual_channel->step_length_for_sample_of_channel = (unsigned long) (current_sample->adjusted_C5_speed * precalculated_notes_table[sample_note * 64l]) ;
						pointer_on_virtual_channel->pointer_on_loop_beginning = current_sample->sample_data + current_sample->loop_begin ;
						pointer_on_virtual_channel->pointer_on_loop_end = current_sample->sample_data + current_sample->loop_end  ;
						pointer_on_virtual_channel->use_loop = current_sample->use_loop ;
						pointer_on_virtual_channel->use_ping_pong_loop = current_sample->loop_direction ;
						
						pointer_on_virtual_channel->pointer_on_sustain_loop_beginning = pointer_on_virtual_channel->pointer_in_sample_of_channel + current_sample->sustain_loop_begin ;
						pointer_on_virtual_channel->pointer_on_sustain_loop_end = pointer_on_virtual_channel->pointer_in_sample_of_channel + current_sample->sustain_loop_end  ;
						pointer_on_virtual_channel->use_sustain_loop = current_sample->use_sustain_loop ;
						pointer_on_virtual_channel->use_ping_pong_sustain_loop = current_sample->sustain_loop_direction ;
						pointer_on_channel->last_instrument_used = instrument_value ;
						pointer_on_channel->last_note_used = note_value ;
						pointer_on_channel->last_sample_note_used = sample_note ;
						pointer_on_channel->last_sub_note_used = 0 ;
						pointer_on_channel->last_instrument_used = instrument_value ;
						pointer_on_channel->last_sample_used = sample_value ;
					}
					else
					{
						current_instrument = pointer_on_virtual_channel->pointer_on_instrument ;
						current_sample = pointer_on_virtual_channel->pointer_on_sample ;
					}
					
					pointer_on_virtual_channel->volume = current_sample->default_volume ;
					pointer_on_virtual_channel->volume_of_instrument = current_instrument->global_volume ;
					pointer_on_virtual_channel->panning_of_instrument = current_instrument->default_panning ;
					pointer_on_virtual_channel->volume_of_sample = current_sample->global_volume ;
					pointer_on_virtual_channel->volume_of_channel = pointer_on_channel->volume_of_channel ;
					pointer_on_virtual_channel->panning_of_channel = ((pointer_on_virtual_channel->panning_of_instrument) > 64)?(pointer_on_channel->panning_of_channel):(((signed short)pointer_on_virtual_channel->panning_of_instrument) - 32) ;
					pointer_on_virtual_channel->volume_envelope = &(current_instrument->volume_envelope) ;
					pointer_on_virtual_channel->interpolated_volume_envelope = current_instrument->interpolated_volume_envelope ;
					pointer_on_virtual_channel->volume_envelope_tick = 0;
					pointer_on_virtual_channel->volume_envelope_value = 64 ;
					if (pointer_on_virtual_channel->volume_envelope->envelope_on_off)
					{
						pointer_on_virtual_channel->volume_envelope_value = pointer_on_virtual_channel->interpolated_volume_envelope[pointer_on_virtual_channel->volume_envelope_tick] ;
					}
					pointer_on_virtual_channel->panning_envelope = &(current_instrument->panning_envelope) ;
					pointer_on_virtual_channel->interpolated_panning_envelope = current_instrument->interpolated_panning_envelope ;
					pointer_on_virtual_channel->panning_envelope_tick = 0 ;
					if (pointer_on_virtual_channel->panning_envelope->envelope_on_off)
					{
						pointer_on_virtual_channel->panning_of_channel = pointer_on_virtual_channel->interpolated_panning_envelope[pointer_on_virtual_channel->panning_envelope_tick] ;
					}
					{
						pointer_on_virtual_channel->volume_envelope = &(current_instrument->volume_envelope) ;
						pointer_on_virtual_channel->interpolated_volume_envelope = current_instrument->interpolated_volume_envelope ;
						pointer_on_virtual_channel->volume_envelope_tick = 0;
						pointer_on_virtual_channel->volume_envelope_value = 64 ;
						if (pointer_on_virtual_channel->volume_envelope->envelope_on_off)
						{
							pointer_on_virtual_channel->volume_envelope_value = pointer_on_virtual_channel->interpolated_volume_envelope[pointer_on_virtual_channel->volume_envelope_tick] ;
						}
						
						pointer_on_virtual_channel->panning_envelope = &(current_instrument->panning_envelope) ;
						pointer_on_virtual_channel->interpolated_panning_envelope = current_instrument->interpolated_panning_envelope ;
						pointer_on_virtual_channel->panning_envelope_tick = 0 ;
						if (pointer_on_virtual_channel->panning_envelope->envelope_on_off)
						{
							pointer_on_virtual_channel->panning_of_channel = pointer_on_virtual_channel->interpolated_panning_envelope[pointer_on_virtual_channel->panning_envelope_tick] ;
						}
						
					}
					pointer_on_virtual_channel->sustain = 1 ;
					pointer_on_virtual_channel->note_fade_component = 1024 ;
					pointer_on_virtual_channel->fade = 0 ;
					pointer_on_virtual_channel->end_of_volume_envelope = 0 ;
					pointer_on_virtual_channel->end_of_panning_envelope = 0 ;
          }
		  
		  
          if (pointer_on_channel->last_slide_to_note_note == pointer_on_channel->last_note_used)
          {
			  pointer_on_channel->slide_to_note_direction = 0 ;
          }
          else if (pointer_on_channel->last_slide_to_note_note > pointer_on_channel->last_note_used)
          {
			  new_note = 0 ;
			  new_instrument = 0 ;
			  pointer_on_channel->effect_update = 1 ;
			  pointer_on_channel->slide_to_note_direction = 1 ;
          }
          else
          {
			  new_note = 0 ;
			  new_instrument = 0 ;
			  pointer_on_channel->effect_update = 1 ;
			  pointer_on_channel->slide_to_note_direction = -1 ;
          }
		  
        }
        else if (command == 19) /* Command S */
        {
			if (command_value == 0x70) /* past note cut */
			{
				unsigned short local_channel_counter ;
				
				for (local_channel_counter = 0 ; local_channel_counter < number_of_virtual_channels_allocated ; local_channel_counter++)
				{
					if (virtual_channels[local_channel_counter]->corresponding_real_channel == channel_value)
					{/* note cut*/
						if (volume_ramp_on_off)
						{
							pointer_on_virtual_channel->use_ramp_down = 1 ;
							pointer_on_virtual_channel->ramp_position = 0 ;
							pointer_on_virtual_channel->end_of_sample = 1 ;
						}
						else
						{
							virtual_channels[local_channel_counter]->channel_playing = 0 ;
							if (virtual_channels[local_channel_counter]->channel_in_background == 0)
							{
								real_channels[virtual_channels[local_channel_counter]->corresponding_real_channel].playing = 0 ;
							}
							
							number_of_virtual_channels_allocated-- ;
							
							{
								p_virtual_channel temp_virtual_channel ;
								
								temp_virtual_channel = virtual_channels[number_of_virtual_channels_allocated] ;
								virtual_channels[number_of_virtual_channels_allocated] = virtual_channels[local_channel_counter] ;
								virtual_channels[local_channel_counter] = temp_virtual_channel ;
								real_channels[temp_virtual_channel->corresponding_real_channel].foreground_virtual_channel_number = (unsigned char)local_channel_counter ;
							}
						}
					} /* end note cut */
					
				} /* end for */
			} /* end command_value == 0x70 */
			else if (command_value == 0x71) /* past note off */
			{
				unsigned short local_channel_counter ;
				
				for (local_channel_counter = 0 ; local_channel_counter < number_of_virtual_channels_allocated ; local_channel_counter++)
				{
					if (virtual_channels[local_channel_counter]->corresponding_real_channel == channel_value)
					{ /* note off */
						virtual_channels[local_channel_counter]->sustain = 0 ;
						
						if (virtual_channels[local_channel_counter]->use_sustain_loop)
						{
							if (virtual_channels[local_channel_counter]->use_loop)
							{
								virtual_channels[local_channel_counter]->pointer_in_sample_of_channel = virtual_channels[local_channel_counter]->pointer_on_loop_beginning ;
								virtual_channels[local_channel_counter]->frac_position_in_sample_of_channel = 0 ;
								virtual_channels[local_channel_counter]->direction = 1 ;
							}
						}
						
						if (virtual_channels[local_channel_counter]->volume_envelope->envelope_on_off)
						{
							if ((virtual_channels[local_channel_counter]->volume_envelope->sustain_loop_on_off) && (virtual_channels[local_channel_counter]->volume_envelope->loop_on_off))
							{
								virtual_channels[local_channel_counter]->volume_envelope_tick = virtual_channels[local_channel_counter]->volume_envelope->loop_begin_tick ;
							}
							if (virtual_channels[local_channel_counter]->volume_envelope->loop_on_off)
							{
								virtual_channels[local_channel_counter]->fade = 1 ;
							}
						}
						else
						{
							virtual_channels[local_channel_counter]->fade = 1 ;
						}
						
						if (virtual_channels[local_channel_counter]->panning_envelope->envelope_on_off)
						{
							if ((virtual_channels[local_channel_counter]->panning_envelope->sustain_loop_on_off) && (virtual_channels[local_channel_counter]->panning_envelope->loop_on_off))
							{
								virtual_channels[local_channel_counter]->panning_envelope_tick = virtual_channels[local_channel_counter]->panning_envelope->loop_begin_tick ;
							}
						}
					} /* end note off */
					
				} /* end for */
			} /* end command_value == 0x71 */
			else if (command_value == 0x72) /* past note fade */
			{
				unsigned short local_channel_counter ;
				
				for (local_channel_counter = 0 ; local_channel_counter < number_of_virtual_channels_allocated ; local_channel_counter++)
				{
					if (virtual_channels[local_channel_counter]->corresponding_real_channel == channel_value)
					{
						virtual_channels[local_channel_counter]->fade = 1 ;
					}
				} /* end for */
			}/* end command_value == 0x72 */
        }
      }
	  
	  
      if ((instrument_value >= pointer_on_current_module->number_of_instruments) || (pointer_on_current_module->instruments[instrument_value] == NULL) && (note_value < 254))
      {
		  {
			  new_note = 0 ;
			  new_instrument = 0 ;
			  if (pointer_on_channel->playing)
			  {
				  pointer_on_virtual_channel->channel_playing = 0 ;
				  pointer_on_channel->playing = 0 ;
				  
				  number_of_virtual_channels_allocated-- ;
				  
				  {
					  p_virtual_channel temp_virtual_channel ;
					  
					  temp_virtual_channel = virtual_channels[number_of_virtual_channels_allocated] ;
					  virtual_channels[number_of_virtual_channels_allocated] = virtual_channels[pointer_on_channel->foreground_virtual_channel_number] ;
					  virtual_channels[pointer_on_channel->foreground_virtual_channel_number] = temp_virtual_channel ;
					  real_channels[temp_virtual_channel->corresponding_real_channel].foreground_virtual_channel_number = pointer_on_channel->foreground_virtual_channel_number ;
				  }
			  }
		  }
      }
	  
      new_note_action = 0 ;
	  
      if (   ((new_note==1) || (new_instrument==1))
          && (note_value <= 119)
          && (instrument_value < pointer_on_current_module->number_of_instruments)
          && (pointer_on_current_module->instruments[instrument_value] != 0)
		  )
      {
		  sample_value = pointer_on_current_module->instruments[instrument_value]->note_sample_keyboard_table[note_value].sample - 1 ;
		  sample_note = pointer_on_current_module->instruments[instrument_value]->note_sample_keyboard_table[note_value].note ;
		  
		  if (pointer_on_channel->playing == 1)
		  {
			  if ((pointer_on_channel->current_new_note_action == 0)||(pointer_on_channel->current_duplicate_check_type == 0)) /* DCT off */
			  {
				  duplicate_check_result = 0 ;
			  }
			  else if (pointer_on_channel->current_duplicate_check_type == 1)/* DCT = NOTE */
			  {
				  if (pointer_on_channel->last_note_used == note_value)
				  {
					  duplicate_check_result = 1 ;
				  }
				  else
				  {
					  duplicate_check_result = 0 ;
				  }
			  }
			  else if (pointer_on_channel->current_duplicate_check_type == 2)/* DCT = SAMPLE */
			  {
				  if (pointer_on_channel->last_sample_used == sample_value)
				  {
					  duplicate_check_result = 1 ;
				  }
				  else
				  {
					  duplicate_check_result = 0 ;
				  }
			  }
			  else if (pointer_on_channel->current_duplicate_check_type == 3)/* DCT = INSTRUMENT */
			  {
				  if (pointer_on_channel->last_instrument_used == instrument_value)
				  {
					  duplicate_check_result = 1 ;
				  }
				  else
				  {
					  duplicate_check_result = 0 ;
				  }
			  }
			  
			  if (duplicate_check_result == 0)
			  {
				  new_note_action = pointer_on_channel->current_new_note_action ;
			  }
			  else
			  {
				  new_note_action = pointer_on_channel->current_duplicate_check_action ;
				  if (new_note_action > 0)
				  {
					  new_note_action++ ;
				  }
			  }
		  }
      }
      /* new note handling */
	  
      if (new_note == 1)
      {
		  if (note_value < 120) /* note_value is a true note */
		  {
			  if (     (!volume_ramp_on_off)
				  && (new_note_action == 0)/* = note cut */
				  && (pointer_on_channel->playing == 1) /* virtual channel is valid */
				  )
			  {
				  pointer_on_virtual_channel = pointer_on_channel->foreground_virtual_channel ;
				  pointer_on_virtual_channel->use_last_sound_value = 1 ;
			  }
			  else
			  {/* a new virtual channel must be allocated */
				  if (
					  (new_note_action == 0)/* = note cut */
					  &&
					  (pointer_on_channel->playing == 1) /* virtual channel is valid */
					  )
				  {
					  pointer_on_virtual_channel->use_ramp_down = 1 ;
					  pointer_on_virtual_channel->end_of_sample = 1 ;
					  pointer_on_virtual_channel->ramp_position = 0 ;
				  }
				  if ((new_note_action == 2) && (pointer_on_channel->playing == 1)) /* note off */
				  {
					  pointer_on_virtual_channel->sustain = 0 ;
					  
					  if (pointer_on_virtual_channel->use_sustain_loop)
					  {
						  if (pointer_on_virtual_channel->use_loop)
						  {
							  pointer_on_virtual_channel->pointer_in_sample_of_channel = pointer_on_virtual_channel-> pointer_on_loop_beginning ;
							  pointer_on_virtual_channel->frac_position_in_sample_of_channel = 0 ;
							  pointer_on_virtual_channel->direction = 1 ;
						  }
					  }
					  
					  if (pointer_on_virtual_channel->volume_envelope->envelope_on_off)
					  {
						  if ((pointer_on_virtual_channel->volume_envelope->sustain_loop_on_off)&& (pointer_on_virtual_channel->volume_envelope->loop_on_off))
						  {
							  pointer_on_virtual_channel->volume_envelope_tick = pointer_on_virtual_channel->volume_envelope->loop_begin_tick ;
						  }
						  
						  if (pointer_on_virtual_channel->volume_envelope->loop_on_off)
						  {
							  pointer_on_virtual_channel->fade = 1 ;
						  }
					  }
					  else
					  {
						  pointer_on_virtual_channel->fade = 1 ;
					  }
					  
					  if (pointer_on_virtual_channel->panning_envelope->envelope_on_off)
					  {
						  if ((pointer_on_virtual_channel->panning_envelope->sustain_loop_on_off)&& (pointer_on_virtual_channel->panning_envelope->loop_on_off))
						  {
							  pointer_on_virtual_channel->panning_envelope_tick = pointer_on_virtual_channel->panning_envelope->loop_begin_tick ;
						  }
					  }
				  }
				  if ((new_note_action == 3) && (pointer_on_channel->playing == 1)) /* note fade */
				  {
					  pointer_on_virtual_channel->fade = 1 ;
				  }
				  
				  if (pointer_on_channel->playing == 1)
				  { /* update the previous virtual channel */
					  p_virtual_channel previous_virtual_channel ;
					  
					  previous_virtual_channel = pointer_on_channel->foreground_virtual_channel ;
					  previous_virtual_channel->channel_in_background = 1 ;
				  }
				  
				  if (number_of_virtual_channels_allocated < max_number_of_virtual_channels_allocated)
				  {
					  pointer_on_virtual_channel = virtual_channels[number_of_virtual_channels_allocated] ;
					  pointer_on_channel->foreground_virtual_channel_number = (unsigned char)number_of_virtual_channels_allocated ;
					  pointer_on_channel->foreground_virtual_channel = pointer_on_virtual_channel;
					  number_of_virtual_channels_allocated ++ ;
					  pointer_on_virtual_channel->ramp_position = 0 ;
					  pointer_on_virtual_channel->use_ramp_down = 0 ;
					  pointer_on_virtual_channel->new_sample = 1 ;
					  pointer_on_virtual_channel->end_of_sample = 0 ;
				  }
				  else
				  {
					  new_note = 0 ;
					  new_instrument = 0 ;
				  }
			  }
			  if (new_note == 1)
			  {
				  pointer_on_channel->playing = 1 ;
				  pointer_on_virtual_channel->channel_playing = 1 ;
				  pointer_on_virtual_channel->channel_in_background = 0 ;
				  pointer_on_virtual_channel->corresponding_real_channel = (channel_variable - 1) & 63 ;
				  pointer_on_virtual_channel->ramp_position = 0 ;
				  pointer_on_virtual_channel->new_sample = 1 ;
				  pointer_on_virtual_channel->end_of_sample = 0 ;
				  pointer_on_virtual_channel->use_ramp_down = 0 ;
			  }
        }
        else
        { /* note cut / note off */
			if ((note_value >= 254) && (pointer_on_channel->playing == 1))
			{
				if (note_value == 254)
				{ /* note cut */
					if (volume_ramp_on_off)
					{
						pointer_on_virtual_channel->use_ramp_down = 1 ;
						pointer_on_virtual_channel->end_of_sample = 1 ;
						pointer_on_virtual_channel->ramp_position = 0 ;
					}
					else
					{
						pointer_on_channel->foreground_virtual_channel->channel_playing = 0 ;
						pointer_on_channel->playing = 0 ;
						number_of_virtual_channels_allocated-- ;
						
						{
							p_virtual_channel temp_virtual_channel ;
							
							temp_virtual_channel = virtual_channels[number_of_virtual_channels_allocated] ;
							virtual_channels[number_of_virtual_channels_allocated] = virtual_channels[pointer_on_channel->foreground_virtual_channel_number] ;
							virtual_channels[pointer_on_channel->foreground_virtual_channel_number] = temp_virtual_channel ;
							real_channels[temp_virtual_channel->corresponding_real_channel].foreground_virtual_channel_number = pointer_on_channel->foreground_virtual_channel_number ;
						}
					}
				} /* end note cut */
				else
				{ /* note off */
					pointer_on_virtual_channel->sustain = 0 ;
					
					if (pointer_on_virtual_channel->use_sustain_loop)
					{
						if (pointer_on_virtual_channel->use_loop)
						{
							pointer_on_virtual_channel->pointer_in_sample_of_channel = pointer_on_virtual_channel-> pointer_on_loop_beginning ;
							pointer_on_virtual_channel->frac_position_in_sample_of_channel = 0 ;
							pointer_on_virtual_channel->direction = 1 ;
						}
					}
					
					if (pointer_on_virtual_channel->volume_envelope->envelope_on_off)
					{
						if ((pointer_on_virtual_channel->volume_envelope->sustain_loop_on_off) && (pointer_on_virtual_channel->volume_envelope->loop_on_off))
						{
							pointer_on_virtual_channel->volume_envelope_tick = pointer_on_virtual_channel->volume_envelope->loop_begin_tick ;
						}
						
						if (pointer_on_virtual_channel->volume_envelope->loop_on_off)
						{
							pointer_on_virtual_channel->fade = 1 ;
						}
					}
					else
					{
						pointer_on_virtual_channel->fade = 1 ;
					}
					
					if (pointer_on_virtual_channel->panning_envelope->envelope_on_off)
					{
						if ((pointer_on_virtual_channel->panning_envelope->sustain_loop_on_off)&& (pointer_on_virtual_channel->panning_envelope->loop_on_off))
						{
							pointer_on_virtual_channel->panning_envelope_tick = pointer_on_virtual_channel->panning_envelope->loop_begin_tick ;
						}
					}
				} /* end note off */
			}
        }
      }  /* end of new note handling */
	  
	  /* new note or new instrument common handling */
      if (   ((new_note==1) || (new_instrument==1))
          && (pointer_on_channel->playing == 1)
          && (note_value <= 119)
          && (instrument_value < pointer_on_current_module->number_of_instruments)
          && (pointer_on_current_module->instruments[instrument_value] != 0)
		  )
      {
		  if (    (sample_value >= pointer_on_current_module->number_of_samples)
			  || (pointer_on_current_module->samples[sample_value] == NULL)
			  || (pointer_on_current_module->samples[sample_value]->length == 0)
			  )
		  {
			  sample_value = pointer_on_virtual_channel->sample_of_channel ;
			  pointer_on_virtual_channel->use_ramp_down = 0 ;
			  pointer_on_virtual_channel->ramp_position = 0 ;
			  pointer_on_virtual_channel->new_sample = 1 ;
			  pointer_on_virtual_channel->end_of_sample = 0 ;
			  pointer_on_virtual_channel->pointer_in_sample_of_channel = pointer_on_current_module->samples[sample_value]->sample_data ;
			  pointer_on_virtual_channel->pointer_on_sample_beginning = pointer_on_current_module->samples[sample_value]->sample_data ;
			  pointer_on_virtual_channel->pointer_on_sample_end = pointer_on_current_module->samples[sample_value]->sample_data + (pointer_on_current_module->samples[sample_value]->length - 1);
			  pointer_on_virtual_channel->direction = 1 ;
			  pointer_on_virtual_channel->frac_position_in_sample_of_channel = 0 ;
			  pointer_on_virtual_channel->note_table_value = (unsigned long)precalculated_notes_table[sample_note * 64l] ;
			  pointer_on_virtual_channel->step_length_for_sample_of_channel = (unsigned long)(pointer_on_current_module->samples[sample_value]->adjusted_C5_speed * precalculated_notes_table[sample_note * 64l]);
		  }
		  else
		  {
			  p_instrument current_instrument ;
			  p_sample current_sample ;

			  pointer_on_virtual_channel->use_ramp_down = 0 ;
			  pointer_on_virtual_channel->ramp_position = 0 ;
			  pointer_on_virtual_channel->new_sample = 1 ;
			  pointer_on_virtual_channel->end_of_sample = 0 ;
			  pointer_on_virtual_channel->instrument_of_channel = instrument_value ;
			  pointer_on_virtual_channel->sample_of_channel = sample_value ;
			  current_instrument = pointer_on_virtual_channel->pointer_on_instrument = pointer_on_current_module->instruments[instrument_value] ;
			  current_sample = pointer_on_virtual_channel->pointer_on_sample = pointer_on_current_module->samples[sample_value] ;
			  
			  pointer_on_virtual_channel->volume = current_sample->default_volume ;
			  pointer_on_virtual_channel->volume_of_instrument = current_instrument->global_volume ;
			  pointer_on_virtual_channel->panning_of_instrument = current_instrument->default_panning ;
			  if (new_note_actions_activated)
			  {
				  pointer_on_channel->current_new_note_action = current_instrument->new_note_action ;
			  }
			  else
			  {
				  pointer_on_channel->current_new_note_action = 0 ;
			  }
			  pointer_on_channel->current_duplicate_check_type = current_instrument->duplicate_check_type ;
			  pointer_on_channel->current_duplicate_check_action = current_instrument->duplicate_check_action ;
			  pointer_on_virtual_channel->volume_of_sample = current_sample->global_volume ;
			  pointer_on_virtual_channel->volume_of_channel = pointer_on_channel->volume_of_channel ;
			  pointer_on_virtual_channel->panning_of_channel = ((pointer_on_virtual_channel->panning_of_instrument) > 64)?(pointer_on_channel->panning_of_channel):(((signed short)pointer_on_virtual_channel->panning_of_instrument) - 32) ;
			  pointer_on_virtual_channel->pointer_in_sample_of_channel = pointer_on_current_module->samples[sample_value]->sample_data ;
			  pointer_on_virtual_channel->pointer_on_sample_beginning = pointer_on_current_module->samples[sample_value]->sample_data ;
			  pointer_on_virtual_channel->pointer_on_sample_end = pointer_on_current_module->samples[sample_value]->sample_data + (pointer_on_current_module->samples[sample_value]->length - 1);
			  pointer_on_virtual_channel->direction = 1 ;
			  pointer_on_virtual_channel->frac_position_in_sample_of_channel = 0 ;
			  pointer_on_virtual_channel->note_table_value = (unsigned long)precalculated_notes_table[sample_note * 64l] ;
			  pointer_on_virtual_channel->step_length_for_sample_of_channel = (unsigned long)(pointer_on_current_module->samples[sample_value]->adjusted_C5_speed * precalculated_notes_table[sample_note * 64l]);
			  pointer_on_virtual_channel->pointer_on_loop_beginning = current_sample->sample_data + current_sample->loop_begin ;
			  pointer_on_virtual_channel->pointer_on_loop_end = current_sample->sample_data + current_sample->loop_end  ;
			  pointer_on_virtual_channel->use_loop = current_sample->use_loop ;
			  pointer_on_virtual_channel->use_ping_pong_loop = current_sample->loop_direction ;
			  
			  pointer_on_virtual_channel->pointer_on_sustain_loop_beginning = pointer_on_virtual_channel->pointer_in_sample_of_channel + current_sample->sustain_loop_begin ;
			  pointer_on_virtual_channel->pointer_on_sustain_loop_end = pointer_on_virtual_channel->pointer_in_sample_of_channel + current_sample->sustain_loop_end  ;
			  pointer_on_virtual_channel->use_sustain_loop = current_sample->use_sustain_loop ;
			  pointer_on_virtual_channel->use_ping_pong_sustain_loop = current_sample->sustain_loop_direction ;
			  pointer_on_channel->last_note_used = note_value ;
			  pointer_on_channel->last_sample_note_used = sample_note ;
			  pointer_on_channel->last_sub_note_used = 0 ;
			  pointer_on_channel->last_instrument_used = instrument_value ;
			  pointer_on_channel->last_sample_used = sample_value ;
		  }
		  
		  /* envelopes */
		  {
			  p_instrument current_instrument ;
			  
			  current_instrument = pointer_on_virtual_channel->pointer_on_instrument ;
			  
			  pointer_on_virtual_channel->volume_envelope = &(current_instrument->volume_envelope) ;
			  pointer_on_virtual_channel->interpolated_volume_envelope = current_instrument->interpolated_volume_envelope ;
			  pointer_on_virtual_channel->volume_envelope_tick = 0;
			  pointer_on_virtual_channel->volume_envelope_value = 64 ;
			  if (pointer_on_virtual_channel->volume_envelope->envelope_on_off)
			  {
				  pointer_on_virtual_channel->volume_envelope_value = pointer_on_virtual_channel->interpolated_volume_envelope[pointer_on_virtual_channel->volume_envelope_tick] ;
			  }
			  
			  pointer_on_virtual_channel->panning_envelope = &(current_instrument->panning_envelope) ;
			  pointer_on_virtual_channel->interpolated_panning_envelope = current_instrument->interpolated_panning_envelope ;
			  pointer_on_virtual_channel->panning_envelope_tick = 0 ;
			  if (pointer_on_virtual_channel->panning_envelope->envelope_on_off)
			  {
				  pointer_on_virtual_channel->panning_of_channel = pointer_on_virtual_channel->interpolated_panning_envelope[pointer_on_virtual_channel->panning_envelope_tick] ;
			  }
		  }
		  
		  pointer_on_virtual_channel->sustain = 1 ;
		  pointer_on_virtual_channel->note_fade_component = 1024 ;
		  pointer_on_virtual_channel->fade = 0 ;
		  pointer_on_virtual_channel->end_of_volume_envelope = 0 ;
		  pointer_on_virtual_channel->end_of_panning_envelope = 0 ;
      }
	  /* end of new note or new instrument common handling */
	  
	  if ((new_volume_panning == 1)&&(pointer_on_channel->playing == 1))
	  {
		  if (volume_panning <= 64)
		  {
			  pointer_on_virtual_channel->volume = volume_panning ;
		  }
		  else if ((volume_panning >= 128) && (volume_panning <= 192))
		  {
			  pointer_on_channel->panning_of_channel = (signed char)(((signed short)volume_panning) - 160) ;
			  pointer_on_virtual_channel->panning_of_channel = (signed char)(((signed short)volume_panning) - 160) ;
		  }
	  }
	  
	  
	  if (new_command == 1)
	  {
		  if ((command == 13)&&(command_value<=0x40)) /* set channel volume : M */
		  {
			  pointer_on_channel->volume_of_channel = command_value ;
			  if (pointer_on_channel->playing == 1)
			  {
				  pointer_on_virtual_channel->volume_of_channel = command_value ;
			  }
		  }
		  else if (command == 14) /* channel volume slide : N */
		  {
			  if (command_value == 0)
			  {
				  pointer_on_channel->previous_command_value = command_value = pointer_on_channel->last_channel_volume_slide_parameter ;
			  }
			  else
			  {
				  pointer_on_channel->last_channel_volume_slide_parameter = command_value ;
			  }
			  if (!(command_value & 0x0F)) /* channel volume slide up */
			  {
				  pointer_on_channel->effect_update = 1 ;
				  pointer_on_channel->slide_value = command_value >> 4 ;
			  }
			  else if (!(command_value & 0xF0)) /* channel volume slide down */
			  {
				  pointer_on_channel->effect_update = 1 ;
				  pointer_on_channel->slide_value = command_value & 0x0F ;
			  }
			  else if ((command_value & 0x0F) == 0x0F) /* fine channel volume slide up */
			  {
				  unsigned char slide_value ;
				  slide_value = command_value >> 4 ;
				  pointer_on_channel->volume_of_channel = ((pointer_on_channel->volume_of_channel + slide_value)<=64)?(pointer_on_channel->volume_of_channel + slide_value):64 ;
				  if (pointer_on_channel->playing == 1)
				  {
					  pointer_on_virtual_channel->volume_of_channel = pointer_on_channel->volume_of_channel ;
				  }
			  }
			  else if ((command_value & 0xF0) == 0xF0) /* fine channel volume slide down */
			  {
				  signed char slide_value ;
				  slide_value = command_value & 0x0F ;
				  pointer_on_channel->volume_of_channel = ((((signed short)pointer_on_channel->volume_of_channel) - slide_value)>=0)?(((signed short)pointer_on_channel->volume_of_channel) - slide_value):0 ;
				  if (pointer_on_channel->playing == 1)
				  {
					  pointer_on_virtual_channel->volume_of_channel = pointer_on_channel->volume_of_channel ;
				  }
			  }
		  }
		  else if (command == 16) /* panning slide : P */
		  {
			  if (command_value == 0)
			  {
				  pointer_on_channel->previous_command_value = command_value = pointer_on_channel->last_panning_slide_parameter ;
			  }
			  else
			  {
				  pointer_on_channel->last_panning_slide_parameter = command_value ;
			  }
			  if (!(command_value & 0x0F)) /* panning slide left */
			  {
				  pointer_on_channel->effect_update = 1 ;
				  pointer_on_channel->slide_value = command_value >> 4 ;
			  }
			  else if (!(command_value & 0xF0)) /* panning slide right */
			  {
				  pointer_on_channel->effect_update = 1 ;
				  pointer_on_channel->slide_value = command_value & 0x0F ;
			  }
			  else if ((command_value & 0x0F) == 0x0F) /* fine panning slide left */
			  {
				  signed char slide_value ;
				  slide_value = command_value & 0x0F ;
				  pointer_on_channel->panning_of_channel = ((pointer_on_channel->panning_of_channel-slide_value) < -32)?-32:pointer_on_channel->panning_of_channel - slide_value ;
				  if (pointer_on_channel->playing)
				  {
					  pointer_on_virtual_channel->panning_of_channel = pointer_on_channel->panning_of_channel ;
				  }
			  }
			  else if ((command_value & 0xF0) == 0xF0) /* fine panning slide right */
			  {
				  unsigned char slide_value ;
				  slide_value = command_value >> 4 ;
				  pointer_on_channel->panning_of_channel = ((pointer_on_channel->panning_of_channel+slide_value) > 32)?32:pointer_on_channel->panning_of_channel + slide_value ;
				  if (pointer_on_channel->playing)
				  {
					  pointer_on_virtual_channel->panning_of_channel = pointer_on_channel->panning_of_channel ;
				  }
			  }
		  }
		  else if (command == 24) /* set panning position : X */
		  {
			  signed short panning ;
			  panning = -128 + (signed short)command_value ;
			  panning = (panning < 0)?panning/4:(panning + 1) / 4 ;
			  pointer_on_channel->panning_of_channel = (signed char)panning ;
			  if (pointer_on_channel->playing == 1)
			  {
				  pointer_on_virtual_channel->panning_of_channel = (signed char)panning ;
			  }
		  }
		  else if  (pointer_on_channel->playing == 1)
		  {
			  if ((command == 4)||(command == 11)||(command == 12)) /* volume slide : D, or dual commands K & L */
			  {
				  if (command_value == 0)
				  {
					  pointer_on_channel->previous_command_value = command_value = pointer_on_channel->last_slide_parameter ;
				  }
				  else
				  {
					  pointer_on_channel->last_slide_parameter = command_value ;
				  }
				  if (!(command_value & 0x0F)) /* volume slide up */
				  {
					  pointer_on_channel->effect_update = 1 ;
					  pointer_on_channel->slide_value = command_value >> 4 ;
				  }
				  else if (!(command_value & 0xF0)) /* volume slide down */
				  {
					  pointer_on_channel->effect_update = 1 ;
					  pointer_on_channel->slide_value = command_value & 0x0F ;
				  }
				  else if ((command_value & 0x0F) == 0x0F) /* fine volume slide up */
				  {
					  unsigned char slide_value ;
					  slide_value = command_value >> 4 ;
					  pointer_on_virtual_channel->volume = ((pointer_on_virtual_channel->volume + slide_value)<=64)?(pointer_on_virtual_channel->volume + slide_value):64 ;
				  }
				  else if ((command_value & 0xF0) == 0xF0) /* fine volume slide down */
				  {
					  signed char slide_value ;
					  slide_value = command_value & 0x0F ;
					  pointer_on_virtual_channel->volume = ((((signed short)pointer_on_virtual_channel->volume) - slide_value)>=0)?(((signed short)pointer_on_virtual_channel->volume) - slide_value):0 ;
				  }
			  }
			  else if (command == 5) /* pitch slide down */
			  {
				  if (command_value == 0)
				  {
					  pointer_on_channel->previous_command_value = command_value = pointer_on_channel->last_slide_to_note_speed ;
				  }
				  else
				  {
					  pointer_on_channel->last_slide_to_note_speed = command_value ;
				  }
				  
				  if (command_value < 0xE0) /* pitch slide down by command_value */
				  {
					  pointer_on_channel->effect_update = 1 ;
				  }
				  else if (command_value < 0xF0) /* extra fine pitch slide down */
				  {
					  if (pointer_on_current_module->type_of_slides == 1)
					  {
						  signed short note ;
						  signed short sub_note ;
						  note = pointer_on_channel->last_sample_note_used ;
						  sub_note = 64 - pointer_on_channel->last_sub_note_used  ;
						  
						  note -= ((signed short) (sub_note += (command_value - 0xE0)))/64 ;
						  if (note < 0)
						  { /* Remove the virtual channel*/
							  
							  pointer_on_virtual_channel->channel_playing = 0 ;
							  pointer_on_channel->playing = 0 ;
							  number_of_virtual_channels_allocated-- ;
							  {
								  p_virtual_channel temp_virtual_channel ;
								  
								  temp_virtual_channel = virtual_channels[number_of_virtual_channels_allocated] ;
								  virtual_channels[number_of_virtual_channels_allocated] = virtual_channels[channel_value] ;
								  virtual_channels[channel_value] = temp_virtual_channel ;
								  real_channels[temp_virtual_channel->corresponding_real_channel].foreground_virtual_channel_number = channel_value ;
							  }
							  continue ; 
						  }
						  pointer_on_channel->last_sample_note_used = (unsigned char)note ;
						  pointer_on_channel->last_sub_note_used = 64 - (sub_note & 0x3F) ;
						  pointer_on_virtual_channel->step_length_for_sample_of_channel = (unsigned long)(pointer_on_virtual_channel->pointer_on_sample->adjusted_C5_speed * precalculated_notes_table[pointer_on_channel->last_sample_note_used * 64l + pointer_on_channel->last_sub_note_used]);
					  }
					  else
					  {
						  p_sample current_sample ;
						  double current_value ;
						  double amiga_value ;
						  
						  current_sample = pointer_on_virtual_channel->pointer_on_sample ;
						  current_value = pointer_on_virtual_channel->note_table_value ;
						  amiga_value = convert_to_amiga (current_value) ;
						  amiga_value += (command_value - 0xE0) * 64.0 * (current_sample->C5_speed/8363.42) ;
						  current_value = convert_from_amiga (amiga_value) ;
						  pointer_on_virtual_channel->note_table_value = (unsigned long)current_value ;
						  pointer_on_virtual_channel->step_length_for_sample_of_channel = (unsigned long)(current_sample->adjusted_C5_speed * current_value) ;
					  }
				  }
				  else /* fine pitch slide down */
				  {
					  if (pointer_on_current_module->type_of_slides == 1)
					  {
						  signed short note ;
						  signed short sub_note ;
						  note = pointer_on_channel->last_sample_note_used ;
						  sub_note = 64 - pointer_on_channel->last_sub_note_used  ;
						  
						  note -= (((signed short) (sub_note += 4*(command_value - 0xF0))))/64 ;
						  if (note < 0)
						  { /* Remove the virtual channel*/
							  
							  pointer_on_virtual_channel->channel_playing = 0 ;
							  pointer_on_channel->playing = 0 ;
							  number_of_virtual_channels_allocated-- ;
							  {
								  p_virtual_channel temp_virtual_channel ;
								  
								  temp_virtual_channel = virtual_channels[number_of_virtual_channels_allocated] ;
								  virtual_channels[number_of_virtual_channels_allocated] = virtual_channels[channel_value] ;
								  virtual_channels[channel_value] = temp_virtual_channel ;
								  real_channels[temp_virtual_channel->corresponding_real_channel].foreground_virtual_channel_number = channel_value ;
							  }
							  continue ;
						  }
						  pointer_on_channel->last_sample_note_used = (unsigned char) note ;
						  pointer_on_channel->last_sub_note_used = 64 - (sub_note & 0x3F) ;
						  pointer_on_virtual_channel->step_length_for_sample_of_channel = (unsigned long)(pointer_on_virtual_channel->pointer_on_sample->adjusted_C5_speed * precalculated_notes_table[pointer_on_channel->last_sample_note_used * 64l + pointer_on_channel->last_sub_note_used]) ;
					  }
					  else
					  {
						  p_sample current_sample ;
						  double current_value ;
						  double amiga_value ;

						  current_sample = pointer_on_virtual_channel->pointer_on_sample ;
						  current_value = pointer_on_virtual_channel->note_table_value ;
						  amiga_value = convert_to_amiga (current_value) ;
						  amiga_value += (command_value - 0xF0) * 256.0 * (current_sample->C5_speed/8363.42) ;
						  current_value = convert_from_amiga (amiga_value) ;
						  pointer_on_virtual_channel->note_table_value = (unsigned long)current_value ;
						  pointer_on_virtual_channel->step_length_for_sample_of_channel = (unsigned long)(current_sample->adjusted_C5_speed * current_value) ;
					  }
				  }
         }
         else if (command == 6) /* pitch slide up */
         {
			 if (command_value == 0)
			 {
				 pointer_on_channel->previous_command_value = command_value = pointer_on_channel->last_slide_to_note_speed ;
			 }
			 else
			 {
				 pointer_on_channel->last_slide_to_note_speed = command_value ;
			 }
			 
			 if (command_value < 0xE0) /* pitch slide up by command_value */
			 {
				 pointer_on_channel->effect_update = 1 ;
			 }
			 else if (command_value < 0xF0) /* extra fine pitch slide up */
			 {
				 if (pointer_on_current_module->type_of_slides == 1)
				 {
					 signed short note ;
					 signed short sub_note ;
					 note = pointer_on_channel->last_sample_note_used ;
					 sub_note = pointer_on_channel->last_sub_note_used  ;
					 
					 note += (((signed short) (sub_note += (command_value - 0xE0))))/64 ;
					 pointer_on_channel->last_sample_note_used = (unsigned char)note ;
					 pointer_on_channel->last_sub_note_used = sub_note & 0x3F ;
					 pointer_on_virtual_channel->step_length_for_sample_of_channel = (unsigned long)(pointer_on_virtual_channel->pointer_on_sample->adjusted_C5_speed * precalculated_notes_table[pointer_on_channel->last_sample_note_used * 64l + pointer_on_channel->last_sub_note_used]) ;
				 }
				 else
				 {
					 p_sample current_sample ;
					 double current_value ;
					 double amiga_value ;

					 current_sample = pointer_on_virtual_channel->pointer_on_sample ;
					 current_value = pointer_on_virtual_channel->note_table_value ;
					 amiga_value = convert_to_amiga (8363.42*current_value/current_sample->C5_speed) ;
					 amiga_value -= (command_value - 0xE0) * 64.0 ;
					 if (amiga_value <= 0)
					 { /* Remove the virtual channel*/

						 pointer_on_virtual_channel->channel_playing = 0 ;
						 pointer_on_channel->playing = 0 ;
						 number_of_virtual_channels_allocated-- ;
						 {
							 p_virtual_channel temp_virtual_channel ;

							 temp_virtual_channel = virtual_channels[number_of_virtual_channels_allocated] ;
							 virtual_channels[number_of_virtual_channels_allocated] = virtual_channels[channel_value] ;
							 virtual_channels[channel_value] = temp_virtual_channel ;
							 real_channels[temp_virtual_channel->corresponding_real_channel].foreground_virtual_channel_number = channel_value ;
						 }
						 continue ;
					 }
					 current_value = convert_from_amiga (amiga_value) ;
					 pointer_on_virtual_channel->note_table_value = (unsigned long)current_value ;
					 pointer_on_virtual_channel->step_length_for_sample_of_channel = (unsigned long)(current_sample->adjusted_C5_speed * current_value) ;
				 }
			 }
			 else /* fine pitch slide up */
			 {
				 if (pointer_on_current_module->type_of_slides == 1)
				 {
					 signed short note ;
					 signed short sub_note ;
					 note = pointer_on_channel->last_sample_note_used ;
					 sub_note = pointer_on_channel->last_sub_note_used  ;
					 
					 note += (((signed short) (sub_note += 4*(command_value - 0xF0))))/64 ;
					 pointer_on_channel->last_sample_note_used = (unsigned char)note ;
					 pointer_on_channel->last_sub_note_used = sub_note & 0x3F ;
					 pointer_on_virtual_channel->step_length_for_sample_of_channel = (unsigned long)(pointer_on_virtual_channel->pointer_on_sample->adjusted_C5_speed * precalculated_notes_table[pointer_on_channel->last_sample_note_used * 64l + pointer_on_channel->last_sub_note_used]) ;
				 }
				 else
				 {
					 p_sample current_sample ;
					 double current_value ;
					 double amiga_value ;

					 current_sample = pointer_on_virtual_channel->pointer_on_sample ;
					 current_value = pointer_on_virtual_channel->note_table_value ;
					 amiga_value = convert_to_amiga (8363.42*current_value/current_sample->C5_speed) ;
					 amiga_value -= (command_value - 0xF0) * 256.0 ;
					 if (amiga_value <= 0)
					 { /* Remove the virtual channel*/
						 
						 pointer_on_virtual_channel->channel_playing = 0 ;
						 pointer_on_channel->playing = 0 ;
						 number_of_virtual_channels_allocated-- ;
						 {
							 p_virtual_channel temp_virtual_channel ;
							 
							 temp_virtual_channel = virtual_channels[number_of_virtual_channels_allocated] ;
							 virtual_channels[number_of_virtual_channels_allocated] = virtual_channels[channel_value] ;
							 virtual_channels[channel_value] = temp_virtual_channel ;
							 real_channels[temp_virtual_channel->corresponding_real_channel].foreground_virtual_channel_number = channel_value ;
						 }
						 continue ;
					 }
					 
					 current_value = convert_from_amiga (amiga_value) ;
					 pointer_on_virtual_channel->note_table_value = (unsigned long)current_value ;
					 pointer_on_virtual_channel->step_length_for_sample_of_channel = (unsigned long)(current_sample->adjusted_C5_speed * current_value) ;
				 }
			 }
         }
         else if (command == 15) /* Set sample offset : O */
         {
			 if ((pointer_on_virtual_channel->pointer_on_sample_beginning + (((unsigned long) command_value) << 8)) < pointer_on_virtual_channel->pointer_on_sample_end)
			 {
				 pointer_on_virtual_channel->pointer_in_sample_of_channel = pointer_on_virtual_channel->pointer_on_sample_beginning + (((unsigned long) command_value) << 8) ;
			 }
         }
        }
       }
	   
	   if (pointer_on_channel->playing == 1)
	   {
		   double volume ;
		   double separation ;
		   
		   separation = pointer_on_current_module->panning_separation / 128.0 ;
		   
		   volume =            pointer_on_virtual_channel->volume
			   * (double) pointer_on_virtual_channel->volume_of_sample
			   * (double) pointer_on_virtual_channel->volume_envelope_value
			   * (double) pointer_on_virtual_channel->volume_of_channel
			   * (double) pointer_on_virtual_channel->note_fade_component ;
		   pointer_on_virtual_channel->final_volume_left = volume * (32-(pointer_on_virtual_channel->panning_of_channel * separation)) ;
		   pointer_on_virtual_channel->final_volume_right = volume * (32+(pointer_on_virtual_channel->panning_of_channel * separation));
	   }
	   
	   
    }
	
	current_line++ ;
	
	if ((new_speed_requested)||(new_tempo_requested))
	{
		set_speed_and_tempo (new_speed, new_tempo) ;
	}
	
	if (new_global_volume_requested)
	{
		global_volume = new_global_volume ;
	}
	
	if (global_volume_slide_requested)
	{
		if (global_volume_slide_value == 0)
		{
			global_volume_slide_value = last_global_volume_slide_parameter ;
		}
		else
		{
			last_global_volume_slide_parameter = global_volume_slide_value ;
		}
		if (!(global_volume_slide_value & 0x0F)) /* global volume slide up */
		{
			global_volume_slide_on = 1 ;
		}
		else if (!(global_volume_slide_value & 0xF0)) /* global volume slide down */
		{
			global_volume_slide_on = 1 ;
		}
		else if ((global_volume_slide_value & 0x0F) == 0x0F) /* fine global volume slide up */
		{
			unsigned char slide_value ;
			slide_value = global_volume_slide_value >> 4 ;
			global_volume = ((global_volume + slide_value)<=128)?(global_volume + slide_value):128 ;
		}
		else if ((global_volume_slide_value & 0xF0) == 0xF0) /* fine global volume slide down */
		{
			signed char slide_value ;
			slide_value = global_volume_slide_value & 0x0F ;
			global_volume = ((((signed short)global_volume) - slide_value)>=0)?(((signed short)global_volume) - slide_value):0 ;
		}
	} /* end global volume slide */
	
	
	if ((pattern_break_requested)||(current_line >= pointer_on_current_pattern->number_of_rows))
	{
		if (last_pattern)
		{
			last_line = 1 ;
		}
		else
		{
			current_line = 0 ;
			new_pattern (pattern_break_line) ;
		}
	}
}







/* new_frame steps to the next frame, updating effects (not implemented for now !) */
void new_frame (void)
{
	unsigned char channel_counter ;
	
	current_frame++ ;
	if (current_frame >= frames_per_line)
	{
		if (last_line)
		{
			last_frame = 1 ;
		}
		else
		{
			current_frame = 0 ;
			new_line () ;
		}
	}
	else
	{
		/* effects update */
		
		if (global_volume_slide_on)
		{
			if (!(global_volume_slide_value & 0x0F)) /* global volume slide up */
			{
				unsigned char slide_value ;
				slide_value = global_volume_slide_value >> 4 ;
				global_volume = ((global_volume + slide_value)<=128)?(global_volume + slide_value):128 ;
			}
			else if (!(global_volume_slide_value & 0xF0)) /* global volume slide down */
			{
				signed char slide_value ;
				slide_value = global_volume_slide_value & 0x0F ;
				global_volume = ((((signed short)global_volume) - slide_value)>=0)?(((signed short)global_volume) - slide_value):0 ;
			}
		}
		
		for (channel_counter = 0 ; channel_counter < number_of_virtual_channels_allocated ; channel_counter++ )
		{
			p_virtual_channel pointer_on_virtual_channel ;
			p_real_channel pointer_on_channel ;
			unsigned char command ;
			unsigned char command_value ;
			
			pointer_on_virtual_channel = virtual_channels[channel_counter] ;
			/*    printf ("Channel : %d\n", channel_counter) ;*/
			
			/* note fading */
			if (pointer_on_virtual_channel->fade)
			{
				signed short volume ;
				
				volume = (signed short) pointer_on_virtual_channel->note_fade_component ;
				volume -= pointer_on_virtual_channel->pointer_on_instrument->fadeout ;
				if (volume < 0)
				{ /* Remove the virtual channel*/
					
					pointer_on_virtual_channel->channel_playing = 0 ;
					
					if (pointer_on_virtual_channel->channel_in_background == 0)
					{
						real_channels[pointer_on_virtual_channel->corresponding_real_channel].playing = 0 ;
					}
					
					number_of_virtual_channels_allocated-- ;
					
					{
						p_virtual_channel temp_virtual_channel ;
						
						temp_virtual_channel = virtual_channels[number_of_virtual_channels_allocated] ;
						virtual_channels[number_of_virtual_channels_allocated] = virtual_channels[channel_counter] ;
						virtual_channels[channel_counter] = temp_virtual_channel ;
						real_channels[temp_virtual_channel->corresponding_real_channel].foreground_virtual_channel_number = channel_counter ;
						
						channel_counter -- ;
						
						continue ; /* continue the channel loop */
					}
				}
				else
				{
					pointer_on_virtual_channel->note_fade_component = (unsigned short) volume ;
				}
			}
			
			if (pointer_on_virtual_channel->channel_in_background == 0)
			{
				pointer_on_channel = & (real_channels[pointer_on_virtual_channel->corresponding_real_channel]) ;
				command = pointer_on_channel->previous_command ;
				command_value = pointer_on_channel->previous_command_value ;
				
				if (pointer_on_channel->effect_update == 1)
				{
					if (pointer_on_channel->slide_to_note_direction < 0)
					{
						if (pointer_on_current_module->type_of_slides == 1)
						{
							signed short note ;
							signed short sub_note ;
							note = pointer_on_channel->last_sample_note_used ;
							sub_note = 63 - pointer_on_channel->last_sub_note_used  ;
							
							note -= (((signed short) (sub_note += 4*pointer_on_channel->last_slide_to_note_speed)))/64 ;
							if (note <= pointer_on_channel->last_slide_to_note_note)
							{
								pointer_on_channel->last_sample_note_used = pointer_on_channel->last_slide_to_note_note ;
								pointer_on_channel->last_sub_note_used = 0 ;
								pointer_on_channel->slide_to_note_direction = 0 ;
							}
							else
							{
								pointer_on_channel->last_sample_note_used = (unsigned char)note ;
								pointer_on_channel->last_sub_note_used = 63 - (sub_note & 0x3F) ;
							}
							pointer_on_virtual_channel->step_length_for_sample_of_channel = (unsigned long)(pointer_on_virtual_channel->pointer_on_sample->adjusted_C5_speed * precalculated_notes_table[pointer_on_channel->last_sample_note_used * 64l + pointer_on_channel->last_sub_note_used]) ;
						}
						else
						{
							p_sample current_sample ;
							double note_value ;
							double current_value ;
							double amiga_value ;
							
							current_sample = pointer_on_virtual_channel->pointer_on_sample ;
							note_value = precalculated_notes_table[64U*pointer_on_current_module->instruments[pointer_on_channel->last_instrument_used]->note_sample_keyboard_table[pointer_on_channel->last_slide_to_note_note].note] ;
							current_value = pointer_on_virtual_channel->note_table_value ;
							amiga_value = convert_to_amiga (current_value*current_sample->C5_speed/8363.42) ;
							amiga_value += pointer_on_channel->last_slide_to_note_speed * 256.0 ;
							current_value = convert_from_amiga (amiga_value) / current_sample->C5_speed * 8363.42;
							if (current_value <= note_value)
							{
								pointer_on_channel->slide_to_note_direction = 0 ;
								pointer_on_virtual_channel->note_table_value = (unsigned long)note_value ;
								pointer_on_virtual_channel->step_length_for_sample_of_channel = (unsigned long)(current_sample->adjusted_C5_speed * note_value) ;
							}
							else
							{
								pointer_on_virtual_channel->note_table_value = (unsigned long)current_value ;
								pointer_on_virtual_channel->step_length_for_sample_of_channel = (unsigned long)(current_sample->adjusted_C5_speed * current_value) ;
							}
						}
					}
					else if (pointer_on_channel->slide_to_note_direction > 0)
					{
						if (pointer_on_current_module->type_of_slides == 1)
						{
							signed short note ;
							note = pointer_on_channel->last_sample_note_used ;
							note += (((signed short) (pointer_on_channel->last_sub_note_used += 4*pointer_on_channel->last_slide_to_note_speed)))/64 ;
							if (note > pointer_on_channel->last_slide_to_note_note)
							{
								pointer_on_channel->last_sample_note_used = pointer_on_channel->last_slide_to_note_note ;
								pointer_on_channel->last_sub_note_used = 0 ;
								pointer_on_channel->slide_to_note_direction = 0 ;
							}
							else
							{
								pointer_on_channel->last_sample_note_used = (unsigned char)note ;
								pointer_on_channel->last_sub_note_used &= 0x3F ;
							}
							{
								pointer_on_virtual_channel->step_length_for_sample_of_channel = (unsigned long)(pointer_on_virtual_channel->pointer_on_sample->adjusted_C5_speed * precalculated_notes_table[pointer_on_channel->last_sample_note_used * 64l + pointer_on_channel->last_sub_note_used]) ;
							}
						}
						else
						{
							p_sample current_sample ;
							double note_value ;
							double current_value ;
							double amiga_value ;
							
							current_sample = pointer_on_virtual_channel->pointer_on_sample ;
							note_value = precalculated_notes_table[64U*pointer_on_current_module->instruments[pointer_on_channel->last_instrument_used]->note_sample_keyboard_table[pointer_on_channel->last_slide_to_note_note].note] ;
							current_value = pointer_on_virtual_channel->note_table_value ;
							amiga_value = convert_to_amiga (current_value*current_sample->C5_speed/8363.42) ;
							amiga_value -= pointer_on_channel->last_slide_to_note_speed * 256.0 ;
							if (amiga_value <= 0)
							{
								current_value = note_value ;
							}
							else
							{
								current_value = convert_from_amiga (amiga_value) / current_sample->C5_speed*8363.42 ;
							}
							if (current_value >= note_value)
							{
								pointer_on_channel->slide_to_note_direction = 0 ;
								pointer_on_virtual_channel->note_table_value = (unsigned long)note_value ;
								pointer_on_virtual_channel->step_length_for_sample_of_channel = (unsigned long)(current_sample->adjusted_C5_speed * note_value) ;
							}
							else
							{
								pointer_on_virtual_channel->note_table_value = (unsigned long)current_value ;
								pointer_on_virtual_channel->step_length_for_sample_of_channel = (unsigned long)(current_sample->adjusted_C5_speed * current_value) ;
							}
						}
					}
					if (command == 14) /* channel volume slide : N */
					{
						if (!(command_value & 0x0F)) /* channel volume slide up */
						{
							unsigned char volume ;
							
							volume = pointer_on_channel->volume_of_channel + pointer_on_channel->slide_value ;
							pointer_on_channel->volume_of_channel = (volume>64)?64:volume ;
							if (pointer_on_channel->playing)
							{
								pointer_on_virtual_channel->volume_of_channel = pointer_on_channel->volume_of_channel ;
							}
						}
						else if (!(command_value & 0xF0)) /* channel volume slide down */
						{
							signed char volume ;
							
							volume = ((signed char)pointer_on_channel->volume_of_channel) - ((signed char)pointer_on_channel->slide_value) ;
							pointer_on_channel->volume_of_channel = (unsigned char)((volume<0)?0:volume) ;
							if (pointer_on_channel->playing)
							{
								pointer_on_virtual_channel->volume_of_channel = pointer_on_channel->volume_of_channel ;
							}
						}
					}
					else if (command == 16) /* channel panning slide : P */
					{
						if (!(command_value & 0x0F)) /* channel panning slide left */
						{
							pointer_on_channel->panning_of_channel = ((pointer_on_channel->panning_of_channel-((signed char)pointer_on_channel->slide_value)) < -32)?-32:pointer_on_channel->panning_of_channel - ((signed char)pointer_on_channel->slide_value) ;
							if (pointer_on_channel->playing)
							{
								pointer_on_virtual_channel->panning_of_channel = pointer_on_channel->panning_of_channel ;
							}
						}
						else if (!(command_value & 0xF0)) /* channel panning slide right */
						{
							pointer_on_channel->panning_of_channel = ((pointer_on_channel->panning_of_channel+pointer_on_channel->slide_value) > 32)?32:pointer_on_channel->panning_of_channel + pointer_on_channel->slide_value ;
							if (pointer_on_channel->playing)
							{
								pointer_on_virtual_channel->panning_of_channel = pointer_on_channel->panning_of_channel ;
							}
						}
					}
					else if ((command == 4)||(command == 11)||(command == 12)) /* volume slide : D, or dual commands K & L */
					{
						if (!(command_value & 0x0F)) /* volume slide up */
						{
							unsigned char volume ;
							
							volume = pointer_on_virtual_channel->volume + pointer_on_channel->slide_value ;
							pointer_on_virtual_channel->volume = (volume>64)?64:volume ;
							/*            printf (" VolSlideUp-> %d <\n", pointer_on_virtual_channel->volume) ;*/
						}
						else if (!(command_value & 0xF0)) /* volume slide down */
						{
							signed char volume ;
							
							volume = pointer_on_virtual_channel->volume - pointer_on_channel->slide_value ;
							pointer_on_virtual_channel->volume = (signed short)((volume<0)?0:volume) ;
							/*            printf (" VolSlideDown-> %d <\n", pointer_on_virtual_channel->volume) ;*/
						}
					}
					else if (command == 5) /* pitch slide down */
					{
						if (pointer_on_current_module->type_of_slides == 1)
						{
							signed short note ;
							signed short sub_note ;
							note = pointer_on_channel->last_sample_note_used ;
							sub_note = 63 - pointer_on_channel->last_sub_note_used  ;
							
							note -= (((signed short) (sub_note += 4*(pointer_on_channel->last_slide_to_note_speed))))/64 ;
							if (note < 0)
							{ /* Remove the virtual channel*/
								
								pointer_on_virtual_channel->channel_playing = 0 ;
								pointer_on_channel->playing = 0 ;
								number_of_virtual_channels_allocated-- ;
								{
									p_virtual_channel temp_virtual_channel ;
									
									temp_virtual_channel = virtual_channels[number_of_virtual_channels_allocated] ;
									virtual_channels[number_of_virtual_channels_allocated] = virtual_channels[channel_counter] ;
									virtual_channels[channel_counter] = temp_virtual_channel ;
									real_channels[temp_virtual_channel->corresponding_real_channel].foreground_virtual_channel_number = channel_counter ;
								}
								continue ;
							}
							pointer_on_channel->last_sample_note_used = (unsigned char)note ;
							pointer_on_channel->last_sub_note_used = 63 - (sub_note & 0x3F) ;
							pointer_on_virtual_channel->step_length_for_sample_of_channel = (unsigned long)(pointer_on_virtual_channel->pointer_on_sample->adjusted_C5_speed * precalculated_notes_table[pointer_on_channel->last_sample_note_used * 64l + pointer_on_channel->last_sub_note_used]) ;
						}
						else
						{
							p_sample current_sample ;
							double current_value ;
							double amiga_value ;
							
							current_sample = pointer_on_virtual_channel->pointer_on_sample ;
							current_value = pointer_on_virtual_channel->note_table_value ;
							/*            printf ("%d, %lf -> ", pointer_on_channel->last_slide_to_note_speed, current_value * current_sample->C5_speed / 16384) ;*/
							amiga_value = convert_to_amiga (current_value*current_sample->C5_speed/8363.42) ;
							/*            printf ("%lf -> ", amiga_value) ;*/
							amiga_value += pointer_on_channel->last_slide_to_note_speed * 256.0 ;
							current_value = convert_from_amiga (amiga_value) / current_sample->C5_speed * 8363.42;
							pointer_on_virtual_channel->note_table_value = (unsigned long)current_value  ;
							pointer_on_virtual_channel->step_length_for_sample_of_channel = (unsigned long)(current_sample->adjusted_C5_speed * current_value) ;
						}
					}
					else if (command == 6) /* pitch slide up */
					{
						if (pointer_on_current_module->type_of_slides == 1)
						{
							signed short note ;
							signed short sub_note ;
							note = pointer_on_channel->last_sample_note_used ;
							sub_note = pointer_on_channel->last_sub_note_used  ;
							
							note += (((signed short) (sub_note += 4*(pointer_on_channel->last_slide_to_note_speed))))/64 ;
							pointer_on_channel->last_sample_note_used = (unsigned char)note ;
							pointer_on_channel->last_sub_note_used = sub_note & 0x3F ;
							pointer_on_virtual_channel->step_length_for_sample_of_channel = 
								(unsigned long)(  pointer_on_virtual_channel->pointer_on_sample->adjusted_C5_speed
								* precalculated_notes_table[pointer_on_channel->last_sample_note_used * 64l + pointer_on_channel->last_sub_note_used] );
						}
						else
						{
							p_sample current_sample ;
							double current_value ;
							double amiga_value ;
							
							current_sample = pointer_on_virtual_channel->pointer_on_sample ;
							current_value = pointer_on_virtual_channel->note_table_value ;
							/*            printf ("%d, %lf -> ", pointer_on_channel->last_slide_to_note_speed, current_value * current_sample->C5_speed / 16384) ;*/
							amiga_value = convert_to_amiga (current_value*current_sample->C5_speed/8363.42) ;
							amiga_value -= (pointer_on_channel->last_slide_to_note_speed) * 256.0 ;/** (current_sample->C5_speed/8363.42) ;*/
							if (amiga_value <= 0)
							{ /* Remove the virtual channel*/
								
								pointer_on_virtual_channel->channel_playing = 0 ;
								pointer_on_channel->playing = 0 ;
								number_of_virtual_channels_allocated-- ;
								{
									p_virtual_channel temp_virtual_channel ;
									
									temp_virtual_channel = virtual_channels[number_of_virtual_channels_allocated] ;
									virtual_channels[number_of_virtual_channels_allocated] = virtual_channels[channel_counter] ;
									virtual_channels[channel_counter] = temp_virtual_channel ;
									real_channels[temp_virtual_channel->corresponding_real_channel].foreground_virtual_channel_number = channel_counter ;
								}
								continue ;
							}
							current_value = convert_from_amiga (amiga_value) /current_sample->C5_speed * 8363.42;
							pointer_on_virtual_channel->note_table_value = (unsigned long)current_value ;
							pointer_on_virtual_channel->step_length_for_sample_of_channel =
								(unsigned long)(current_sample->adjusted_C5_speed * current_value) ;
						}
					}
					if ((current_frame + 1) >= frames_per_line)
					{
						pointer_on_channel->effect_update = 0 ;
					}
      }
    } /* end of foreground channel specific commands */
	
	
    /* envelopes update */
    {
		p_envelope volume_envelope ;
		unsigned char *interpolated_volume_envelope ;
		unsigned short volume_envelope_tick ;
		
		p_envelope panning_envelope ;
		signed char *interpolated_panning_envelope ;
		unsigned short panning_envelope_tick ;
		
		volume_envelope = pointer_on_virtual_channel->volume_envelope ;
		interpolated_volume_envelope = pointer_on_virtual_channel->interpolated_volume_envelope ;
		volume_envelope_tick = pointer_on_virtual_channel->volume_envelope_tick ;
		
		panning_envelope = pointer_on_virtual_channel->panning_envelope ;
		interpolated_panning_envelope = pointer_on_virtual_channel->interpolated_panning_envelope ;
		panning_envelope_tick = pointer_on_virtual_channel->panning_envelope_tick ;
		
		/* volume envelope */
		
		if ((volume_envelope->envelope_on_off)&&(!pointer_on_virtual_channel->end_of_volume_envelope))
		{
			volume_envelope_tick++ ;
			if ((pointer_on_virtual_channel->sustain) && (pointer_on_virtual_channel->volume_envelope->sustain_loop_on_off))
			{ /* sustain loop */
				if (volume_envelope_tick > volume_envelope->sustain_loop_end_tick)
				{
					volume_envelope_tick = volume_envelope->sustain_loop_begin_tick ;
				}
				pointer_on_virtual_channel->volume_envelope_value = interpolated_volume_envelope[volume_envelope_tick] ;
			}
			else if (volume_envelope->loop_on_off)
			{ /* normal loop */
				if (volume_envelope_tick > volume_envelope->loop_end_tick)
				{
					volume_envelope_tick = volume_envelope->loop_begin_tick ;
				}
				pointer_on_virtual_channel->volume_envelope_value = interpolated_volume_envelope[volume_envelope_tick] ;
			}
			else
			{ /* no loop */
				if (volume_envelope_tick > volume_envelope->last_tick)
				{ /* end of envelope : */
					if (pointer_on_virtual_channel->volume_envelope_value == 0)
					{ /* Remove the virtual channel*/
						
						pointer_on_virtual_channel->channel_playing = 0 ;
						
						if (pointer_on_virtual_channel->channel_in_background == 0)
						{
							real_channels[pointer_on_virtual_channel->corresponding_real_channel].playing = 0 ;
						}
						
						number_of_virtual_channels_allocated-- ;
						
						{
							p_virtual_channel temp_virtual_channel ;
							
							temp_virtual_channel = virtual_channels[number_of_virtual_channels_allocated] ;
							virtual_channels[number_of_virtual_channels_allocated] = virtual_channels[channel_counter] ;
							virtual_channels[channel_counter] = temp_virtual_channel ;
							real_channels[temp_virtual_channel->corresponding_real_channel].foreground_virtual_channel_number = channel_counter ;
							
							channel_counter -- ;
							
							continue ; /* continue the channel loop */
						}
					}
					else
					{
						pointer_on_virtual_channel->fade = 1 ;
						pointer_on_virtual_channel->end_of_volume_envelope = 1 ;
					}
				}
				else
				{
					pointer_on_virtual_channel->volume_envelope_value = interpolated_volume_envelope[volume_envelope_tick] ;
				}
			}
		}
		
		/* panning envelope */
		
		if ((panning_envelope->envelope_on_off)&&(!pointer_on_virtual_channel->end_of_panning_envelope))
		{
			panning_envelope_tick++ ;
			
			if ((pointer_on_virtual_channel->sustain) && (panning_envelope->sustain_loop_on_off))
			{ /* sustain loop */
				if (panning_envelope_tick > panning_envelope->sustain_loop_end_tick)
				{
					panning_envelope_tick = panning_envelope->sustain_loop_begin_tick ;
				}
				pointer_on_virtual_channel->panning_of_channel = interpolated_panning_envelope[volume_envelope_tick] ;
			}
			else if (panning_envelope->loop_on_off)
			{ /* normal loop */
				if (panning_envelope_tick > panning_envelope->loop_end_tick)
				{
					panning_envelope_tick = panning_envelope->loop_begin_tick ;
				}
				pointer_on_virtual_channel->panning_of_channel = interpolated_panning_envelope[volume_envelope_tick] ;
			}
			else
			{ /* no loop */
				if (panning_envelope_tick > panning_envelope->last_tick)
				{ /* end of envelope : */
					pointer_on_virtual_channel->end_of_panning_envelope = 1 ;
				}
				else
				{
					pointer_on_virtual_channel->panning_of_channel = interpolated_panning_envelope[volume_envelope_tick] ;
				}
			}
		}
		pointer_on_virtual_channel->volume_envelope_tick = volume_envelope_tick ;
		pointer_on_virtual_channel->panning_envelope_tick = panning_envelope_tick ;
      }
      {
		  double volume ;
		  double separation ;
		  
		  separation = pointer_on_current_module->panning_separation / 128.0 ;
		  volume =            pointer_on_virtual_channel->volume
			  * (double) pointer_on_virtual_channel->volume_of_sample
			  * (double) pointer_on_virtual_channel->volume_envelope_value
			  * (double) pointer_on_virtual_channel->volume_of_channel
			  * (double) pointer_on_virtual_channel->note_fade_component ;
		  pointer_on_virtual_channel->final_volume_left = volume * (32-(pointer_on_virtual_channel->panning_of_channel * separation)) ;
		  pointer_on_virtual_channel->final_volume_right = volume * (32+(pointer_on_virtual_channel->panning_of_channel * separation));
	  }
    }
  }
}

void *memalloc (size_t size)
/* Allocate 'size' bytes of memory by calling 'malloc'.
If the allocation fail, a message is printed on the
'stderr' stream and the program exit. */
{
	/* local variables of 'memalloc' */
	/*_______________________________*/
	
	void *return_value ;
	
	
	/* 'memalloc' body */
	/*_________________*/
	
	return_value = malloc (size) ;
	
	if (return_value == (void *) NULL)
    {
		(void) fprintf (stderr, "Memory allocation of %ld bytes failed.\nExiting\n", (long) size) ;
		exit (1) ;
    }
	
	return return_value ;
}

void init_sample_output_frequency (unsigned short sampling_rate)
{
	unsigned char sample_counter ;
	
	for (sample_counter = 0 ; sample_counter < pointer_on_current_module->number_of_samples ; sample_counter++)
	{
		pointer_on_current_module->samples[sample_counter]->adjusted_C5_speed = pointer_on_current_module->samples[sample_counter]->C5_speed / (double)sampling_rate ;
	}
}

void init_notes_table(void)
{
	unsigned char octave ;
	unsigned char note ;
	unsigned char sub_note ;
	double factor ;
	
	{
		for (octave = 0 ; octave < 9 ; octave ++)
		{
			for (note = 0 ; note < 12 ; note ++)
			{
				for (sub_note = 0 ; sub_note < 64 ; sub_note ++)
				{
					factor = pow(2, (((double)sub_note)/768.0 + ((double)note) / 12.0 + (octave - 5))) ;
					precalculated_notes_table[(octave * 12L + note) * 64L + sub_note] = (16384 * factor) ;
				}
			}
		}
	}
}

void init_real_and_virtual_channels (void)
{
	unsigned short channel_counter ;
	
	for (channel_counter = 0 ; channel_counter < 256 ; channel_counter ++)
	{ /* allocate memory for the virtual channels */
		virtual_channels[channel_counter] = (p_virtual_channel)memalloc(sizeof (virtual_channel)) ;
		virtual_channels[channel_counter]->last_sound_value = 0.0 ;
		virtual_channels[channel_counter]->use_last_sound_value = 0 ;
		virtual_channels[channel_counter]->use_ramp_down = 0 ;
		virtual_channels[channel_counter]->ramp_position = 0 ;
		virtual_channels[channel_counter]->new_sample = 1 ;
		virtual_channels[channel_counter]->end_of_sample = 0 ;
	}
	
	for (channel_counter = 0 ; channel_counter < 64 ; channel_counter++)
	{ /* init the real channels variables */
	  real_channels[channel_counter].playing = 0; /* prevent the use of the virtual
	  channel before allocation */
	  
	  real_channels[channel_counter].previous_mask_variable = 0;
	  real_channels[channel_counter].previous_note_value = 0 ;
	  real_channels[channel_counter].previous_instrument = 0 ;
	  real_channels[channel_counter].previous_volume_panning = 0 ;
	  real_channels[channel_counter].previous_command = 0 ;
	  real_channels[channel_counter].previous_command_value = 0 ;
	  real_channels[channel_counter].effect_update = 0 ;
	  real_channels[channel_counter].slide_value = 0 ;
	  real_channels[channel_counter].last_slide_parameter = 0 ;
	  real_channels[channel_counter].last_channel_volume_slide_parameter = 0 ;
	  
	  real_channels[channel_counter].current_new_note_action = 0 ; /* note cut */
	  
	  real_channels[channel_counter].volume_of_channel = (pointer_on_current_module->channels_volumes[channel_counter]<=64)?pointer_on_current_module->channels_volumes[channel_counter]:64 ;
	  real_channels[channel_counter].panning_of_channel = (pointer_on_current_module->channels_pannings[channel_counter]<=64)?(((signed short)pointer_on_current_module->channels_pannings[channel_counter]) - 32):0 ;
	  /*    printf ("Channel %d : %3d, %3d\n", channel_counter, pointer_on_current_module->channels_volumes[channel_counter], pointer_on_current_module->channels_pannings[channel_counter]) ;*/
	  if (pointer_on_current_module->channels_pannings[channel_counter] >= 128)
	  {
		  /*      printf ("Channel %d muted\n", channel_counter) ; */
		  real_channels[channel_counter].muted_channel = 1 ;
	  }
	  else
	  {
		  real_channels[channel_counter].muted_channel = 0 ;
	  }
	}
	
}

void init_mixing_variables (p_module it_module,
                            unsigned short sampling_rate,
                            int activate_new_note_actions,
                            unsigned short number_of_virtual_channels,
                            double mixing_volume_parameter,
                            int v_ramp,
                            unsigned short v_ramp_len,
                            int loop
							)
{
	loop_allowed = loop ;
	volume_ramp_on_off = v_ramp ;
	volume_ramp_length = v_ramp_len ;
	new_note_actions_activated = activate_new_note_actions ;
	pointer_on_current_module = it_module ;
	init_notes_table() ;
	init_sample_output_frequency (sampling_rate) ;
	init_real_and_virtual_channels () ;
	samples_per_minutes = ((unsigned long) sampling_rate) * 60UL ;
	set_speed_and_tempo (pointer_on_current_module->initial_speed, pointer_on_current_module->initial_tempo) ;
	global_volume = pointer_on_current_module->global_volume ;
	mixing_volume = mixing_volume_parameter ;
	max_number_of_virtual_channels_allocated = number_of_virtual_channels ;
	number_of_virtual_channels_allocated = 0 ;
	
	last_pattern = 0 ;
	last_line = 0 ;
	last_frame = 0 ;
	
	current_sample = 0 ;
	current_frame = 0 ;
	current_line = 0 ;
	current_order = 0 ;
	current_pattern = pointer_on_current_module->orders[current_order] ;
	new_pattern (0) ;
	new_line () ;
	new_frame () ;
}

void init_mixer (void)
{
}
