/***************************************************************************/
/*  ReSample v0.12                               (c) May 1996 by MLF/SLI  */
/*-------------------------------------------------------------------------*/
/*  This proggy takes an S3I file and resamples it by approximating the    */
/*  the sample with a cubic spline and then storing the new, intermediate  */
/*  sampledata-values.                                                     */
/***************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>

#include <alloc.h>

#include "ems.hpp"

#define stdrate (long)8363
#define maxsize (long)64000

#define lamda 0.5	// 1/(1+1) = 0.5
#define my 0.5   	// 1-0.5 = 0.5
  // for more details, see below

#define EMSpagesize 16384

#define offset(x) (long)((long)(x)%(long)(EMSpagesize/sizeof(float)))
#define page(x) (long)((long)(x)/(long)(EMSpagesize/sizeof(float)))

#define soffset(x) (long)((long)(x)%(long)(EMSpagesize/sizeof(unsigned char)))
#define spage(x) (long)((long)(x)/(long)(EMSpagesize/sizeof(unsigned char)))

long emmhandle;		// global variable - needed for EMS functions

// this structure contains the format of the S3I header
typedef struct {
  char type;
  char filename[12];
  char MemSegHi;	int MemSegLo;
  long Length;
  long LoopBegin;
  long LoopEnd;
  char Volume;
  char x1;	// not used
  char pack;
  char flags;	// 1 for loop to be on
  long C2Spd;
  char x2[4];	// unused
  int IntGp, Int512;	// internal stuff - irrelevant
  long Intlastused;	// more internal stuff
  char samplename[28];
  char ID[4];
  unsigned char* sampledata;	// pointer to the actual sample data
} S3Iheader;

//---------------------------------------------------------------------------
// This function reads the sample into memory
//---------------------------------------------------------------------------
int readS3I(char *filename, S3Iheader* fileread)
{
  FILE *f;
  unsigned pagediff;
  unsigned char* data;
  long x;

  if ((f = fopen(filename, "rb")) == NULL)
    return 1;	// an error has occured

  fread(fileread, sizeof(S3Iheader)-4, 1, f);
    // read header

  if (memcmp("SCRS", fileread->ID, 4)) {
    printf("\n  %s is not a ST3-samplefile\n", filename);
    exit(0);
  }

  x = (page(fileread->Length)+1)*3+spage(fileread->Length)+1;
  if ((emmhandle = emmalloc(x)) < 0) {
    printf("\n  Not enough EMS found\n");
    exit(4);
  }

  memset(fileread->filename, 0, 12);
  memccpy(fileread->filename, filename, 0, 12);
    // In case they don't match

  // read actual sample
  if ((data = (unsigned char*)malloc(EMSpagesize)) == NULL) return 2;
  pagediff = (page(fileread->Length))+1;
  x = fileread->Length;
  while (x > 0) {
    fread(data, (x>EMSpagesize)?EMSpagesize:x, 1, f);
    emmmap(emmhandle, 3*pagediff+spage(fileread->Length-x), 0);
    emmmove(0, (char*)data, (x>EMSpagesize)?EMSpagesize:x);
    x-=EMSpagesize;
  }
  farfree(data);

  fclose(f);

  fileread->sampledata = NULL;

  return 0;
}

//---------------------------------------------------------------------------
// This function takes the passed sample and saves is to disk
//---------------------------------------------------------------------------
int writeS3I(S3Iheader* file2write)
{
  FILE *f;

  if ((f = fopen(file2write->filename, "wb")) == NULL)
    return 0;	// an error has occured

  fwrite(file2write, sizeof(S3Iheader)-4, 1, f);
  fwrite(file2write->sampledata, file2write->Length, 1, f);

  fclose(f);

  return 1;
}

int solveforM(float *M, unsigned char* sampledata, long oldsize)
{
  float *y, *r;
  float y0, M0;
  float l1, l2;
  float d;
  long x;
  unsigned pagediff = (page(oldsize))+1;
  float done;	// needed for progress meter
  unsigned char s0, s1, s2;

  if ((y = (float*)malloc(EMSpagesize)) == NULL) return 0;
  if ((r = (float*)malloc(EMSpagesize)) == NULL) return 0;
    // Let's hope we don't run into any memory problems...

    printf("\n  Free Memory: %ld bytes\n", coreleft());
    x = pagediff*3 + spage(oldsize) + 1;
    printf("  EMS pages allocated: %u (%ld bytes)\n", (unsigned)x, x*16384);

    printf("\n  calculating interpolative spline:");

  // The matrix is tridiagonal in nature. It can be easily converted into
  // a product of a right and a left matrix, which may, in turn, be solved.
  //
  // The Matrix has the value 2 as its diagonal elements (a), my below the
  // diagonal (c) and lamda above (b).
  //   lamda<j> := h<j+1>/(h<j>+h<j+1>)
  //   my<j> := 1-lamda<j>
  //   d<j> := 6/(h<j>+h<j+1>)*{(f<j+1>-f<j>)/h<j+1>-(f<j>-f<j-1>)/h<j>}
  // We choose h (the distance between two control points) constant and equal
  // to 1, since, in a sample the distance should be constant, but is
  // arbitrary. (therefore lamda = my = 0.5!)
  //
  // First, a new vector y is introduced and solved for by Ly = d:
  //   y<1> = d<1>/a<1>
  //   y<i> = (d<i> - c<i>*y<i-1>) / l<i>, with
  //     l<i> = a<i> - c<i>*b<i-1>/l<i-1>
  // Then, Rx = y is solved to get M:
  //   x<n> = y<n>
  //   x<i> = y<i> - r<i>*x<i+1>, with r<i> = b<i>/l<i> and l<1> = a<1>
  // Ready?!

  emmmap(emmhandle, 3*pagediff+spage(0), 0);
  emmget(0, (char*)sampledata, EMSpagesize);

  y[offset(1)] = (3.0*((float)sampledata[soffset(2)] - 2.0*(float)sampledata[soffset(1)] + (float)sampledata[soffset(0)]))/2.0;
  r[offset(1)] = lamda/2.0;
  l1 = 2.0;	// The diagonal elements are 2

  s0 = sampledata[soffset(1)];
  s1 = sampledata[soffset(2)];

  y0 = y[offset(1)];
  for (x = 2; x < (oldsize-1); x++) {
    l2 = 2.0 - my*lamda/l1;

    if (spage(x) != spage(x+1)) {
      emmmap(emmhandle, 3*pagediff+spage(x+1), 0);
      emmget(0, (char*)sampledata, EMSpagesize);
    }

    s2 = sampledata[soffset(x+1)];

    d = 3.0*((float)s2 - 2.0*(float)s1 + (float)s0);

    s0 = s1;
    s1 = s2;

    if (page(x-1) != page(x)) {
      emmmap(emmhandle, pagediff+page(x-1), 0);
      emmmove(0, (char*)y, EMSpagesize);
      emmmap(emmhandle, 2*pagediff+page(x-1), 0);
      emmmove(0, (char*)r, EMSpagesize);
    }

    y[offset(x)] = (d - my*y0)/l2;
    r[offset(x)] = lamda/l2;

    l1 = l2;
    y0 = y[offset(x)];

    // pure user-friendliness here - displaying a progress meter
    if ((x % 20000) == 2) {
      done = (float)x/(float)(oldsize<<1);
      gotoxy(37, wherey());
      printf("%3.1f%%", done*100);
    }
  }
  // store anything that hasn't been stored yet
  emmmap(emmhandle, pagediff+page(oldsize-2), 0);
  emmmove(0, (char*)y, EMSpagesize);
  emmmap(emmhandle, 2*pagediff+page(oldsize-2), 0);
  emmmove(0, (char*)r, EMSpagesize);

//  emmmap(emmhandle, page(oldsize-2), 0);
//  emmget(0, (char*)M, EMSpagesize);
  emmmap(emmhandle, pagediff+page(oldsize-2), 0);
  emmget(0, (char*)y, EMSpagesize);
  emmmap(emmhandle, 2*pagediff+page(oldsize-2), 0);
  emmget(0, (char*)r, EMSpagesize);

  M[offset(oldsize-1)] = 0;	// natural spline
  M0 = y[offset(oldsize-2)];
  M[offset(oldsize-2)] = M0;
  for (x = (oldsize-3); x > 0; x--) {
    if (page(x+1) != page(x)) {
      emmmap(emmhandle, page(x+1), 0);
      emmmove(0, (char*)M, EMSpagesize);
      emmmap(emmhandle, pagediff+page(x), 0);
      emmget(0, (char*)y, EMSpagesize);
      emmmap(emmhandle, 2*pagediff+page(x), 0);
      emmget(0, (char*)r, EMSpagesize);
    }

    M[offset(x)] = y[offset(x)] - r[offset(x)]*M0;
    M0 = M[offset(x)];

    // progress
    if ((x % 20000) == 0) {
      done = 0.5+((float)(oldsize-x)/(float)(oldsize<<1));
      gotoxy(37, wherey());
      printf("%3.1f%%", done*100);
    }
  }
  // store the last little bit
  M[offset(0)] = 0;	// this is what makes it a natural spline
  emmmap(emmhandle, page(0), 0);
  emmmove(0, (char*)M, EMSpagesize);

  gotoxy(37, wherey());
  printf("done  \n", done);

  farfree(y); farfree(r);
    // clean-up
  return 1;
}

unsigned char* resample(long oldrate, long newrate, long oldsize, long newsize)
{
  float far *M;
    // M are the moments, ie the second derivitives of the spline
  float M0, M1;
  unsigned char *newsample, *sampledata;
  float coefa, coefb, coefc, coefd;
  long x, seg, prevseg;
  float q, v, tmp;
  float done;
  unsigned pagediff = (page(oldsize))+1;
  unsigned char s0, s1;

  if ((sampledata = (char*)malloc(EMSpagesize)) == NULL) return NULL;
  if ((M = (float*)malloc(EMSpagesize)) == NULL) return NULL;
    // allocate just enough memory in the far heap to hold one page of EMS
    // If not enough memory was found, the function returns NULL to caller

  if (!solveforM(M, sampledata, oldsize)) return NULL;
	// The moments are calculated from a tridiagonal matrix

  if ((newsample = (unsigned char*)malloc(newsize*sizeof(unsigned char))) == NULL)
    return NULL;

  printf("  calculating new sample:");

  emmmap(emmhandle, 3*pagediff+spage(0), 0);
  emmget(0, (char*)sampledata, EMSpagesize);
  emmmap(emmhandle, page(0), 0);
  emmget(0, (char*)M, EMSpagesize);

  for (x = 0; x < newsize; x++) {
    q = (float)x*(float)oldrate/(float)newrate;
    seg = (long)q+1;
    if (seg > (oldsize-1)) seg = oldsize-1;
    if (x == 0) prevseg = seg-1;
    v = q - (float)(seg-1);

    // I'll reload portions of M when they're needed - i.e. when EMSpagesize
    // bytes have passed.
    if (page(prevseg) != page(seg-1)) {
      emmmap(emmhandle, page(seg-1), 0);
      emmget(0, (char*)M, EMSpagesize);
    }
    M0 = M[offset(seg-1)];

    if (spage(prevseg) != spage(seg-1)) {
      emmmap(emmhandle, 3*pagediff+spage(seg-1), 0);
      emmget(0, (char*)sampledata, EMSpagesize);
    }
    s0 = sampledata[soffset(seg-1)];

    prevseg = seg-1;

    // I'll reload portions of M when they're needed - i.e. when EMSpagesize
    // bytes have passed.
    if (page(prevseg) != page(seg)) {
      emmmap(emmhandle, page(seg), 0);
      emmget(0, (char*)M, EMSpagesize);
    }
    M1 = M[offset(seg)];
      // M1 is the moment corresponding to the current position

    if (spage(prevseg) != spage(seg)) {
      emmmap(emmhandle, 3*pagediff+spage(seg), 0);
      emmget(0, (char*)sampledata, EMSpagesize);
    }
    s1 = sampledata[soffset(seg)];

    prevseg = seg;	// on to the next round

    // The coefficients of the individual splines are only calculated
    // when needed. This is an advantage when the new samplerate is lower
    // than the original. It is, however, a disadvantage if the new rate
    // is higher. The major advantage as opposed to storing all coefficients
    // in a huge [(oldsize-1)*4*4] byte size array is memory! And that
    // overrides all other considerations, since <oldsize> can easily
    // be 200k.
    coefa = (float)s0;
    coefb = (float)s1 - (float)s0 - (2.0*M0+M1)/6.0;
    coefc = M0/2.0;
    coefd = (M1-M0)/6.0;

    tmp = coefa+v*(coefb+v*(coefc+v*coefd));
    if (tmp < 0) tmp = 0; else if (tmp > 255) tmp = 255;
    newsample[x] = (unsigned char)(tmp+0.5);

    if ((x % 20000) == 0) {
      done = (float)x/(float)newsize;
      gotoxy(37, wherey());
      printf("%3.1f%%", done*100);
    }
  }

  gotoxy(37, wherey());
  printf("done  \n", done);

  farfree(M); farfree(sampledata);
  return newsample;
}

// Good old main, right at the bottom, where it's supposed to be :-)
void main(int argc, char* argv[])
{
  long samplerate, newsize, newloopbegin, newloopend;
  S3Iheader *S3I;
  unsigned x;

  printf("ReSample v0.12 - (c) May 1996 by MLF/SLI\n");

  if ((S3I = (S3Iheader*)malloc(sizeof(S3Iheader)-4)) == NULL) {
    printf("\n  Out of Memory\n");
    exit(3);
  }

  if (argc < 2) {
    printf("  Usage: ReSample <filename> [samplerate]\n");
    printf("    filename   : Now that should be obvious :-)\n");
    printf("    samplerate : 1-2147483648, although not every samplerate is feasable...\n");
    printf("                 MAX, if the sample's size should be maximized (%ld bytes)\n", maxsize);
    printf("                 if not specified, the sample is resampled to %ld Hz\n", stdrate);
    exit(0);
  }

  if (!emmtest()) {
    printf("\n  No EMS driver found\n");
    exit(4);
  }
  if (!emmok()) {
    printf("\n  Expanded Memory Manager not available\n");
    exit(4);
  }

  x = readS3I(argv[1], S3I);
  switch (x) {
    case 1: printf("\n  Couldn't open %s\n", argv[1]);
	    if (!emmclose(emmhandle)) {
	      printf("\n  Could not deallocate EMS memory\n  please reboot\n");
	      exit(5);
	    } // leave no job unfinished :-)
	    exit(2); break; // :-)
    case 2: printf("\n  Out of Memory\n");
	    if (!emmclose(emmhandle)) {
	      printf("\n  Could not deallocate EMS memory\n  please reboot\n");
	      exit(5);
	    } // leave no job unfinished :-)
	    exit(3); break;
  }

  if (argc == 2) {
    samplerate = stdrate;
  } else {
    if (sscanf(argv[2], "%ld", &samplerate) == 0) {
      strlwr(argv[2]);
      if (!strcmp("max", argv[2]))
	samplerate = (long)(float)maxsize*(float)S3I->C2Spd/(float)S3I->Length;
      else {
	printf("\n  Invalid Option: %s\n", argv[2]);
	exit(0);
      }
    } else if (samplerate < 1) {
      printf("\n  Cannot resample at a negative rate!\n");
      if (!emmclose(emmhandle)) {
	printf("\n  Could not deallocate EMS memory\n  please reboot\n");
	exit(5);
      } // leave no job unfinished :-)
      exit(0);
    }
  }

  newsize = (long)((float)samplerate*(float)S3I->Length/(float)S3I->C2Spd);
  newloopbegin = (long)((float)samplerate*(float)S3I->LoopBegin/(float)S3I->C2Spd);
  newloopend = (long)((float)samplerate*(float)S3I->LoopEnd/(float)S3I->C2Spd);

  if (newsize > maxsize) {
    printf("\n  The new size (%ld bytes) exceeds the maximum (%ld bytes)\n", newsize, maxsize);
    if (!emmclose(emmhandle)) {
      printf("\n  Could not deallocate EMS memory\n  please reboot\n");
      exit(5);
    } // leave no job unfinished :-)
    exit(0);
  }

  printf("  resampling %s from %ld Hz to %ld Hz\n", argv[1], S3I->C2Spd, samplerate);
  printf("  size changes from %ld bytes to %ld bytes\n", S3I->Length, newsize);
  printf("  loop will be adjusted from (%ld; %ld) to (%ld; %ld)\n", S3I->LoopBegin, S3I->LoopEnd, newloopbegin, newloopend);

  if((S3I->sampledata = resample(S3I->C2Spd, samplerate, S3I->Length, newsize)) == NULL) {
    printf("\n  Out of Memory\n");
    if (!emmclose(emmhandle)) {
      printf("\n  Could not deallocate EMS memory\n  please reboot\n");
      exit(5);
    } // leave no job unfinished :-)
    exit(3);
  }

  // Adjust dimentions
  S3I->Length = newsize;
  S3I->LoopBegin = newloopbegin;
  S3I->LoopEnd = newloopend;
  S3I->C2Spd = samplerate;

  // write to disk
  if (!writeS3I(S3I))
    printf("\n  Error writing %s\n", argv[1]);

  if (!emmclose(emmhandle)) {
    printf("\n  Could not deallocate EMS memory\n  please reboot\n");
    exit(5);
  } // leave no job unfinished :-)

  farfree(S3I->sampledata); free(S3I);
}