//////////////////////////////////////////////////////////////////////
//TargetD64 - C64 archive related conversion tool and emulator frontend
//////////////////////////////////////////////////////////////////////
//Copyright (C) 1998, 1999  Karlheinz Langguth klangguth@netscape.net
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//////////////////////////////////////////////////////////////////////

//avoid multiple include of header file
#ifndef _APPLICATION_HEADER
#define _APPLICATION_HEADER

//concept of an application
//should be always used like that (example):
//class CFSpecialApplication : public CFSpecialApplication<CFTargetD64>
//it is necessary to pass CFSpecialApplication to the template to have
//access to the methods of CFSpecialApplication which can be callbacks
//for special options.
//- executes command line parsing
//Any option has a registered callback method which is called if
//option occurs. Callback methods are also called "handlers"
//P1: class of special application
template <class T>
class CFApplication
{
protected:
	//type for pointer to a member function of template param (callback)
	//handlers will normally be in class of template param!
	typedef void (T::* tMemberFunc)(const string&);

	//description of an option
	class CFCommandLineOption
	{
	public:
		//option string - has to be a single character
		string option;
		//true indicates option is followed by an argument
		bool bArg;
		//callback method to be called when option occurs
		tMemberFunc handler;
	public:
		CFCommandLineOption(const string& s, const bool bArgFollows
			, const tMemberFunc m)
			: option(s), bArg(bArgFollows), handler(m) {}
	};
	
	//contains all registered options
	typedef vector<CFCommandLineOption> tVectorOptions;
	
	//helper class for STL algorithm find_if
	//for searching an option in the option vector
	class CAlgoFindOptionByString
	{
	public:
		CAlgoFindOptionByString(const string& opt) : m_opt(opt) {}
		CAlgoFindOptionByString(const char c) : m_opt(string("") + c) {}

	public:
		//this operator is called for each element in array
		//R: true when elemt matches
		int operator() (CFCommandLineOption& opt)
		{
			//allow only one char options
			ASSERT(opt.option.size() <= 1);
			if (opt.option == m_opt)
				return true;
			else
				return false;
		}

	private:
		string m_opt; //option to find

	};

	//all handlers for command line options
	tVectorOptions m_options;

protected:
	//Registers standard option -h
	//Parameters as used for main()
	CFApplication(int argc, char *argv[], char *envp[]);

protected:
	//class derived from this template is abstract
	//specialisation has to implement main() which should implement
	//application processing
	//command line has been parsed while construction
	virtual int main(void) const = 0;
	//add a option and register handler for it
	//P1 I: option string (e.g. "h" for -h)
	//P2: true means argument follows option
	//P3: pointer to member function handling option
	void AddCommandOptionHandler(const string& param, const bool bArg, const tMemberFunc memFunc);
	//parse the command line (m_argc, m_argv)
	//look for options and call handler. call HandleUnknownOption if option
	//is not registered. call HandleStandardArg if no option but command
	//line string encountered. after "--" do not search for options.
	//if parameters have default values arguments come from m_arg, m_argv
	//P1 I: number of command line arguments
	//P2 I: array of command line arguments, index 0 for prog name
	void ParseCommandLine(int argc = -1, char **argv = NULL);
	//called when not registered option encountered.
	//P1 I: unexpected option string encountered
	virtual void HandleUnknownOption(const string& opt);
	//called when command line string not related to option encountered
	//P1 I: command lien string encountered
	virtual void HandleStandardArg(const string& arg);
	//called when option "-h" encountered
	//P1: not used - has to be there because of type tMemberFunc
	virtual void HandleOptionHelp(const string&);

private:
	int m_argc; //argc of main() function
	char **m_argv; //argv of main() function
	char **m_envp; //envp of main() function

};


template <class T> CFApplication<T>::CFApplication(int argc, char *argv[], char *envp[])
	: m_argc(argc),  m_argv(argv), m_envp(envp)

{
	//help option is default
#ifdef _MSC_VER
	AddCommandOptionHandler((string)"h", false, HandleOptionHelp);
#else
	AddCommandOptionHandler((string)"h", false, &HandleOptionHelp);
#endif
}


template <class T> void CFApplication<T>::AddCommandOptionHandler
	(const string& param, const bool bArg, const tMemberFunc memFunc)
{
	//link in another command line option with it's handler
	m_options.push_back(CFCommandLineOption(param, bArg, memFunc));
}


template <class T> void CFApplication<T>::HandleUnknownOption(const string& opt)
{
	cout << "Unknown option ignored: " << opt <<endl;
}


template <class T> void CFApplication<T>::HandleStandardArg(const string& arg)
{
	cout << "Argument ignored: " << arg <<endl;
}


template <class T> void CFApplication<T>::HandleOptionHelp(const string&)
{
	cout << "TODO: implement HandleOptionHelp" <<endl;
}


template <class T>
void CFApplication<T>::ParseCommandLine(int argc /*= -1*/
										, char **argv /*= NULL*/)
{
	bool moreOptions = true; //after -- no more options possible
	ASSERT(dynamic_cast<T *>(this));

	//if we have default parameters use the member variables
	if ((argc == -1) || (argv == NULL))
	{
		argc = m_argc;
		argv = m_argv;
	}
	int i = 1;
	while (i < argc)
	{
		//check if we have an option
		string test = argv[i];
		if ((test[0] == '-') && moreOptions)
		{
			if (test == "--")
			{
				//-- means no more options to follow
				moreOptions = false;
			}
			else
			{
				//take care of options concatenated like -xy for -x -y
				for (int nChIdx = 1; nChIdx < test.size(); nChIdx++)
				{
					//template enforces typename to make sure a type is meant
					typename tVectorOptions::iterator j =
						find_if(m_options.begin(), m_options.end()
						, CAlgoFindOptionByString(test[nChIdx]));
					if (j != m_options.end())
					{
						//found - now check for following argument
						if ((*j).bArg)
						{
							string arg = test.substr(nChIdx + 1, test.size() - nChIdx - 1);
							//if argument is not directly appended
							if (arg.size() == 0)
							{
								//take next parameter from command line as argument
								arg = argv[++i];
							}
							//call handler with option argument
							(dynamic_cast<T *>(this)->*(*j).handler)(arg);
							break; //no option to follow appended to argument
						}
						else
						{
							//no argument, call handler
							(dynamic_cast<T *>(this)->*(*j).handler)(string(""));
						}
					}
					else
					{
						HandleUnknownOption(test);
						break; //no further option after error happened
					}
				} //ENDFOR
				if (test == "-")
				{
					//this error will not be handled by the FOR loop
					HandleUnknownOption(test);
				}
			}
		}
		else
		{
			//we have no option so call the std. argument handler
			HandleStandardArg(test);
		}
		//check next parameter from command line
		i++;
	}			
}

#endif

