//////////////////////////////////////////////////////////////////////
//TargetD64 - C64 archive related conversion tool and emulator frontend
//////////////////////////////////////////////////////////////////////
//Copyright (C) 1998, 1999  Karlheinz Langguth klangguth@netscape.net
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//////////////////////////////////////////////////////////////////////

//Download locations for Win32 archivers
//unzip: ftp://ftp.uu.net/pub/archiving/zip/WIN32/unz540xN.exe
//gzip: ftp://ftp.uu.net/pub/archiving/zip/WIN32/gzip124xN.zip
//lha: http://helpdesk.uvic.ca/how-to/support/msdos/lha/lha255e.exe

//avoid multiple include of header file
#ifndef _ARCHIVE_HEADER
#define _ARCHIVE_HEADER

//general concept of archive (superclass)
//mind that a archive is ANY file processed by TD64 - this includes
//even C64 raw files (which are not resolvable of course)
//most important are Extract() to extract the containing files and
//ProcessFilesInArchive() to process the files gained by Extract()
class CFArchive
{
protected:
	//archive state - top archive is always ROOT
	//CREATED: constructor has run
	//EXTRACTED: files from the archive are extracted into working dir
	//PROCESSED: resolvable archives that have been extracted have been processed
	//           final archives that have been extracted have been put into emu images
	typedef enum { ROOT, CREATED, EXTRACTED, PROCESSED } tArchiveState;

public:
	//constructor splits P1 into components and sets working dir
	//P1 I: pathname of the archive within filesystem
	//P2 I: blocksize of the archive
	//P3 I: command to extract the archive in TD64 notation (placeholders)
	//      passed by the subclasses
	CFArchive(const string& pathname, const int blockSize = 0
		, const string& extractCmd = string(""));
	//ATTENTION:
	//destructor deletes all archives in m_files
	//those have to be created by CreateSubClassObject() or at least a new operator.
	//the same counts for the archives in ms_emuArchives
	~CFArchive();
protected:
	CFArchive();

public:
	//this method changes an archive to the topmost archive
	//and intializes all processing (reads environ variables,
	//creates directories, etc.)
	//IT HAS TO BE CALLED ONLY ONCE AND BEFORE ANYTHING ELSE IS DONE
	//P1 I: pathname of the archive within filesystem
	//P2 I: dir for emulator images
	void InitRootArchiveAndEnvironment(const string& pathname, const string& emuImageDir)
		throw (CFException);
	//extract the archive
	//use m_extractCommand with replaces placeholders as command
	virtual void Extract(void) throw (CFException);
	//extract archive
	//iterate over files in extraction directory make filename host conform
	//process each file found, create archive of it and do recursion
	//put final archives found into D64 emu image
	virtual void ProcessFilesInArchive(void) throw (CFException);
	//executes ProcessFilesInArchive() for the complete archive tree
	//collect all multipart archives of the whole archive tree
	//with CollectAndProcessMultipartArchives(). if multipart archive
	//completed process it and search again.
	void ProcessRootArchive(void)throw (CFException);
	//called when found an incomplete multipart
	//scans level for belonging multiparts searching from nearest
	//to farest neighbors (on the same level)
	//P1 I: the incomplete multipart archive to complete
	//P2 I: if 0 we are on the right level to scan
	//R: true means all part found (completed)
	bool TraverseAllArchivesOfLevelSearchingMultipart(
		CFArchive& candidate, unsigned int actualLevel);
	//archive factory checks class of P1 and creates object accordant
	//link in parent archive (pointer to) to ensure up traverse
	//P1 I: pathname of the archive within filesystem
	//P2 I: blocksize of the archive (passed to constructor)
	CFArchive *CreateSubClassObject(const string& path
		, const int blockSize = 0) const;
	//check if further archives included
	//R: true if unresolvable (finalized)
	inline bool Finalized(void) const { return m_final; }
	//R: blocksize
	inline int ReadBlockSize(void) const { return m_blockSize; }
	//R: filename which is pathname without dirname and extension
	//   e.g. filename of /a/b.x is b
	inline const string& GetFilename(void) const { return m_filename; }
	//R: complete pathname
	inline const string& GetPathname(void) const { return m_pathname; }
	//R: dirname which is directory part of pathname with trailing /
	//   e.g. dirname of /a/b.x is /a/
	inline const string& GetDirname(void) const { return m_dirname; }
	//R: extension of pathname with leading .
	//   e.g. extension of /a/b.x is .x
	inline const string& GetExtension(void) const { return m_extension; }
	//determine temporary directory within dir tree and create it
	//set m_dirname according to temporary dir as archive will reside here
	void CreateAndSetWorkdir(void) throw (CFException);
	//copy this archive into a target directory P1
	//did not make it a const method as CFZipC64Archive::
	//CopyThisArchiveToDirectory sets members
	//P1 I: directory to copy archive to
	virtual void CopyThisArchiveToDirectory(const string& directory)
		throw (CFException);
	//clean up the filesystem for the dir tree beneath this archive
	//call this method for all childs
	//clean up dir of this archive
	void CleanUpFilesystem(void) const throw (CFException);
	//clean up the temp dir tree starting with root archive
	//do the same as above but also clean image/misc dir
	void CleanUpRootArchiveFilesystem(void) const throw (CFException);
	//final archives are written into D64 emu images
	//close the actual emu image - write it back to disk and delete memory
	void CloseActualEmuImage(void);
	//R: pointer to parent archive for up traversing
	inline const CFArchive* GetParentArchivePointer(void) const { return m_pParentArchive; }
	//R: pointer to actual emu image for final archives to be put into
	inline CFDiskImage*& GetReferenceToD64EmuImagePointer(void)
		{ return m_pDiskImage; }
	//set the CBM fileinfo
	//P1 I: CBM fileinfo
	inline void SetFileInfoCBM(const CFFileInfoCBM& fileInfoCBM)
		{ m_fileInfoCBM = fileInfoCBM; }
	//get the CBM fileinfo
	//R: CBM fileinfo
	inline const CFFileInfoCBM& GetFileInfoCBM(void) const
		{ return m_fileInfoCBM; }
	//create a unique pathname for an emulator image
	//this pathname may be derived from the archive name (option -k)
	//or can be targetXXX.d64 (without -k)
	const string& BuildUniqueEmuImageName(void) const throw (CFException);
	//add a new archive contained in this archive to the vector
	//P1 I: pointer to archive to add
	inline void AddNewArchiveToChildren(CFArchive * const newEntry)
	{ m_files.push_back(newEntry); }
protected:
	//create the temp dir fitting into the temp dir tree
	//R: created directory name
	string& NameAndCreateTmpDir(void) throw (CFException);
	//make a transition on m_state to P1 - mind that ROOT remains ROOT
	//P1 I: state to move to
	//R: new state
	tArchiveState TransitionTo(const tArchiveState newState);
private:
	//after an entry has been found in the extraction directory
	//use this function to process this entry
	//create archive of it and process it if needed.
	//P1: pointer to directory entry read by readdir()
	//R: true means that a new archive has been found and processed
	//   false means nothing done
	bool ProcessSingleFileOfArchive(struct dirent *pEntry)
		throw (CFException);
	//split P1 into components and write it to members
	//P1 I: pathname to be put into components
	void SetPathComponents(const string& pathname);
	//multipart archive processing takes place while normal
	//processing goes on. after all processing is finished
	//check if there is still an uncompleted multipart archive
	//if so write out a warning
	void ProduceWarningsForIncompleteMultipart(void);

public:
	//execute P1 as a shell command
	//if TRACE is indicated dump outputs on console
	//else redirect output into misc/redirected...
	//for M$ build a batchfile and execute this batchfile as no && and ;
	//for command.com
	//P1 I: command to execute
	//R: return code of system() execution
	static int SystemRedirectOutput(const char *command) throw (CFException);
	//R: reference to first produced emu image archive
	static const CFArchive& GetFirstEmuImageArchive(void);
	//R: dirname of the emu image directory
	static inline string& GetEmuImageDir(void) { return ms_emuImageDir; }
	//R: dirname of the temporary directory used by TargetD64
	static inline string& GetTmpDir(void) { return ms_tmpdir; }
	//set the naming mode of produced D64 images
	//P1 I: true indicates to use archive name as part of emu image name
	//      option (-k), false leads to targetXXX.d64
	static inline void SetEmuImageNameDueOriginal(const bool keep)
		{ ms_bEmuImageNameDueOriginal = keep; }
	//set usage of external helpers for otherwise builtin archives
	//P1 I: true indicates usage of external helpers (anachronistic)
	static inline void SetUsageExternalHandler(const bool state)
		{ ms_bUseExternalHelper = state; };
	//replace the placeholders %s[1-5] in P1 by accordant parameters
	//for M$ also consider %d[1-5] and %l[1-5] (convert Unix->Win filenames)
	//s for 8.3 filename, l for long filename, d for drive (e.g. c:)
	//P1 I: string with placeholders to be replaced
	//P2-6 I: strings for accordant placeholders (P2<->%s1 ans so on)
	//this stuff is confusing:
	//a prefix of "basename " for P2-P6 marks that basename of filename
	//is wanted. this is needed to get a short and long filename version
	//for Win32 otherwise we could have created the basename before calling
	//R: processed string
	static string& ReplacePlaceHoldersByStrings(
		const string& inString
		, const string& s1
		, const string& s2
		, const string& s3 = ""
		, const string& s4 = ""
		, const string& s5 = "");
#ifdef _MSC_VER
	//replace slash occurrences by backslash
	//P1: filename to be converted
	//R: converted filename
	static string ConvertFilenameUnixToWin(const string& filename);
	//same as above but convert result into a 8.3 format (short)
	//P1: filename to be converted
	//R: converted filename
	static string ConvertFilenameUnixToWinShort(const string& filename)
		throw (CFException);
#endif
private:
#ifdef _MSC_VER
	//build a batchfile in miscdir containing statements of P1
	//single commands are seperated by ; e.g. echo a; echo b
	//P1: single command seperated by ; to be executed in a row
	//R: return code system() of batchfile call
	static int BuildBatchFileAndExecuteSystem(const string& command)
		throw (CFException);
#endif
	//read environment variables for archiving and set members
	//accordingly.
	static void EvaluateEnvironmentVariables(void)
		throw (CFException);
	//iterate over each files and dirs in directory P1
	//and delete them.
	//P1 I: dirname of dir to be cleaned
	static void CleanUpDirectory(string dirName) throw (CFException);
	//remove characters from a filename that are not allowed for host
	//P1 I: filename to be conformant
	//R: conformant filename
	static string& BuildHostConformFilename(const string& filename);

protected:
	CFArchive *m_pParentArchive; //pointer to parent archive (for traversing up)
	tArchiveState m_state; //state of archive (root will always stay ROOT)
	bool m_final; //false if further archives included
	string m_extractCommand; //command to extract files from archive
	string m_pathname; //complete pathname
	string m_filename; //filename of archive in host filesystem
	string m_dirname; //directory of m_pathname
	string m_extension; //extension of m_pathname
	string m_workdir; //directory for operations on archive
	int m_blockSize; //blocksize of archive file
	//if CBM filename info is available e.g. for a file from a lynx archive
	//store this info here to use it when writing file into D64 image
	CFFileInfoCBM m_fileInfoCBM;
	//pointer to disk image containing in which final archives of this archive are put
	//NULL means no such image opened
	CFDiskImage *m_pDiskImage;
	vector<CFArchive *> m_files; //all files of archive for processing
private:
	//to sort emulator archives in the request order - this is not necessarily
	//the order in which they have been constructed because emu images can be yet existing
	//as a disk image BUT NOT YET as an archive. Important is the order of image
	//creation. Store this order here.
	mutable unsigned int m_nRequestOrderIndex;

protected:
	static vector<CFArchive *> ms_emuArchives; //archive(s) to pass to the emulator
	//true indicates usage of external helper applications which is old style
	//and not recommended as a few things are improved for the builtins
	//NOT FOR defined generic archives (ALWAYS external helpers used).
	static bool ms_bUseExternalHelper;
private:
	static string ms_emuImageDir; //here to place the emulator images
	static string ms_miscDir; //here to place temporary output
	static string ms_emuArchiveBasename; //basename of the archive passed to the emu
	static int ms_emuArchiveCount; //count of the archive passed to the emu
	static bool ms_bEmuImageCleanup; //false avoids cleaning of emu image archive
	//false counts result images from target000 on
	//true uses part of the source archive for name
	static bool ms_bEmuImageNameDueOriginal;
	static string ms_tmpdir; //ABSOLUTE path of temporary directory
};


//a little POSIX conformance for M$
#ifdef _MSC_VER

//provide UNIX scan directory functions
#include <windows.h>

typedef HANDLE DIR;

struct dirent
{
	char d_name[MAX_PATH];
};

DIR *opendir(const char *name);
int closedir(DIR *dir);
struct dirent *readdir(DIR *dir);

#define S_ISREG(a) ((a) & _S_IFREG)
#define S_ISDIR(a) ((a) & _S_IFDIR)

#endif

#endif
