//////////////////////////////////////////////////////////////////////
//TargetD64 - C64 archive related conversion tool and emulator frontend
//////////////////////////////////////////////////////////////////////
//Copyright (C) 1998, 1999  Karlheinz Langguth klangguth@netscape.net
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//////////////////////////////////////////////////////////////////////

#ifndef _C64_ARCHIVE_HEADER
#define _C64_ARCHIVE_HEADER


//concept of a C64 related archive in opposite to a host
//archive (.zip, .lha)
class CFC64Archive : public CFArchive
{
public:
	//P1 I: pathname of archive in host filesystem
	//P2 I: blocksize of archive
	//P3 I: extract command (with placeholders)
	CFC64Archive(const string& path, const int blockSize = 0
		, const string& extractCmd = string(""))
	: CFArchive(path, blockSize, extractCmd) {}
};


//concept of a D64 archive (raw dump of a 1541 disk)
//completly built in - no way to use an extract command
class CFD64Archive : public CFC64Archive
{
public:
	//P1 I: pathname of archive in host filesystem
	//P2 I: blocksize of archive
	inline CFD64Archive(const string& path, const int blockSize = 0)
		: CFC64Archive(path, blockSize)
		, m_bMoreFilesThanExtractedOnDisk(false) {}

protected:
	//extract all extractable files in D64
	//overloaded because no extract command is used (built in)
	virtual void Extract(void) throw (CFException);
	//process all extracted files
	//if unresolvable file remaining in D64 use it as emu image
	virtual void ProcessFilesInArchive(void) throw (CFException);

public:
	//checks if P1 is of this archive class
	//P1 I: pathname of archive in host filesystem
	static bool CheckFileIsOfClass(const string& filename);

private:
	//vector of all extracted archives from disk (with CBM file info)
	vector<CFExtractedFileInfo> m_extractedFiles;
	//true when D64 contains more files than have been extracted (e.g. REL)
	//this means that this d64 image has to be passed to emu unconditionally
	bool m_bMoreFilesThanExtractedOnDisk;

public:
	//constant size of real data (exclusive sector link) in a disk block 
	static int BLOCKSIZE; //blocksize in bytes
	//constant capacity of disk in blocks
	static int BLOCKCAPACITY;
};


//concept of a X64 archive
//A X64 archive can be extracted into a D64 archive
//skip header
//Attention: I had this class specialized from CFD64Archive before
//and morphed objects into the superclass. so the original X64 archive
//disappeared (morphed). while cleaning up the working directory
//of the disappeared X64 archive has not been cleaned up!! Now
//X64s are handled in the common way.
class CFX64Archive : public CFC64Archive
{
public:
	inline CFX64Archive(const string& path, const int blockSize = 0)
		: CFC64Archive(path, blockSize) {}

public:
	//extract the D64 image - there is always a 1:1 relation
	virtual void Extract(void) throw (CFException);
	//process the extracted D64 image
	virtual void ProcessFilesInArchive(void) throw (CFException);

public:
	//checks if P1 is of this archive class
	//P1 I: pathname of archive in host filesystem
	static bool CheckFileIsOfClass(const string& filename);
};


class CFT64Archive : public CFC64Archive
{
public:
	inline CFT64Archive(const string& path, const int blockSize = 0)
		: CFC64Archive(path, blockSize) {}

protected:
	//extract all extractable files in D64
	//overloaded because no extract command is used (built in)
	virtual void Extract(void) throw (CFException);
	//process all extracted files - put final archives into D64 images
	//take care that more D64 images are produced if one disk is exceeded
	virtual void ProcessFilesInArchive(void) throw (CFException);

public:
	//checks if P1 is of this archive class
	//P1 I: pathname of archive in host filesystem
	static bool CheckFileIsOfClass(const string& filename);

private:
	//vector of all extracted archives from tape (with CBM file info)
	vector<CFExtractedFileInfo> m_extractedFiles;
};


class CFLnxArchive : public CFC64Archive
{
public:
	//P1 I: pathname of archive in host filesystem
	//P2 I: blocksize of archive
	inline CFLnxArchive(const string& path, const int blockSize = 0)
		: CFC64Archive(path, blockSize, ms_staticExtractCommand) {}

protected:
	//extract all files of the archive into filesystem and remember
	//original CBM file data (for putting into D64 image later)
	//use VICE function for this.
	//ONLY if external helpers are indicated:
	//extract all files of the archive using an external helper
	//application. this is old style and only left for compatibility
	virtual void Extract(void) throw (CFException);
	//call the superclass ProcessFilesInArchive()
	//ONLY if external helpers are indicated:
	//a D64 image is produced which will be further processed
	//OTHERWISE all archives are extracted into filesystem and remembered with
	//CBM fileinfo. iterate over this vector.
	virtual void ProcessFilesInArchive(void) throw (CFException);

public:
	//checks if P1 is of this archive class
	//P1 I: pathname of archive in host filesystem
	static bool CheckFileIsOfClass(const string& filename);
	//set the extract command (with placeholders)
	//P1 I: extract command (with placeholders)
	inline static void SetExtractCommand(const string &cmd) { ms_staticExtractCommand = cmd; }

private:
	//vector of all extracted archives from lynx (with CBM file info)
	vector<CFExtractedFileInfo> m_extractedFiles;

private:
	//extract command (with placeholders)
	static string ms_staticExtractCommand;
};


//concept of Pc64 Archives .p00, .s00, .d00, .u00, .r00
class CFPc64Archive : public CFC64Archive
{
public:
	//P1 I: pathname of archive in host filesystem
	//P2 I: blocksize of archive
	CFPc64Archive(const string& path, const int blockSize = 0)
		throw (CFException);
	inline ~CFPc64Archive() { delete m_pInStream; }

public:
	//extract the raw c64 data contained in this file into a single file
	//overloaded because no extract command is used (built in)
	virtual void Extract(void) throw (CFException);
	//process the extracted c64 raw file (may be sesolvable archive)
	//mind that extracted c64 archive is created as BROTHER (not son) of
	//this archive. This will ensure that it is not put into a single (new)
	//D64 image. It would be nonsense to have a 1:1 relation for CFPc64Archive
	//to emulator D64 image.
	virtual void ProcessFilesInArchive(void) throw (CFException);

public:
	//checks if P1 is of this archive class
	//P1 I: pathname of archive in host filesystem
	static bool CheckFileIsOfClass(const string& filename);

private:
	//I had this stream first as a direct (non pointer) member
	//of this class - strange enough g++ caused an segmentation
	//violation while cleaning up the executable, even when the
	//member variable has not been accessed by me (while flushing IO).
	//this happened outside the application code. I could not figure
	//out what I have done wrong. so I have introduced this pointer.
	//the stream is created and deleted explicitly and in control of
	//the application. this helped - strange!
	ifstream *m_pInStream; //to read this archive from filesystem
	unsigned int m_contentFileByteSize;
	string m_contentFilename;
};


//concept of C64 zipped archives [1-4]!
//ATTENTION: each multipart archive has to inherit from CFMultiPart
class CFZipC64Archive : public CFC64Archive, public CFMultiPart
{
public:
	CFZipC64Archive(const string& path, const int blockSize = 0);

protected:
	//parts may be spread over several directories
	//move each single part into a common directory
	//do the extraction and move back each single part
	//where it came from. in this way it will not confuse
	//the directory scan which may go on for this directory
	//as the directory before and after the extraction will be
	//exactly the same.
	void ExtractWithExternalHelper(void) throw (CFException);
	//extract the single parts of the multipart archive into a D64
	//file. use VICE function for this.
	virtual void Extract(void) throw (CFException);
	//process the files in archive
	//only do something if archive complete
	virtual void ProcessFilesInArchive(void) throw (CFException);
	//check if P1 belongs to this multipart
	//if so assimilate it
	//P1 I: candidate to be tested
	//R: true if belongs
	virtual bool CheckBelongingAndMergeIfBelongs(const CFArchive& archive);
	//copy all four parts of archive to directory P1
	//P1 I: directory to copy parts to
	virtual void CopyThisArchiveToDirectory(const string& directory)
		throw (CFException);

public:
	//checks if P1 is of this archive class
	//P1 I: pathname of archive in host filesystem
	static bool CheckFileIsOfClass(const string& filename);
	//set the extract command (with placeholders)
	//P1 I: extract command (with placeholders)
	inline static void SetExtractCommand(const string &cmd) { ms_staticExtractCommand = cmd; }

private:
	//extract command (with placeholders)
	static string ms_staticExtractCommand;
};


class CFFinalArchive : public CFC64Archive
{
public:
	//P1 I: pathname of archive in host filesystem
	//P2 I: blocksize of archive
	inline CFFinalArchive(const string& path, const int blockSize = 0)
	: CFC64Archive(path, blockSize)
	{ m_final = true; }

public:
	//extraction of final archive does nothing
	virtual void Extract(void) throw (CFException);
	//processing of final archive may not happen
	virtual void ProcessFilesInArchive(void) throw (CFException);
};

#endif

