//////////////////////////////////////////////////////////////////////
//TargetD64 - C64 archive related conversion tool and emulator frontend
//////////////////////////////////////////////////////////////////////
//Copyright (C) 1998, 1999  Karlheinz Langguth klangguth@netscape.net
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
//COMPILE SWITCHES
// _MSC_VER
// indicates MS compiler
// _DEBUG
// for debug version which activates ASSERT and TRACE
// TD64_MODIFIED
// marks changes in foreign sources to fit into TargetD64
// must be set everywhere because also used for header files

#ifdef _MSC_VER
#pragma warning(disable:4786) //identifier truncation warning
#endif

#ifdef _MSC_VER
#include <io.h>
#else
#include <stdio.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>

#include <algorithm>
#include <vector>
#include <string>
#ifdef _MSC_VER
#include <typeinfo.h>
#include <iostream>
#include <fstream>
#include <strstream>
#else
#include <typeinfo>
#include <iostream.h>
#include <fstream.h>
#include <strstream.h>
#endif

using namespace std;

#include "Exception.h"
#include "Image.h"
#include "DiskImage.h"
#include "Tracing.h"

//1541 disk geometry
const unsigned int tracks = 35;
const unsigned int sectorPerTrack[tracks] =
{
    21, //1
	21, 
	21,
	21,
	21,
	21,
	21,
	21,
	21,
	21, //10
	21,
	21,
	21,
	21,
	21,
	21,
	21,
	19,
	19,
	19, //20
	19,
	19,
	19,
	19,
	18,
	18,
	18,
	18,
	18,
	18, //30
    17,
	17,
	17,
	17,
	17
};
const CFBlockAddress controlBlock(18, 0);
const CFBlockAddress firstDirBlock(18, 1);


//use 1541 geometry
CFDiskLayout CFDiskImage::m_geom =
	CFDiskLayout(tracks, sectorPerTrack
					, controlBlock
					, firstDirBlock
					, (char)0xa0
					, (unsigned char)0xff);



//-------------------------------------------------------------------
//CFDiskBlock
//-------------------------------------------------------------------
CFDiskBlock::CFDiskBlock(void)
{
	for (int i = 0; i < BLOCKSIZE; i++)
		m_content[i] = 0;
}


unsigned char& CFDiskBlock::operator[](const unsigned int index)
{
	ASSERT(index < BLOCKSIZE);
	return m_content[index];
}


unsigned int CFDiskBlock::GetHeaderAndDataFromStream(istream& in) throw (CFException)
{
	int i = 0;
	char ch;
	//mind that in.get() is only evaluated when i < BLOCKSIZE
	while ((i < BLOCKSIZE) && in.get(ch))
	{
		(*this)[i++] = ch;
	}
	if (in.bad())
	{
		exc = CFException(CFException::FILE_OPERATION_FAILED,
			__LINE__, __FILE__, "input stream", "reading");
		throw exc;
	}
	return i;
}


unsigned int CFDiskBlock::GetRealDataOnlyFromStream(istream& in)
{
	int i = LINKSIZE; //skip sector link
	char ch;
	//mind that in.get() is only evaluated when i < BLOCKSIZE
	//attention: we have started with LINKSIZE so we have to check for BLOCKSIZE
	while ((i < BLOCKSIZE) && in.get(ch))
	{
		(*this)[i++] = ch;
	}
	if (in.bad())
	{
		exc = CFException(CFException::FILE_OPERATION_FAILED,
			__LINE__, __FILE__, "input stream", "reading");
		throw exc;
	}
	return i - LINKSIZE;
}


unsigned int CFDiskBlock::PutRealDataOnlyToStream(ostream& out) const throw (CFException)
{
	CFBlockAddress link = ReadSectorLink();

	int length = 0;
	if (link.IsLastBlock())
	{
		length = link.m_sector - 1;
		//last block has to contain data but not more than 254 bytes
		if ((length <= 0) ||
			(length > BLOCKSIZE - LINKSIZE))
		{
			return 0;
		}
	}
	else
	{
		if (!link.IsValidLink())
		{
			return 0;
		}
		else
		{
			//always BLOCKSIZE - LINKSIZE data available in a block with sector linkage
			length = BLOCKSIZE - LINKSIZE;
		}
	}
	out.write(reinterpret_cast<const char *>(&m_content[2]), length);

	if (out.bad())
	{
		exc = CFException(CFException::FILE_OPERATION_FAILED,
			__LINE__, __FILE__, "output stream", "writing");
		throw exc;
	}
	return length;
}


unsigned int CFDiskBlock::PutHeaderAndDataToStream(ostream& out) const throw (CFException)
{
	out.write(reinterpret_cast<const char *>(m_content), BLOCKSIZE);

	if (out.bad())
	{
		exc = CFException(CFException::FILE_OPERATION_FAILED,
			__LINE__, __FILE__, "output stream", "writing");
		throw exc;
	}
	return BLOCKSIZE;
}


//-------------------------------------------------------------------
//CFBlockAddress
//-------------------------------------------------------------------


CFBlockAddress CFBlockAddress::operator++(void)
{
	ASSERT((m_track <= CFDiskImage::m_geom.GetTracks())
		&& (m_track > 0) && (m_sector < CFDiskImage::m_geom[m_track]));
	if (m_sector < CFDiskImage::m_geom[m_track] - 1)
	{
		m_sector++;
	}
	else
	{
		if (m_track == CFDiskImage::m_geom.GetTracks())
		{
			*this = OUT_OF_BLOCKS;
		}
		else
		{
			m_track++;
			m_sector = 0;
		}
	}
	return *this;
}


bool CFBlockAddress::operator==(const CFBlockAddress& a) const
{
	return ((m_track == a.m_track) && (m_sector == a.m_sector));
}


bool CFBlockAddress::operator!(void) const
{
	return (!(*this == OUT_OF_BLOCKS));
}


bool CFBlockAddress::IsValidLink(void) const
{
	return ((m_track <= CFDiskImage::m_geom.GetTracks())
		&& (m_track > 0)
		&& (m_sector < CFDiskImage::m_geom[m_track]));
}


bool CFBlockAddress::IsDirectoryLink(void) const
{
	unsigned int dirTrack = CFDiskImage::m_geom.GetFirstDirBlockAddr().m_track;
	if ((*this == CFDiskImage::m_geom.GetControlBlockAddr())
		|| (m_track != dirTrack)
		|| (m_sector >= CFDiskImage::m_geom[dirTrack]))
		return false;
	else
		return true;
}


//-------------------------------------------------------------------
//CFDirectoryEntry
//-------------------------------------------------------------------


CFDirectoryEntry::CFDirectoryEntry(const CFDiskLayout::tDirEntry& dirEntry)
	: m_locked(dirEntry.locked)
	, m_closed(dirEntry.closed)
	, m_size(dirEntry.lengthLow + 256 * dirEntry.lengthHigh)
	, m_filetype(static_cast<CFDiskLayout::tFiletype>(dirEntry.filetype))
	, m_firstBlockAddr(dirEntry.dataTrack, dirEntry.dataSector)
{
	//limit the CBM filename to 16 characters
	//there are is no trailing '\0' for CBM strings
	char tmp[sizeof(dirEntry.name) + 1];
	strncpy(tmp, dirEntry.name, sizeof(dirEntry.name));
	tmp[sizeof(dirEntry.name)] = '\0';
	m_filename = CFCbmFilename(tmp);
}


CFDirectoryEntry::CFDirectoryEntry(const string& filename, const unsigned int blockSize
		, const CFBlockAddress& firstBlock
		, const CFDiskLayout::tFiletype filetype
		, const bool locked, const bool closed)
	: m_locked(locked)
	, m_closed(closed)
	, m_filename(filename)
	, m_size(blockSize)
	, m_filetype(filetype)
	, m_firstBlockAddr(firstBlock)
{
}


CFDirectoryEntry::operator CFDiskLayout::tDirEntry&(void) const
{
	static CFDiskLayout::tDirEntry ret;

	ret.notused1[0] = 0x0;
	ret.notused1[1] = 0x0;
	ret.filetype = m_filetype;
	ret.notused2 = 0;
	ret.locked = m_locked;
	ret.closed = m_closed;
	ret.dataTrack = m_firstBlockAddr.m_track;
	ret.dataSector = m_firstBlockAddr.m_sector;
	//filename conversion from FS to cbm must happen outside
	strncpy(ret.name, m_filename.c_str(), sizeof(ret.name));
	ret.name[16];
	ret.notused3[0] = 0x0;
	ret.notused3[1] = 0x0;
	ret.notused3[2] = 0x0;
	ret.notused3[3] = 0x0;
	ret.notused3[4] = 0x0;
	ret.notused3[5] = 0x0; 
	ret.notused3[6] = 0x0;
	ret.newTrack = 0;
	ret.newSector = 0;
	ret.lengthLow = m_size % 256;
	ret.lengthHigh = m_size / 256;

	return ret;
}

//-------------------------------------------------------------------
//CFDiskImage
//-------------------------------------------------------------------


CFDiskImage::CFDiskImage(const string& filename
	, const CFCbmFilename& diskname
	, const char diskId[2])
	: m_filename(filename)
	, m_control(diskname, diskId)
{
	//a few constants are double defined - check equality.
	ASSERT(sizeof(((CFDiskLayout::tDirEntry *)NULL)->name)
		== CFCbmFilename::m_nMaxSize);
	ASSERT(m_geom.GetFileTermChar()
		== CFCbmFilename::m_termChar);

	tSectorsInTrack diskBlock;
	CFDiskBlock block;

	//Add all block buffers for the disk
	for (int i = 1; i < m_geom.GetTracks() + 1; i++)
	{
		//add a new track
		m_diskBlock.push_back(diskBlock);
		for (int j = 0; j < m_geom[i]; j++)
		{
			//add a new sector to the track
			m_diskBlock[m_diskBlock.size() - 1].push_back(block);
		}
	}
	//set length of first directory block to default
	(*this)[m_geom.GetFirstDirBlockAddr()]
		.WriteSectorLink(CFBlockAddress(0, m_geom.GetLastDirBlockLength()));
}


CFDiskBlock& CFDiskImage::operator[](const CFBlockAddress& block)
{
	return m_diskBlock[block.m_track - 1][block.m_sector];
}


void CFDiskImage::Attach(const string& filename) throw (CFException)
{
	ifstream in(filename.c_str(), ios::in | ios::binary);
	if (!in)
	{
		exc = CFException(CFException::FILE_OPEN_FAILED,
			__LINE__, __FILE__, filename, "reading");
		throw exc;
	}

	CFBlockAddress sectorIterator;

	int bytesWritten = 0;
	while (!sectorIterator && in)
	{
		bytesWritten = (*this)[sectorIterator].GetHeaderAndDataFromStream(in);
		++sectorIterator;
	}
	if (in.bad())
	{
		exc = CFException(CFException::FILE_OPERATION_FAILED,
			__LINE__, __FILE__, filename, "reading");
		throw exc;
	}
	//diskimage must fill all blocks completely
	if ((bytesWritten != BLOCKSIZE) || in.eof())
	{
		exc = CFException(CFException::DISK_IMAGE_SIZE_INCORRECT,
			__LINE__, __FILE__, filename, "size too small (174848 bytes expected)");
		throw exc;
	}
	//as reading is cut of at BLOCKSIZE bytes it has not been read
	//across the end - do it to get EOF
	char ch = in.get();
	if (!in.eof())
	{
		exc = CFException(CFException::DISK_IMAGE_SIZE_INCORRECT,
			__LINE__, __FILE__, filename,  "size too big (174848 bytes expected)");
		throw exc;
	}
	m_filename = filename;
	
	//process and attach the control block
	try {
	m_control.AttachControlBlock();
	}
	catch (CFException& actExc)
	{
		actExc.WriteOutExceptionWithWarningHeader(cerr);
	}
	//fetch the directory information
	try {
	ParseAndCheckBlockAllocDirectory();
	}
	catch (CFException& actExc)
	{
		actExc.WriteOutExceptionWithWarningHeader(cerr);
	}
}


void CFDiskImage::ParseAndCheckBlockAllocDirectory(void) throw (CFException)
{
	CFException::tException excType =
		CFException::EXCEPTION_INVALID;
	string param2;
	vector<CFBlockAddress> linkTracker; //to avoid circular reads

	CFBlockAddress link;

	//point to the first directory block
	link = m_geom.GetFirstDirBlockAddr();

	do
	{
		//check if this link has already been traversed
		if (find(linkTracker.begin(), linkTracker.end(), link) !=
			linkTracker.end())
		{
			excType = CFException::DISK_IMAGE_HAS_CIRCULAR_LINKAGE;
			param2 = "Disk directory";
		}
		else
		{
			CFDiskBlock& actualBlock = (*this)[link];
			//check if directory block is allocated - must be!
			if (!m_control.TestBlockAllocation(link))
			{
				m_control.AllocBlock(link);
				excType = CFException::DISK_DIRECTORY_ALLOC_ERROR;
				param2 = (string)"track: " + CFException::IntToString(link.m_track)
					+ " sector: " + CFException::IntToString(link.m_sector);					
			}
			//now we asume that we have found a valid directory block
			//so we process it

			//lay the logical structure over the raw content
			CFDiskLayout::tDirEntry *aEntry
				= reinterpret_cast<CFDiskLayout::tDirEntry *>(actualBlock.GetBlockContentPointer());
			ASSERT((BLOCKSIZE % sizeof(CFDiskLayout::tDirEntry)) == 0);
			//loop over complete block
			for (int i = 0; i < BLOCKSIZE / sizeof(CFDiskLayout::tDirEntry); i++)
			{
				//we ignore deleted files
				if ((aEntry[i].filetype != CFDiskLayout::DEL)
					|| (aEntry[i].locked == 1))
				{
					//append the new file to the directory
					CFDirectoryEntry tmp(aEntry[i]);
					m_directory.push_back(tmp);
				}
			}

			linkTracker.push_back(link); //we processed actual block (remember that)
			link = actualBlock.ReadSectorLink(); //get next block

			//do a few consistency checks
			if (link.IsLastBlock())
			{
				//last directory block has always same size
				if (link.m_sector != m_geom.GetLastDirBlockLength())
				{
					excType = CFException::DISK_IMAGE_DIRECTORY_CORRUPTED;
					param2 = "Unallowed last block size";
				}
			}
			else
			{
				//directory must link into directory track and nowhere else
				if (link.IsValidLink())
				{
					if (!link.IsDirectoryLink())
					{
						excType = CFException::DISK_IMAGE_DIRECTORY_CORRUPTED;
						param2 = "Links outside directory track";
					}
				}
				else
				{
					excType = CFException::DISK_IMAGE_DIRECTORY_CORRUPTED;
					param2 = "Corrupted link (not within disk geometry)";
				}
			}
		}
	} while (!link.IsLastBlock()
		&& ((excType == CFException::EXCEPTION_INVALID)
		//only missing allocation - parse on
		|| (excType == CFException::DISK_DIRECTORY_ALLOC_ERROR)));

	if (excType != CFException::EXCEPTION_INVALID)
	{
		exc = CFException(excType, __LINE__, __FILE__
			,m_filename,  param2);
		throw exc;
	}
}


void CFDiskImage::ExtractFile(const CFDirectoryEntry& entry
		, const string& dirname
		, CFExtractedFileInfo& extractInfo
		, const string& filename /*= ""*/) const throw (CFException)
{
	ASSERT(!dirname.empty());

	//strip off trailing / if existing
	string fsDirname = 	dirname;
	if (fsDirname[fsDirname.size() - 1] == '/')
	{
		fsDirname.resize(fsDirname.size() - 1);
	}

	//check existance of directory
	struct stat buf;
	if (stat(fsDirname.c_str(), &buf) != 0)
	{
		exc = CFException(CFException::FILE_STAT_FAILED,
			__LINE__, __FILE__, fsDirname, "", errno);
		throw exc;
	}
	//only directory is allowed
	if (!S_ISDIR(buf.st_mode))
	{
		exc = CFException(CFException::UNEXPECTED_FILE_TYPE,
			__LINE__, __FILE__, fsDirname, CFException::IntToString(buf.st_mode));
		throw exc;
	}

	//a few consistency checks
	if ((entry.GetFiletype() != CFDiskLayout::PRG) && (entry.GetFiletype() != CFDiskLayout::SEQ))
	{
		exc = CFException(CFException::DISK_IMAGE_FILETYPE_NOT_SUPPORTED,
			__LINE__, __FILE__, m_filename, entry.GetFilename());
		throw exc;
	}
	CFBlockAddress link = entry.GetFirstFileDataBlockAddr();
	if (!link.IsValidLink())
	{
		exc = CFException(CFException::DISK_FILE_LINK_ERROR,
			__LINE__, __FILE__, m_filename, entry.GetFilename());
		throw exc;
	}

	ofstream out;
	string fsFilename;
	if (filename != "")
	{
		fsFilename = fsDirname + "/" + filename;
		//if explicit filename is given use it in any case
		out.open(fsFilename.c_str(), ios::out | ios::binary);
	}
	else
	{		
		fsFilename = entry.GetFilename().ConvertThisToFilesystemFilename();
		fsFilename = fsDirname + "/" + fsFilename;
		//now make filename unique (if necessary) and open
		fsFilename = ::UniqueFileNameGenerator(fsFilename);
		out.open(fsFilename.c_str(), ios::out | ios::binary);
	}
	if (!out)
	{
		exc = CFException(CFException::FILE_OPEN_FAILED,
			__LINE__, __FILE__, fsFilename, "writing");
		throw exc;
	}

	unsigned int nFileSizeInBytes = 0;
	CFException::tException excType =
		CFException::EXCEPTION_INVALID;
	vector<CFBlockAddress> linkTracker; //to avoid circular reads
	do
	{
		//check if this link has already been traversed
		if (find(linkTracker.begin(), linkTracker.end(), link) !=
			linkTracker.end())
		{
			excType = CFException::DISK_IMAGE_HAS_CIRCULAR_LINKAGE;
		}
		else
		{
			//this const cast is necessary because we get a reference to
			//a member variable which could be modified later. This would
			//violate the const character of this member function
			CFDiskBlock& actualBlock = (*const_cast<CFDiskImage *>(this))[link];

			 //we are processing actual block (remember that)
			linkTracker.push_back(link);
			link = actualBlock.ReadSectorLink(); //get next block

			unsigned int nActualBlockSize;
			 //transfer the data to the output file
			if ((nActualBlockSize = actualBlock.PutRealDataOnlyToStream(out))
				== 0)
			{
				//access has failed this means we have a defect link
				//physical access failure would have lead to exception 
				if (link.IsLastBlock())
				{
					//last block has to contain data but not more than 254 bytes
					int length = link.m_sector - 1;
					if ((length <= 0) ||
						(length > BLOCKSIZE - LINKSIZE))
					{
						excType = CFException::DISK_FILE_LAST_BLOCK_SIZE_ERROR;
					}
				}
				else
				{
					if (!link.IsValidLink())
					{
						excType = CFException::DISK_FILE_LINK_ERROR;
					}
				}
			}
			nFileSizeInBytes += nActualBlockSize;
		}
	} while (!link.IsLastBlock()
		&& (excType == CFException::EXCEPTION_INVALID));

	if (!out)
	{
		exc = CFException(CFException::FILE_OPERATION_FAILED,
			__LINE__, __FILE__, fsFilename, "writing");
		throw exc;
	}

	if (excType != CFException::EXCEPTION_INVALID)
	{
		//those exceptions are only warnings - dont throw
		exc = CFException(excType, __LINE__, __FILE__
			, m_filename, entry.GetFilename());
		exc.WriteOutExceptionWithWarningHeader(cerr);
	}
	//everything worked well
	//set the info to put the extracted file into a disk image
	extractInfo.SetFileType(entry.GetFiletype());
	extractInfo.SetCbmFilename(entry.GetFilename());
	extractInfo.SetFilename(fsFilename);
	extractInfo.SetFileSize(nFileSizeInBytes);
}


bool CFDiskImage::ExtractAllFiles(const string& dirname)
{
	bool bRet = false; //indicates all archives extracted

	//iterate over all files within directory
	for (int i = 0; i < m_directory.size(); i++)
	{
		//extract suitable files
		if ((m_directory[i].GetFiletype() == CFDiskLayout::PRG) ||
			(m_directory[i].GetFiletype() == CFDiskLayout::SEQ))
		{
			try {
			CFExtractedFileInfo fileInfo;
			ExtractFile(m_directory[i], dirname, fileInfo);
			//if extraction worked out fine remeber all the info about
			//the extracted files to process them later
			m_extractedFiles.push_back(fileInfo);
			}
			catch (CFException& actExc)
			{
				//handle missing directory not as warning
				if ((actExc.GetExceptionId() == CFException::UNEXPECTED_FILE_TYPE)
					|| (actExc.GetExceptionId() == CFException::FILE_STAT_FAILED))
					throw;
				actExc.WriteOutExceptionWithWarningHeader(cerr);
			}
		}
		else
		{
			//at least one archive could not be extracted because of type
			//potentially pass this disk to emulator
			bRet = true;
		}
	}
	return bRet;
}


void CFDiskImage::InsertNewDirectoryEntryWithBlockAlloc(const CFDirectoryEntry& entry)
{
	CFException::tException excType =
		CFException::EXCEPTION_INVALID;
	string param2;
	vector<CFBlockAddress> linkTracker; //to avoid circular reads

	//point to the first directory block
	CFBlockAddress link = m_geom.GetFirstDirBlockAddr();

	do
	{
		//reference - we write directly into diskimage
		CFDiskBlock& actualBlock = (*this)[link]; 

		//lay the logical structure over the raw content
		CFDiskLayout::tDirEntry *aEntry
			= reinterpret_cast<CFDiskLayout::tDirEntry *>(actualBlock.GetBlockContentPointer());
		ASSERT((BLOCKSIZE % sizeof(CFDiskLayout::tDirEntry)) == 0);
		//loop over complete block
		for (int i = 0; i < BLOCKSIZE / sizeof(CFDiskLayout::tDirEntry); i++)
		{
			if ((aEntry[i].filetype == CFDiskLayout::DEL) &&
				(aEntry[i].locked == 0))
			{
				//we have found a free slot - use it!
				link = actualBlock.ReadSectorLink();
				aEntry[i] = (CFDiskLayout::tDirEntry)entry;
				//directory entry with index 0 overwrites link - restore it!
				actualBlock.WriteSectorLink(link);
				//insert this entry into parsed directory
				m_directory.push_back(entry);
				return;
			}
		}

		linkTracker.push_back(link); //we processed actual block (remember that)
		link = actualBlock.ReadSectorLink(); //get next block

		//check if we have to allocate a new directory block
		//1. if we have encountered the last block
		//2. and 3. if we link outside directory (repair)
		//4. if we have encountered an already processed block (repair
		//circle linkage).
		if (link.IsLastBlock()
			|| !link.IsValidLink()
			|| !link.IsDirectoryLink()
			|| (find(linkTracker.begin(), linkTracker.end(), link) !=
				linkTracker.end()))
		{
			CFBlockAddress newBlockAddr = m_control.FindAndAllocFreeDirBlock();
			if (newBlockAddr == OUT_OF_BLOCKS)
			{
				excType = CFException::DISK_DIRECTORY_FULL_ERROR;
			}
			else
			{
				//link in the new directory block
				actualBlock.WriteSectorLink(newBlockAddr);
				//write the new entry
				CFDiskLayout::tDirEntry *aEntry
				= reinterpret_cast<CFDiskLayout::tDirEntry *>((*this)[newBlockAddr].GetBlockContentPointer());
				aEntry[0] = (CFDiskLayout::tDirEntry)entry;
				//mark the new block as last block of directory
				(*this)[newBlockAddr]
					.WriteSectorLink(CFBlockAddress(0, m_geom.GetLastDirBlockLength()));
				//insert this entry into parsed directory
				m_directory.push_back(entry);
				return;
			}
		}
	} while (excType == CFException::EXCEPTION_INVALID);

	//exception occured due to loop termination
	exc = CFException(excType, __LINE__, __FILE__
		,m_filename,  param2);
	throw exc;
}


static inline bool CheckEndOfFileToDiskWrite(istream &in)
{
	//the following sequence determines if last block
	//has been written. If exactly BLOCKSIZE - LINKSIZE bytes
	//were written we may not have detected the fileend.
	//Peek to check.
	bool end = in.eof();
	if (!end)
	{
		char a = in.get();
		end = in.eof();
		if (!end)
		{
			in.putback(a);
		}
	}
	return end;
}


void CFDiskImage::InsertFile(const string& filename
	, const CFFileInfoCBM& fileInfoCBM) throw (CFException)
{
	//parameter 2 has a more complete description of the disk entry
	//if this is set use it. Otherwise use only parameter1
	//(filename on this machine) to determine everything about the disk entry
	bool bTapeFile = !fileInfoCBM.TestInitialValues();

	struct stat buf;

	if (stat(filename.c_str(), &buf) != 0)
	{
		exc = CFException(CFException::FILE_STAT_FAILED,
			__LINE__, __FILE__, filename, "", errno);
		throw exc;
	}
	//only regular files are to be processed
	if (!S_ISREG(buf.st_mode))
	{
		exc = CFException(CFException::UNEXPECTED_FILE_TYPE,
			__LINE__, __FILE__, filename
			, CFException::IntToString(buf.st_mode));
		throw exc;
	}

	unsigned int byteSize = buf.st_size;

	if (byteSize == 0)
	{
		exc = CFException(CFException::FILE_ZERO_LENGTH_ERROR,
			__LINE__, __FILE__, filename, "");
		throw exc;
	}

	//calculate exact number of blocks needed by the file
	int nBlocks = (byteSize + BLOCKSIZE - LINKSIZE - 1) / (BLOCKSIZE - LINKSIZE);
	if (nBlocks > m_control.GetFreeBlockCountDisk())
	{
		exc = CFException(CFException::FILE_DOES_NOT_FIT_ON_DISK,
			__LINE__, __FILE__, m_filename, filename);
		throw exc;
	}

	ifstream in(filename.c_str(), ios::in | ios::binary);
	if (!in)
	{
		exc = CFException(CFException::FILE_OPEN_FAILED,
			__LINE__, __FILE__, filename, "reading");
		throw exc;
	}

	//now file will fit on disk for sure - insert directory entry
	//search first data block for file - it has to be written into dir
	CFBlockAddress firstAddr = m_control
		.FindWithoutAllocFirstFreeFileBlock();
	//this should not happen because of previous checkings
	ASSERT(!(firstAddr == OUT_OF_BLOCKS));
	if (firstAddr == OUT_OF_BLOCKS)
	{
		exc = CFException(CFException::FILE_DOES_NOT_FIT_ON_DISK,
			__LINE__, __FILE__, m_filename, filename);
		throw exc;
	}
	if (bTapeFile)
	{
		//Build up a directory entry that relates to tape file
		CFDiskLayout::tDirEntry newDirEntry;
		memset(&newDirEntry, 0, sizeof(newDirEntry));
		//this byte is normally not available via notused1 component
		//I have been to lazy to introduce a union. BIG hack!
		newDirEntry.notused1[2] = fileInfoCBM.GetFileType();
		newDirEntry.lengthLow = nBlocks % 256;
		newDirEntry.lengthHigh = nBlocks / 256;
		newDirEntry.dataTrack = firstAddr.m_track;
		newDirEntry.dataSector = firstAddr.m_sector;
		strncpy(newDirEntry.name, fileInfoCBM.GetCbmFilename().c_str()
			, sizeof(newDirEntry.name));
		CFDirectoryEntry newEntry(newDirEntry);
		//insert into directory. This may throw exception.
		InsertNewDirectoryEntryWithBlockAlloc(newEntry);
	}
	else
	{
		CFCbmFilename cbmFilename;
		cbmFilename.ConvertThisFromFilesystemFilename(filename);
		CFDirectoryEntry newEntry(cbmFilename, nBlocks, firstAddr
			, CFDiskLayout::PRG, false, true);
		//insert into directory. This may throw exception.
		InsertNewDirectoryEntryWithBlockAlloc(newEntry);
	}
	//now that entry is written into dir alloc the first data block of file
	m_control.AllocBlock(firstAddr);

	CFBlockAddress addr = firstAddr; //iterator

	bool end = false; //termination flag
	unsigned int written;

	written = (*this)[addr].GetRealDataOnlyFromStream(in);
	ASSERT((written > 0) && (written <= BLOCKSIZE - LINKSIZE));

	while (!::CheckEndOfFileToDiskWrite(in))
	{
		ASSERT(written == BLOCKSIZE - LINKSIZE);
		//more blocks to follow
		CFBlockAddress next =
			m_control.FindAndAllocFirstFreeFileBlock();
		//this should not happen because of previous checkings
		ASSERT(!(next == OUT_OF_BLOCKS));
		//for release version we throw an exception
		if (next == OUT_OF_BLOCKS)
		{
			exc = CFException(CFException::FILE_DOES_NOT_FIT_ON_DISK,
				__LINE__, __FILE__, m_filename, filename);
			throw exc;
		}
		(*this)[addr].WriteSectorLink(next);
		addr = next;
		written = (*this)[addr].GetRealDataOnlyFromStream(in);
		ASSERT((written > 0) && (written <= BLOCKSIZE - LINKSIZE));
	}
	//stamp last block
	(*this)[addr].WriteSectorLink(CFBlockAddress(0, written + 1));

	if (in.bad())
	{
		exc = CFException(CFException::FILE_OPERATION_FAILED,
			__LINE__, __FILE__, filename, "reading");
		throw exc;
	}
}


void CFDiskImage::WriteDiskMemoryToFilesystem(void) const throw (CFException)
{
	m_control.WriteToMemoryDisk(); //first of all commit the control block
	if (!m_filename.empty() && (m_filename != MEMORY_DISKIMAGE_NAME))
	{
		//we will first write into a temporary file and then move to original
		string filename = m_filename + ".tmp";

		ofstream out(filename.c_str(), ios::out | ios::binary);
		if (!out)
		{
			exc = CFException(CFException::FILE_OPEN_FAILED,
				__LINE__, __FILE__, filename, "writing");
			throw exc;
		}

		//write the whole disk to temporary file
		for (CFBlockAddress sectorIterator; !(sectorIterator == OUT_OF_BLOCKS); ++sectorIterator)
		{
			//this const cast is necessary because we get a reference to
			//a member variable which could be modified later. This would
			//violate the const character of this member function
#ifdef _DEBUG
			unsigned int nRead =
#endif
			(*const_cast<CFDiskImage *>(this))[sectorIterator]
				.PutHeaderAndDataToStream(out);
			ASSERT(nRead == BLOCKSIZE);
		}

		if (!out)
		{
			exc = CFException(CFException::FILE_OPERATION_FAILED,
				__LINE__, __FILE__, filename, "writing");
			throw exc;
		}
		out.close(); //close before renaming

#ifdef _MSC_VER
		//only writable files can be removed for M$
		::chmod(m_filename.c_str(), _S_IWRITE);
		//rename does not overwrite old files for M$
		::remove(m_filename.c_str());
#endif
		//move temporary file to original
		if (::rename(filename.c_str(), m_filename.c_str()))
		{
			exc = CFException(CFException::FILE_RENAME_FAILED,
				__LINE__, __FILE__, filename, m_filename, errno);
			throw exc;
		}
	}
}


//-------------------------------------------------------------------
//CFDiskImage::CFControlBlock
//-------------------------------------------------------------------


CFDiskImage::CFControlBlock::CFControlBlock(
		const CFCbmFilename& diskname
		, const char formatId[2])
{
	//calc pointer to enclosing class
	//THIS IS UGLY CODE.
	//I'm sick and tired of passing the pointer to enclosing
	//class in another way
	static CFDiskImage a((char *)NULL); //use constructor that does not do allocations
	unsigned int offset = (char *)&(a.m_control) -
		(char *)&a;
	m_pEnclosing = (CFDiskImage *)((char *)this - (char*)offset);
	//give style to new control block
	InitVirginControlBlock(diskname, formatId);
}


void CFDiskImage::CFControlBlock::InitVirginControlBlock(
	const CFCbmFilename& diskname, const char diskId[2])
{
	int i;

	m_controlBlock.linkFirstDirBlockTrack = 
		m_geom.GetFirstDirBlockAddr().m_track;
	m_controlBlock.linkFirstDirBlockSector =
		m_geom.GetFirstDirBlockAddr().m_sector;
	m_controlBlock.formatID = 0x41;
	m_controlBlock.unused1 = 0;

	//calculate a virgin BAM
	static unsigned char freeMap[] = { 0xff, 0x7f, 0x3f, 0x1f, 0x0f, 0x07, 0x03, 0x01, 0x00, 0x74, 0x64, 0x36, 0x34, 0x6b, 0x68, 0x6c };
	for (int t = 0; t < m_geom.GetTracks(); t++)
	{
		int nFree = m_controlBlock.Bam[t].numberFreeBlocks = m_geom[t + 1];
		for (i = 0; i < sizeof(m_controlBlock.Bam[0].trackBam); i++)
		{
			//check if there are nonexisting sectors in this part of track
			int tooMuch = ((i + 1) * 8) - nFree;
			if (tooMuch < 0)
			{
				//no nonexisting sectors
				tooMuch = 0;
			}
			if (tooMuch > 7)
			{
				//whole byte has nonexisting sectors
				tooMuch = 8;
			}
			m_controlBlock.Bam[t].trackBam[i] = freeMap[tooMuch];
		}
	}
	strncpy(m_controlBlock.diskName, diskname.c_str(), sizeof(m_controlBlock.diskName));
	m_controlBlock.delimiter1[0] = m_geom.GetFileTermChar();
	m_controlBlock.delimiter1[1] = m_geom.GetFileTermChar();
	m_controlBlock.diskId[0] = diskId[0];
	m_controlBlock.diskId[1] = diskId[1];
	m_controlBlock.delimiter2 = 0xa0;
	m_controlBlock.format[0] = 0x32;
	m_controlBlock.format[1] = 0x41;
	for (i = 0; i < sizeof(m_controlBlock.delimiter3); i++)
		m_controlBlock.delimiter3[i] = m_geom.GetFileTermChar();
	for (i = 0; i < sizeof(m_controlBlock.unused2); i++)
		m_controlBlock.unused2[i] = 0;
	//allocate the only two blocks always allocated after format
	AllocBlock(m_geom.GetControlBlockAddr());
	AllocBlock(m_geom.GetFirstDirBlockAddr());
}


void CFDiskImage::CFControlBlock::AttachControlBlock(void) throw (CFException)
{
	ASSERT(sizeof(CFDiskLayout::tControlBlock) == BLOCKSIZE);
	ASSERT(m_pEnclosing);

	CFBlockAddress controlBlockAddr = m_geom.GetControlBlockAddr();

	//get pointer to the content of the control block
	CFDiskLayout::tControlBlock const *pControl =
		reinterpret_cast<CFDiskLayout::tControlBlock *>
		((*m_pEnclosing)[controlBlockAddr].GetBlockContentPointer());

	//copy the control block from diskimage to local copy
	m_controlBlock = *pControl;

	//now a punch of consistency checks and error corrections
	CFException::tException excType =
		CFException::EXCEPTION_INVALID;
	string param2;

	CFBlockAddress firstDirLink(m_controlBlock.linkFirstDirBlockTrack
		, m_controlBlock.linkFirstDirBlockTrack);
	if (!firstDirLink.IsDirectoryLink())
	{
		excType = CFException::DISK_CONTROL_BLOCK_DIRECTORY_LINK_ERROR;
		param2 = (string)"track: " + CFException::IntToString(firstDirLink.m_track)
			+ " sector: " + CFException::IntToString(firstDirLink.m_sector);					
	}

	//check if control block allocated
	if (!TestBlockAllocation(controlBlockAddr))
	{
		//Alloc it - that must be
		AllocBlock(controlBlockAddr);
		excType = CFException::DISK_CONTROL_BLOCK_ALLOC_ERROR;
		param2 = (string)"track: " 
			+ CFException::IntToString(controlBlockAddr.m_track)
			+ " sector: " 
			+ CFException::IntToString(controlBlockAddr.m_sector);					
	}

	//check BAM and correct if necessary
	//we don't use the private Test/Alloc/Free functions because of optimization

	//Loop over tracks. Attention: counts from 0 which is unusual.
	for (int t = 0; t < m_geom.GetTracks(); t++)
	{
		unsigned int index = 0; //index in trackBam
		unsigned int bit = 1; //sector in actual trackBam Byte
		unsigned int freeBlocks = m_geom[t + 1];
		for (int s = 0; s < m_geom[t + 1]; s++)
		{
			if (!(m_controlBlock.Bam[t].trackBam[index] & bit))
			{
				//found allocated block
				freeBlocks--;
			}
			bit *= 2;
			if (bit == 256)
			{
				index++;
				bit = 1;
			}
		}
		if (m_controlBlock.Bam[t].numberFreeBlocks != freeBlocks)
		{
			m_controlBlock.Bam[t].numberFreeBlocks = freeBlocks;
			excType = CFException::DISK_BAM_FREE_COUNT_ERROR;
			param2 = (string)"track: " + CFException::IntToString(t + 1);					
		}
	}
	if (excType != CFException::EXCEPTION_INVALID)
	{
		exc = CFException(excType, __LINE__, __FILE__
			,m_pEnclosing->GetFilename(), param2);
		throw exc;
	}
}


bool CFDiskImage::CFControlBlock::TestBlockAllocation(const CFBlockAddress& addr) const
{
	ASSERT(addr.IsValidLink());

	unsigned int sector = addr.m_sector;
	//OK what the heck. We have three bytes which represent bitwise sector 0..7,
	//8..15, 16..23 - so calc byte and bit within byte. Attention: 0 means allocated
	//we have an index problem with the track. within BAM we need track - 1
	//for numer of sectors we need track (counted from 1 on).
	return (!(m_controlBlock.Bam[addr.m_track - 1].trackBam[sector / 8]
		& (1 << (sector % 8))));
}


void CFDiskImage::CFControlBlock::AllocBlock(const CFBlockAddress& addr)
{
	ASSERT(addr.IsValidLink());
	ASSERT(!TestBlockAllocation(addr)); //still free

	unsigned int sector = addr.m_sector;
	//see TestBlockAllocation for access description. We have to RESET the bit
	m_controlBlock.Bam[addr.m_track - 1].trackBam[sector / 8] &= ~(1 << (sector % 8));
	m_controlBlock.Bam[addr.m_track - 1].numberFreeBlocks--;
}


void CFDiskImage::CFControlBlock::FreeBlock(const CFBlockAddress& addr)
{
	ASSERT(addr.IsValidLink());
	ASSERT(TestBlockAllocation(addr)); //still allocated

	unsigned int sector = addr.m_sector;
	//see TestBlockAllocation for access description. We have to SET the bit.
	m_controlBlock.Bam[addr.m_track - 1].trackBam[sector / 8] |= (1 << (sector % 8));
	m_controlBlock.Bam[addr.m_track - 1].numberFreeBlocks++;
}


CFBlockAddress CFDiskImage::CFControlBlock::FindAndAllocFreeDirBlock(void)
{
	CFBlockAddress sectorIterator(m_geom.GetFirstDirBlockAddr());
	sectorIterator.m_sector = 0;

	//loop over each sector of directory track
	for (int i = 0; i < m_geom[sectorIterator.m_track]; i++)
	{
		//control block does not count as directory block
		if (!(sectorIterator == m_geom.GetControlBlockAddr()))
		{
			if (!TestBlockAllocation(sectorIterator))
			{
				//found free directory block
				AllocBlock(sectorIterator);
				return sectorIterator;
			}
		}
		++sectorIterator;
	}
	return OUT_OF_BLOCKS;
}


unsigned int CFDiskImage::CFControlBlock::GetFreeBlockCountDisk(void) const
{
	unsigned int count = 0;
	for (int t = 0; t < m_geom.GetTracks(); t++)
	{
		//do not consider directory block
		if (t + 1 != m_geom.GetFirstDirBlockAddr().m_track)
		{
			//check if free block count is plausible for the track
			ASSERT(m_controlBlock.Bam[t].numberFreeBlocks
				<= m_geom[t + 1]);
			count += m_controlBlock.Bam[t].numberFreeBlocks;
		}
	}
	return count;
}


CFBlockAddress CFDiskImage::CFControlBlock::FindWithoutAllocFirstFreeFileBlock(void) const
{
	CFBlockAddress freeSector;

	for (int t = 0; t < m_geom.GetTracks(); t++)
	{
		//do not consider directory block
		if (t + 1 != m_geom.GetFirstDirBlockAddr().m_track)
		{
			//check if free block count is plausible for the track
			ASSERT(m_controlBlock.Bam[t].numberFreeBlocks
				<= m_geom[t + 1]);
			//if free block found take it
			if (m_controlBlock.Bam[t].numberFreeBlocks > 0)
			{
				freeSector.m_track = t + 1;
				for (int s = 0; s < m_geom[t + 1]; s++)
				{
					freeSector.m_sector = s;
					if (!TestBlockAllocation(freeSector))
					{
						return freeSector;
					}
				}
				//if free block indicated it must be found
				ASSERT(false);
			}
		}
	}

	return OUT_OF_BLOCKS;
}


CFBlockAddress CFDiskImage::CFControlBlock::FindAndAllocFirstFreeFileBlock(void)
{
	CFBlockAddress blockAddr = FindWithoutAllocFirstFreeFileBlock();
	if (!(blockAddr == OUT_OF_BLOCKS))
	{
		//free block found - alloc it
		AllocBlock(blockAddr);
	}
	return blockAddr;
}


void CFDiskImage::CFControlBlock::WriteToMemoryDisk(void) const
{
	ASSERT(m_pEnclosing);
	ASSERT(sizeof(m_controlBlock) == BLOCKSIZE);

	//this writes the control block back into diskimage
	*(reinterpret_cast<CFDiskLayout::tControlBlock *>
		((*m_pEnclosing)[m_geom.GetControlBlockAddr()]
		.GetBlockContentPointer()))
	= m_controlBlock;
}

