//////////////////////////////////////////////////////////////////////
//TargetD64 - C64 archive related conversion tool and emulator frontend
//////////////////////////////////////////////////////////////////////
//Copyright (C) 1998, 1999  Karlheinz Langguth klangguth@netscape.net
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//////////////////////////////////////////////////////////////////////

#ifndef _DISK_IMAGE_MANAGER_HEADER
#define _DISK_IMAGE_MANAGER_HEADER

const unsigned int BLOCKSIZE = 256; //overall size of block
const unsigned int LINKSIZE = 2; //size of the sector link within block (track, sector)


//concept of addressing a block on disk
//this is done by (track, sector)
class CFBlockAddress
{
public:
	inline CFBlockAddress(void) : m_track(1), m_sector(0) {}
	//P1 I: track
	//P2 I: sector
	inline CFBlockAddress(const unsigned int track, const unsigned int sector)
		: m_track(track), m_sector(sector) {}

public:
	//move this block to next block (change to next track when track overflow)
	//R: next block or OUT_OF_BLOCKS when disk end reached
	CFBlockAddress operator++(void);
	//copmpare to block addresses
	//R: true if equal
	bool operator==(const CFBlockAddress& a) const;
	//check if out of disk reached
	//R: true if still within disk
	bool operator!() const;
	//check if last data block indicated
	//R: true indicates last data block
	inline bool IsLastBlock(void) const { return (m_track == 0); }
	//check if block resides within directory (NOT control block)
	//R: true indicates directory track
	bool IsDirectoryLink(void) const;
	//check if a valid block is addressed (NOT last data block)
	//R: addresses valid block (within disk geometry)
	bool IsValidLink(void) const;

public:
	unsigned int m_track;
	unsigned int m_sector;
};
const CFBlockAddress OUT_OF_BLOCKS(255, 255); //out of disk indicator


//concept of capsulating the disk geometry
//right now for 1541 disks only
class CFDiskLayout
{
public:
	//these are the filetypes for directory entries with the real values
	typedef enum { DEL = 0, SEQ = 1, PRG = 2, USR = 3, REL = 4, UNKNOWN }
		tFiletype;

//it seems that g++ has default alignment 1 byte (ASSERTS work)
#ifdef _MSC_VER
#pragma pack(push, l1)
#pragma pack(1)
#endif
	//memory layout for the BAM of a single track
	typedef struct
	{
		unsigned char numberFreeBlocks;
		unsigned char trackBam[3]; //Byte 1: sector 0-7 and so on. 0 means allocated.
	} tTrackBam;

	//memory layout of the control block
	typedef struct
	{
		unsigned char linkFirstDirBlockTrack;
		unsigned char linkFirstDirBlockSector; //links to first dir block
		unsigned char formatID; //should be 0x41 for 1541
		unsigned char unused1;
		tTrackBam Bam[35]; //BAM for the complete disk
		char diskName[16]; //this is the name passed while formating
		unsigned char delimiter1[2]; //should be both 0xa0
		unsigned char diskId[2]; //this is the disk ID passed while formating
		unsigned char delimiter2; //should be 0xa0
		unsigned char format[2]; //should be 0x32 0x41
		unsigned char delimiter3[4]; //should be 0xa0
		unsigned char unused2[85];
	} tControlBlock;

	//memory layout of a directory entry
	typedef struct
	{
		unsigned char notused1[2];
		unsigned char filetype : 3; //compare tFileType
		unsigned char notused2 : 3;
		unsigned char locked   : 1; //true means locked against deletion
		unsigned char closed   : 1; //true means has been properly closed
		unsigned char dataTrack; 
		unsigned char dataSector; //link to first data block
		char name[16];
		unsigned char notused3[7]; //usage only for relative files
		unsigned char newTrack;
		unsigned char newSector; //normally not used
		unsigned char lengthLow;
		unsigned char lengthHigh; //block length of entry
	} tDirEntry;
#ifdef _MSC_VER
#pragma pack(pop, l1)
#endif

public:
	//transfer sector count per track into private vector
	//initialize all vectors
	//parameters matched to members directly (look there)
	inline CFDiskLayout(const unsigned int tracks
		, const unsigned int sectorPerTrack[]
		, const CFBlockAddress& controlBlock
		, const CFBlockAddress& firstDirBlock
		, const char fileterm
		, const unsigned char lastDirBlockLength)
		: m_tracks(tracks), m_controlBlock(controlBlock)
		, m_firstDirBlock(firstDirBlock)
		, m_fileTermChar(fileterm)
		, m_lastDirBlockLength(lastDirBlockLength)
		{
			//move the sector information into the vector
			for (int i = 0; i < m_tracks; i++)
				m_sectorPerTrack.push_back(sectorPerTrack[i]);
		}

public:
	//R: number of tracks of disk
	inline unsigned int GetTracks(void) const { return m_tracks; }
	//R: termination char for directory name entry ($a0 for 1541)
	inline char GetFileTermChar(void) const { return m_fileTermChar; }
	//R: block address of control block containing BAM
	inline  const CFBlockAddress& GetControlBlockAddr(void) const { return m_controlBlock; }
	//R: length of the last directory block which should always be const
	inline unsigned char GetLastDirBlockLength(void) const { return m_lastDirBlockLength; }
	//R: block address of the first directory block
	inline const CFBlockAddress& GetFirstDirBlockAddr(void) const { return m_firstDirBlock; }
	//get the number of sectors of a particular track
	//P1 I: track index - attention: counting starts with 1 (not 0)
	//R: number of sectors for this track
	inline unsigned int operator[](const unsigned int track)
		{ return m_sectorPerTrack[track - 1]; }

private:
	unsigned int m_tracks; //number of tracks on disk
	//character that ends an filename within a directory entry name
	char m_fileTermChar;
	//length of the last directory block which should always be const
	unsigned char m_lastDirBlockLength;
	//sector per track table
	vector<unsigned int> m_sectorPerTrack;
	//block address of the control block containing BAM
	CFBlockAddress m_controlBlock;
	//block address of the first directory block
	CFBlockAddress m_firstDirBlock;
};


//concept of a directory entry within disk directory
class CFDirectoryEntry
{
public:
	CFDirectoryEntry(void);
	//construct from single values
	//member variables accordingly set to parameters
	CFDirectoryEntry(const string& filename, const unsigned int blockSize
		, const CFBlockAddress& firstBlock
		, const CFDiskLayout::tFiletype filetype
		, const bool locked = false, const bool closed = true);
	//construct from a physical directory entry from disk
	CFDirectoryEntry(const CFDiskLayout::tDirEntry& dirEntry);

public:
	//R: CBM filename
	inline const CFCbmFilename& GetFilename(void) const
		{ return m_filename; }
	//R: block size of entry
	inline unsigned int GetSize(void) const
		{ return m_size; }
	//R: filetype (PRG, SEQ, etc.)
	inline CFDiskLayout::tFiletype GetFiletype(void) const
		{ return m_filetype; }
	//R: block address of the first data block
	inline const CFBlockAddress& GetFirstFileDataBlockAddr(void) const
		{ return m_firstBlockAddr; }
	//conversion operator to prduce a physical layout of this logical entry
	//R: physical layout (which may directly be put into dir block)
	operator CFDiskLayout::tDirEntry&(void) const; 

private:
	bool m_locked; //true indicates locking against deletion
	bool m_closed; //true indicates properly closed file
	CFCbmFilename m_filename; //CBM filename (planar, size limited)
	unsigned short m_size; //blocksize
	CFDiskLayout::tFiletype m_filetype; //filetype (PRG, SEQ, etc.)
	CFBlockAddress m_firstBlockAddr; //first data block address of file
};


//concept of a disk block containing raw data
//structured into header (linkage) and real data
class CFDiskBlock
{
public:
	CFDiskBlock(void);

public:
	//read a byte of raw data from block
	//P1 I: byte index
	//R: reference to read character
	unsigned char& operator[](const unsigned int index);
	//R: pointer to the raw data
	inline unsigned char *GetBlockContentPointer(void) { return m_content; }
	//R: block linkage (header part of raw data)
	inline CFBlockAddress ReadSectorLink(void) const
		{ return CFBlockAddress(m_content[0], m_content[1]); }
	//overwrite the linkage (header part of raw data)
	//P1 I: block address to write
	inline void WriteSectorLink(const CFBlockAddress& link)
		{ m_content[0] = link.m_track; m_content[1] = link.m_sector; }
	//read data from P1 as long not EOF and still space in block
	//put data starting with beginning of raw data (index 0)
	//P1 IO: stream to read data from
	//R: number of bytes got
	unsigned int GetHeaderAndDataFromStream(istream& in) throw (CFException);
	//read data from P1 as long not EOF and still space in block
	//put data starting with beginning of real data (skip header = linkage)
	//P1 IO: stream to read data from
	//R: number of bytes got
	unsigned int GetRealDataOnlyFromStream(istream& in);
	//write raw data (header + real data) to P1
	//P1 IO: stream to write data to
	//R: number of bytes put - always BLOCKSIZE (else exception)
	unsigned int PutHeaderAndDataToStream(ostream& out) const throw (CFException);
	//write real data only to stream P1.
	//if block is a last data block (link track == 0) write only data
	//indicated else write BLOCKSIZE byte
	//P1 IO: stream to write data to
	//R: number of bytes put
	unsigned int PutRealDataOnlyToStream(ostream& out) const throw (CFException);

private:
	//raw data of block
	unsigned char m_content[BLOCKSIZE];
};


const string MEMORY_DISKIMAGE_NAME("Memory diskimage"); //indicates memory image
//concept of a disk image
//image may be attached to a file. after that all processing in memory
//until WriteDiskMemoryToFilesystem().
class CFDiskImage
{
//control block is part of image - make it inner class
class CFControlBlock;
//control block may access private members of disk image
friend class CFControlBlock;

public:
	//allocate memory for disk (alls disk blocks)
	//prepare directory
	//P1 I: host filesystem name of disk image
	//P2 I: disk name as passed for formatting
	//P3 I: disk ID as passed for formatting
	CFDiskImage(const string& filename = MEMORY_DISKIMAGE_NAME
		, const CFCbmFilename& diskname = CFCbmFilename()
		, const char diskId[2] = "64");
private:
	//just to calculate a few offsets in CFControlBlock::CFControlBlock
	//need constructor which does no alloc
	CFDiskImage(char *) {}

public:
	//extract all files (except unextractable liek REL) on disk into
	//directory P1
	//map the CBM filenames to host filenames and resolve collisions
	//by building unique filenames
	//keep track of all extracted files in m_extractedFiles
	//P1 I: directory to extract to
	//R: files remaining not extracted (e.g. REL files)
	bool ExtractAllFiles(const string& dirname);
	//R: vector with info about extracted files (needed for transfer into D64)
	inline const vector<CFExtractedFileInfo>& GetVectorOfExtractedFiles(void) const
		{ return m_extractedFiles; }
	//get a block of disk addressed by P1
	//P1 I: block address of block to get
	//R: reference to block addressed
	CFDiskBlock& operator[](const CFBlockAddress& block); 
	//R: host filename of disk image
	const string& GetFilename(void) const { return m_filename; }
	//attach a disk image existing in host filesystem
	//this means read all data in and check integrity
	//attach control block - correct it if necessary
	//parse directory
	//P1 I: disk image to attach within host filesystem
	void Attach(const string& filename) throw (CFException);
	//R: reference to parsed directory
	inline const vector<CFDirectoryEntry>& GetDirectory(void) const
		{ return m_directory; }
	//insert a file into disk image (in memory)
	//this includes writing of a directory entry slot on disk
	//always take the host filename from P1
	//if P2 not set:
	//get the file directly from host filesystem. map filename from host to CBM
	//else:
	//get the CBM filename info from P2 (no info loosing by conversion).
	//P1 I: filename in host filesystem
	//P2 I: info about file in CBM filesystem
	void InsertFile(const string& filename
		, const CFFileInfoCBM& tapeFileInfo = CFFileInfoCBM()) throw (CFException);
	//flush the memory image to host filesystem file
	//write first to a temporary file and rename afterwards (transaction)
	void WriteDiskMemoryToFilesystem(void) const throw (CFException); 
	//R: the free block count of the whole disk
	inline unsigned int GetFreeBlockCountDisk(void) const
	{ return m_control.GetFreeBlockCountDisk(); }
private:
	//parse the directory of the disk image
	void ParseAndCheckBlockAllocDirectory(void) throw (CFException);
	//extract a file from disk image into host filesystem
	//use P4 as filename if set else derive filename from CBM filename
	//P1 I: the directory entry describing the file
	//P2 I: directory to extract to
	//P3 O: additional file info from directory entry
	//P4 I: host filesystem filename ("" means not set)
	void ExtractFile(const CFDirectoryEntry& entry
		, const string& dirname
		, CFExtractedFileInfo& extractInfo
		, const string& filename = "") const throw (CFException);
	//search for a free directory entry slot on disk
	//if found write P1 into it. do alloc of dir block if needed.
	//add entry to parsed directory m_directory
	//P1 I: directory entry to be put into directory
	void InsertNewDirectoryEntryWithBlockAlloc(const CFDirectoryEntry& entry);

private:
	//fileinfo of all extracted files of this disk
	vector<CFExtractedFileInfo> m_extractedFiles;
	typedef vector<CFDiskBlock> tSectorsInTrack;
	vector<tSectorsInTrack> m_diskBlock; //all blocks of the disk
	vector<CFDirectoryEntry> m_directory; //logical representation of disk dir
	string m_filename; //host filesystem filename of diskimage

public:
	//disk geometry of disk
	static CFDiskLayout m_geom;

private:
	//concept of the control block which contains management info like BAM
	class CFControlBlock
	{
	public:
		//construct virgin control block
		//P1 I: disk name as passed for formatting
		//P2 I: disk ID as passed for formatting
		CFControlBlock(const CFCbmFilename& diskname = CFCbmFilename()
			, const char formatID[2] = "64");
	public:
		//attach control block from memory disk
		//check consistency - correct if necessary (especially free block count)
		void AttachControlBlock(void) throw (CFException);
		//test the allocation of a block P1
		//P1 I: addresses block to test
		//R: true if allocated
		bool TestBlockAllocation(const CFBlockAddress& addr) const;
		//allocate block P1 in BAM. must be free before calling this method.
		//P1 I: addresses block to test
		void AllocBlock(const CFBlockAddress& addr);
		//free block P1 in BAM. must be allocated before calling this method.
		//P1 I: addresses block to test
		void FreeBlock(const CFBlockAddress& addr);
		//R: the free block count of the whole disk
		unsigned int GetFreeBlockCountDisk(void) const;
		//search the directory track for a free block and alloc it if found
		//control block is not considered.
		//R: OUT_OF_BLOCKS if not found else address of found block
		CFBlockAddress FindAndAllocFreeDirBlock(void);
		//peek for a free block ouside directory track - do not alloc!
		//R: OUT_OF_BLOCKS if not found else address of found block
		CFBlockAddress FindWithoutAllocFirstFreeFileBlock(void) const;
		//find a free block outside directory track and alloc it if found
		//R: OUT_OF_BLOCKS if not found else address of found block
		CFBlockAddress FindAndAllocFirstFreeFileBlock(void);
		//write the control block back to memory image
		void WriteToMemoryDisk(void) const;
	private:
		//provide a virgin initialized control block
		//virgin BAM and control block first dir block allocated
		//P1 I: disk name as passed for formatting
		//P2 I: disk ID as passed for formatting
		void InitVirginControlBlock(const CFCbmFilename& diskname
			, const char diskId[2]);
	private:
		//pointer to enclosing class (calculated while construction!)
		CFDiskImage* m_pEnclosing;
		//raw data of control block
		CFDiskLayout::tControlBlock m_controlBlock;
	} m_control;
};

#endif
