//////////////////////////////////////////////////////////////////////
//TargetD64 - C64 archive related conversion tool and emulator frontend
//////////////////////////////////////////////////////////////////////
//Copyright (C) 1998, 1999  Karlheinz Langguth klangguth@netscape.net
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
//COMPILE SWITCHES
// _MSC_VER
// indicates MS compiler
// _DEBUG
// for debug version which activates ASSERT and TRACE
// TD64_MODIFIED
// marks changes in foreign sources to fit into TargetD64
// must be set everywhere because also used for header files

#include <string.h>
#include <string>
#ifdef _MSC_VER
#include <iostream>
#include <strstream>
#else
#include <iostream.h>
#include <strstream.h>
#endif

using namespace std;

#include "Exception.h"

CFException exc; //this is the global exception used for this project


//check integrity only once
bool CFException::m_bIntegrityChecked = false;
//all messages will be printed out by default
int CFException::ms_nMaxWarningLevel = 1;

//list of all exceptions that can happen
//id is used for integrity check at construction of first exception
CFException::tExceptionList CFException::excTable[] =
{
	EXCEPTION_INVALID
	,"Invalid Exception - never expected to show up."
	,FILE_OPEN_FAILED
	,"File open failed for file <param1> with access <param2>."
	,CHANGE_DIRECTORY_FAILED
	,"Change to directory <param1> failed."
	,GET_CURRENT_DIRECTORY_FAILED
	,"Getting of current directory failed."
	,OPEN_DIRECTORY_FAILED
	,"Open on directory <param1> failed."
	,FILE_STAT_FAILED
	,"Stat on file <param1> failed."
	,UNEXPECTED_FILE_TYPE
	,"Unexpected file type (e.g. directory where regular file expected) for file <param1> with stat_mode <param2>."
	,FILE_OPERATION_FAILED
	,"File operation failed for file <param1> with operation <param2>."
	,CLOSE_DIRECTORY_FAILED
	,"Close on directory <param1> failed."
	,SYSTEM_CALL_FAILED
	,"System call of command <param1> failed."
	,FILE_RENAME_FAILED
	,"Renaming of old file <param1> to new file <param2> failed."
	,INCOMPLETE_MULTI_PART_ARCHIVE
	,"Missing part in multipart archive <param1>."
	,UNIQUE_NAME_CREATION_FAILED
	,"Unique name creation for temporary pathname <param1> failed."
	,CREATE_DIRECTORY_FAILED
	,"Creation of directory <param1> failed."
	,FILE_REMOVAL_FAILED
	,"Removal of file <param1> failed."
	,FILE_TOO_BIG_FOR_D64
	,"File <param1> with blocksize <param2> too big for a D64 files (>664 blocks)."
	,MULTIPLE_PART_OF_MULTIPART_ARCHIVE
	,"A part of the multipart archive <param1> exists more than once."
#ifdef _MSC_VER
	,MEMORY_ALLOCATION_FAILED
	,"Memory allocation of <param1> bytes failed."
	,FILENAME_CONVERSION_LONG_SHORT_FAILED
	,"Could not convert long filename <param1> into it's short form."
#endif
	,DISK_IMAGE_SIZE_INCORRECT
	,"Diskimage with filename <param1> has wrong size as explained in <param2>."
	,DISK_IMAGE_HAS_CIRCULAR_LINKAGE
	,"The sector linkage of file <param2> in diskimage <param1> is cyclic."
	,DISK_IMAGE_DIRECTORY_CORRUPTED
	,"The directory of the disk image <param1> is corrupted. The reason is stated in <param2>."
	,DISK_IMAGE_FILETYPE_NOT_SUPPORTED
	,"An operation was requested upon file <param2> of diskimage <param2>. The file has an unsupported filetype."
	,DISK_FILE_LINK_ERROR
	,"The sector linkage of file <param2> in diskimage <param1> is corrupted."
	,UNIQUE_FILENAME_GENERATION_ERROR
	,"It was not possible to create a unique filename. Last trial before giving up has been <param1>."
	,DISK_FILE_LAST_BLOCK_SIZE_ERROR
	,"The last block of file <param2> in diskimage <param1> contains an impossible data length."
	,DISK_DIRECTORY_ALLOC_ERROR
	,"Directory block <param2> in diskimage <param1> is not allocated in BAM."
	,DISK_CONTROL_BLOCK_DIRECTORY_LINK_ERROR
	,"The control block of diskimage <param1> does not link to the first directory block."
	,DISK_CONTROL_BLOCK_ALLOC_ERROR
	,"The control block <param2> of diskimage <param1> is not allocted in BAM."
	,DISK_BAM_FREE_COUNT_ERROR
	,"The count of free blocks in the BAM of diskimage <param1> of track <param2> is wrong."
	,DISK_DIRECTORY_FULL_ERROR
	,"The directory of diskimage <param1> is full. Further file insertion is impossible."
	,FILE_ZERO_LENGTH_ERROR
	,"File <param1> has zero length. It can not be inserted into a diskimage."
	,FILE_DOES_NOT_FIT_ON_DISK
	,"File <param2> can not be inserted into diskimage <param1> because it is too large."
	,FILE_DUP_FAILED
	,"Duplication of file handle <param1> with system function dup failed."
	,FILE_DUP2_FAILED
	,"Copying of file handle <param1> to file handle <param2> with system function dup2 failed."
	,TAPE_IMAGE_TOO_SMALL
	,"The tapeimage <param1> is too small to contain tape and/or file records."
	,TAPE_IMAGE_NOT_ENOUGH_ENTRIES
	,"The number of directory entries in tapeimage <param1> is greater than the maximum number of entries."
	,TAPE_FILE_OFFSET_OUT_OF_BOUND_ERROR
	,"Offset of file <param2> in tapeimage <param1> points to header area."
	,TAPE_FILE_OFFSETS_ERROR
	,"Offset of the file succeeding file <param2> in tapeimage <param1> is not greater."
	,TAPE_FILE_CORRUPTED
	,"File <param2> in tapeimage <param1> is corrupted. Tapeimage too small to contain file."
	,TAPE_IMAGE_UNKNOWN_FILE_TYPE
	,"File <param2> in tapeimage <param1> has invalid file type."
	,ARCHIVE_EMPTY
	,"Archive <param1> does not contain data. Maybe it is empty. Maybe the extraction command failed unrecognized."
	,APPLICATION_ERROR
	,"Could not recover from previous errors - nothing done."
	,GENERIC_EXTENSION_MISSING_ERROR
	,"Found generic extraction command variable <param1> but generic extension variable <param2> is missing."
	,C64ZIP_EXTRACT_FAILED
	,"Extraction of c64 zipcoded file <param1> failed - invalid file format."
	,LYNX_EXTRACT_FAILED
	,"Extraction of lynx file <param1> failed - invalid file format."
	,LYNX_FILE_TOO_SHORT
	,"Lynx file <param1> is too short. Extracted file <param2> misses bytes."
	,GZIP_EXTRACT_FAILED
	,"Extraction of gnuzipped file <param1> failed. Reason stated in <param2>."
	,PKZIP_EXTRACT_FAILED
	,"Extraction of pkzipped file <param1> failed - invalid file format."
	,LHA_EXTRACT_FAILED
	,"Extraction of LHA file <param1> failed. Reason stated in <param2>."
	,EMULATOR_CALL_FAILED
	,"The emulator call with command <param1> failed with return code <param2>. Maybe your emulator throws undefined exit codes. Maybe the emulator executable is not on your execution path. Maybe the emulator is configured wrong."
};


CFException::CFException(
	const tException nException
	,const int nLineno
	,const string filename
	,const string param1, string param2
	,const int nSystemError)
	: m_nException(nException)
	,m_nLinenumber(nLineno)
	,m_filename(filename)
	,m_param1(param1)
	,m_param2(param2)
	,m_nSystemError(nSystemError)
	,m_nWarningLevel(1)
{
	if (!m_bIntegrityChecked)
	{
		//check the consistency of Exc.-ID and Exc.-Message
		for (int i = 0; i < sizeof(excTable) / sizeof(tExceptionList); i++)
		{
			if ((int)excTable[i].id != i)
			{
				cerr << "Exception table corrupted" <<endl;
				cerr << "Report this error to the programmer" <<endl;
				cerr << "Program terminated!" <<endl;
				exit(1);
			}
		}
		m_bIntegrityChecked = true;
	}
}


ostream& operator<<(ostream& ostr, const CFException& exc)
{
	ostr << "Exception ID    : " << exc.m_nException <<endl;
	ostr << "Exception text  : " << CFException::excTable[exc.m_nException].message <<endl;
	if (exc.m_param1 != INVALID)
		ostr << "Exception param1: " << exc.m_param1 <<endl;
	if (exc.m_param2 != INVALID)
		ostr << "Exception param2: " << exc.m_param2 <<endl;
	if (exc.m_nLinenumber >= 0)
	{
		ostr << "Location source : " << exc.m_filename << endl;
		ostr << "Location lineno : " << exc.m_nLinenumber << endl;
	}
	if (exc.m_nSystemError >= 0)
	{
		ostr << "Operating system gives additional details" <<endl;
		ostr << "OS Error number: " << exc.m_nSystemError <<endl;
		ostr << "OS Error text:   " << ::strerror(exc.m_nSystemError) <<endl;	
	}
	return ostr;
}


void CFException::WriteOutExceptionWithWarningHeader(ostream &out) const
{
	if (m_nWarningLevel <= ms_nMaxWarningLevel)
	{
		out << endl
			<< "----------------------------------------------------" <<endl
			<< "------Exception occured - processing continues------" <<endl
			<< "----------------------------------------------------" <<endl
			<< *this << endl;
	}
}


void CFException::WriteOutExceptionWithErrorHeader(ostream &out) const
{
	if (m_nWarningLevel <= ms_nMaxWarningLevel)
	{
		out << endl
			<< "----------------------------------------------------" <<endl
			<< "-------Exception occured - program terminated-------" <<endl
			<< "----------------------------------------------------" <<endl
			<< *this << endl;
	}
}
