//////////////////////////////////////////////////////////////////////
//TargetD64 - C64 archive related conversion tool and emulator frontend
//////////////////////////////////////////////////////////////////////
//Copyright (C) 1998, 1999  Karlheinz Langguth klangguth@netscape.net
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
//COMPILE SWITCHES
// _MSC_VER
// indicates MS compiler
// _DEBUG
// for debug version which activates ASSERT and TRACE
// TD64_MODIFIED
// marks changes in foreign sources to fit into TargetD64
// must be set everywhere because also used for header files

#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <string>
#ifdef _MSC_VER
#include <typeinfo.h>
#include <iostream>
#include <fstream>
#include <strstream>
#else
#include <typeinfo>
#include <iostream.h>
#include <fstream.h>
#include <strstream.h>
#endif

using namespace std;

#include "Exception.h"
#include "Image.h"
#include "Tracing.h"


//a helper function to split a pathname into components
void SplitPathname(const string& pathName
	, string& dirname, string& filename, string& extension) throw (CFException)
{
	extension.resize(0);
	filename.resize(0);
	dirname.resize(0);

#ifdef _MSC_VER

	char *a = static_cast<char *>(calloc(pathName.size() + 1, 1));
	if (a == NULL)
	{
		exc = CFException(CFException::MEMORY_ALLOCATION_FAILED,
			__LINE__, __FILE__, CFException::IntToString(pathName.size() + 1), "");
		throw exc;
	}
	char *b = static_cast<char *>(calloc(pathName.size() + 1, 1));
	if (b == NULL)
	{
		free(a);
		exc = CFException(CFException::MEMORY_ALLOCATION_FAILED,
			__LINE__, __FILE__, CFException::IntToString(pathName.size() + 1), "");
		throw exc;
	}
	char *c = static_cast<char *>(calloc(pathName.size() + 1, 1));
	if (c == NULL)
	{
		free(a);
		free(b);
		exc = CFException(CFException::MEMORY_ALLOCATION_FAILED,
			__LINE__, __FILE__, CFException::IntToString(pathName.size() + 1), "");
		throw exc;
	}
	char *d = static_cast<char *>(calloc(pathName.size() + 1, 1));
	if (d == NULL)
	{
		free(a);
		free(b);
		free(c);
		exc = CFException(CFException::MEMORY_ALLOCATION_FAILED,
			__LINE__, __FILE__, CFException::IntToString(pathName.size() + 1), "");
		throw exc;
	}

	_splitpath(pathName.c_str(), a, b, c, d);

	dirname = strcat(a, b); //concat drive and dirname
	filename = c;
	extension = d;

	free(a);
	free(b);
	free(c);
	free(d);

#else

	int posSlash = pathName.find_last_of("/");
	int posDot =  pathName.find_last_of(".");  
	ASSERT((int)string::npos < 0) //asure that following comparisons work
	if (posDot > posSlash)
	{
		extension = pathName.substr(posDot, pathName.size() - posDot); 
	}
	else
		posDot = pathName.size(); //prepare extraction of filename

	if (posSlash != string ::npos)
	{
		dirname = pathName.substr(0, posSlash + 1);
	}
	filename = pathName.substr(posSlash + 1, posDot - posSlash - 1);

#endif
}


string UniqueFileNameGenerator(const string& filename) throw (CFException)
{
	ASSERT(!filename.empty());

	struct stat buf;

	//check if original filename is valid
	if (stat(filename.c_str(), &buf) != 0) 
	{
		if (errno == ENOENT)
		{
			return filename;
		}
		else
		{
			exc = CFException(CFException::FILE_STAT_FAILED,
				__LINE__, __FILE__, filename, "", errno);
			throw exc;
		}
	}

	string a, b, c;
	::SplitPathname(filename, a, b, c);

	int trial = 200;
	string uniqueAppendix("0");
	//change filename to be unique
	while (trial-- > 0)
	{
		//already there. Try to make filename unique.
		string retFilename = a + b + uniqueAppendix + c;
		if (uniqueAppendix[uniqueAppendix.size() - 1] < '9')
		{
			uniqueAppendix[uniqueAppendix.size() - 1]++;
		}
		else
		{
			uniqueAppendix[uniqueAppendix.size() - 1] = '0';
			uniqueAppendix += '0';
		}
		//check if newly build filename is unique
		if (stat(retFilename.c_str(), &buf) != 0) 
		{
			if (errno == ENOENT)
			{
				return retFilename;
			}
			else
			{
				exc = CFException(CFException::FILE_STAT_FAILED,
					__LINE__, __FILE__, retFilename, "", errno);
				throw exc;
			}
		}
	}
	//if previous loop is left normally no filename has been found
	//tried hard enough - give up
	exc = CFException(CFException::UNIQUE_FILENAME_GENERATION_ERROR,
		__LINE__, __FILE__, filename, "");
	throw exc;
	//never reached - avoid compiler error M$
	return filename;
}


//-------------------------------------------------------------------
//CFCbmFilename
//-------------------------------------------------------------------


//this is the character mapping from a CBM filename
//to a filesystem filename
char CFCbmFilename::mapCbmToFilesys[256] =
{
  '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_'  //0-15
, '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_'  //16-31
#ifdef _MSC_VER //M$ does not allow < > " : in filenames (open fails)
, '_', '!', '_', '#', '$', '%', '_', '\'', '(', ')', '_', '+', ',', '-', '.', '_' //32-47
, '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '_', '_', '_', '=', '_', '_'  //48-63
#else
, '_', '!', '"', '#', '$', '%', '_', '\'', '(', ')', '_', '+', ',', '-', '.', '_' //32-47
, '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', '_', '<', '=', '>', '_'  //48-63
#endif
, '@', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o'  //64-79
, 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '[', '_', ']', '_', '_'  //80-95
, '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_'  //96-111 
, '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_'  //112-127
, '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_'  //128-143
, '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_'  //144-159
, '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_'  //160-175
, '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_'  //176-191
, '_', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o'  //192-207
, 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '_', '_', '_', '_', '_'  //208-223
, '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_'  //224-239
, '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_', '_'  //240-255
};


//this is the character mapping from a filesystem filename
//to a CBM filename
#define C(a) (a - 0x20) //map lower case letters to PETSCI
char CFCbmFilename::mapFilesysToCbm[256] =
{
  (char)0xa0, ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' '  //0-15
, ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' '  //16-31
, ' ', '!', '"', '#', '$', '%', ' ', '\'', '(', ')', ' ', '+', ',', '-', '.', ' ' //32-47
, '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ' ', '<', '=', '>', ' '  //48-63
, '@', C('a'), C('b'), C('c'), C('d'), C('e'), C('f'), C('g'), C('h'), C('i'), C('j'), C('k'), C('l'), C('m'), C('n'), C('o')  //64-79
, C('p'), C('q'), C('r'), C('s'), C('t'), C('u'), C('v'), C('w'), C('x'), C('y'), C('z'), '[', ' ', ']', ' ', ' '  //80-95
, ' ', C('a'), C('b'), C('c'), C('d'), C('e'), C('f'), C('g'), C('h'), C('i'), C('j'), C('k'), C('l'), C('m'), C('n'), C('o')  //96-111 
, C('p'), C('q'), C('r'), C('s'), C('t'), C('u'), C('v'), C('w'), C('x'), C('y'), C('z'), ' ', ' ', ' ', ' ', ' '  //112-127
, ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' '  //128-143
, ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' '  //144-159
, ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ' //160-175
, ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' '  //176-191
, ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ' //160-175
, ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' '  //176-191
, ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' '  //224-239
, ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' '  //240-255
};


//maximum size of a cbm filename
const unsigned int CFCbmFilename::m_nMaxSize = 16;
const char CFCbmFilename::m_termChar = (char)0xa0;

CFCbmFilename::CFCbmFilename(void)
{
	//empty filename string contains 0xa0
	resize(m_nMaxSize, m_termChar);
}


CFCbmFilename::CFCbmFilename(const string& cbmFilename)
	: string(cbmFilename)
{
	ASSERT(cbmFilename.size() <= m_nMaxSize);
	//filename always is appended by termination chars til end
	resize(m_nMaxSize, m_termChar);
}


void CFCbmFilename::ConvertThisFromFilesystemFilename(const string& filename)
{
	string a, b, c;
	::SplitPathname(filename, a, b, c);
	if ((!c.empty()) && (c.size() <= 4))
	{
		//extensions less than 4 may be important for conversion - keep them
		if (b.size() > m_nMaxSize - c.size())
		{
			b.resize(m_nMaxSize - c.size());
		}
	}
	*static_cast<string *>(this) = b + c;

	for (int i = 0; i < size(); i++)
	{
		//cast to unsigned char is very important - negative index without!!!
		(*this)[i] = mapFilesysToCbm[(unsigned char)(*this)[i]];
	}
	//if filename is less than maximum number of characters fill it
	//with the termination character
	resize(m_nMaxSize, m_termChar);
}


const string& CFCbmFilename::ConvertThisToFilesystemFilename(void) const
{
	static string retString;

	retString.resize(0);
	for (int i = 0; i < size(); i++)
	{
		if ((*this)[i] == m_termChar)
			break; //end of filename reached
		//cast to unsigned char is very important - negative index without!!!
		retString += mapCbmToFilesys[(unsigned char)(*this)[i]];
	}
	//CBM filename can be empty - transform this into _
	if (retString.empty())
	{
		retString = "_";
	}
	return retString;
}
