//////////////////////////////////////////////////////////////////////
//TargetD64 - C64 archive related conversion tool and emulator frontend
//////////////////////////////////////////////////////////////////////
//Copyright (C) 1998, 1999  Karlheinz Langguth klangguth@netscape.net
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
//COMPILE SWITCHES
// _MSC_VER
// indicates MS compiler
// _DEBUG
// for debug version which activates ASSERT and TRACE
// TD64_MODIFIED
// marks changes in foreign sources to fit into TargetD64
// must be set everywhere because also used for header files

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#ifdef _MSC_VER
#define MAXPATHLEN _MAX_PATH
#include <fcntl.h>
#include <io.h>
#else
#include <sys/param.h>
#endif

#include <string>
#ifdef _MSC_VER
#include <iostream>
#include <fstream>
#include <strstream>
#else
#include <iostream.h>
#include <fstream.h>
#include <strstream.h>
#endif

using namespace std;

#include "zlib.h"
#include "unzip.h"
#include "Tracing.h"
#include "Exception.h"
#include "Image.h"
#include "ZlibAdapter.h"

//-------------------------------------------------------------------
//Gunzip Functions
//-------------------------------------------------------------------


#define GUNZIP_BUFLEN      16384
void Gunzip(const string& gzipFilename, const string& extractDir) throw (CFException)
{
	ASSERT(extractDir[extractDir.size() - 1] != '/');
    static char *buf = new char[GUNZIP_BUFLEN];

    gzFile in;
    in = gzopen(gzipFilename.c_str(), "rb");
    if (in == NULL)
	{
		exc = CFException(CFException::FILE_OPEN_FAILED,
			__LINE__, __FILE__, gzipFilename, "reading");
		throw exc;
    }

	try {
	//set together the filename for the decompressed file
	string dirname, filename, extension;
	SplitPathname(gzipFilename, dirname, filename, extension);
	//consider extract directory and drop exception .gz
	string outName = extractDir + "/" + filename;
	//make filename unique
	outName = UniqueFileNameGenerator(outName);

	fstream out(outName.c_str(), ios::out | ios::binary);
	if (!out)
	{
		exc = CFException(CFException::FILE_OPEN_FAILED,
			__LINE__, __FILE__, outName, "write from scratch");
		throw exc;
	}	

    int len;
    while (true)
	{
        len = gzread(in, buf, sizeof(buf));
        if (len < 0)
		{
			int err;
			exc = CFException(CFException::GZIP_EXTRACT_FAILED,
				__LINE__, __FILE__, outName, gzerror(in, &err));
			throw exc;
		}
        if (len == 0)
			break;

        out.write(buf, len);
    }
	if (!out)
	{
		exc = CFException(CFException::FILE_OPERATION_FAILED,
			__LINE__, __FILE__, outName, "writing");
		throw exc;
	}
	} //END TRY
	catch (...)
	{
		//always close gzip file in case of error
		gzclose(in);
		throw;
	}

    gzclose(in);
}


//-------------------------------------------------------------------
//Unzip Functions
//-------------------------------------------------------------------


#define UNZIP_BUFLEN 8192
static void ZipExtractCurrentFile(unzFile zipFile, const string& extractDir) throw (CFException)
{
	ASSERT(extractDir[extractDir.size() - 1] != '/');
	static char *buf = new char[UNZIP_BUFLEN];
	
	unz_file_info fileInfo;
	char currentFilename[MAXPATHLEN];
    int err = UNZ_OK;
	err = unzGetCurrentFileInfo(zipFile, &fileInfo, 
		currentFilename, sizeof(currentFilename)
		,NULL ,0 ,NULL ,0);

	if (err != UNZ_OK)
	{
		exc = CFException(CFException::PKZIP_EXTRACT_FAILED,
			__LINE__, __FILE__, "", "");
		throw exc;
	}

	char *filenameWithoutDir, *p;
	p = filenameWithoutDir = currentFilename;
	while ((*p) != '\0')
	{
		if (((*p) == '/') || ((*p) == '\\'))
			filenameWithoutDir = p + 1;
		p++;
	}

	try {
	//do nothing for directories contained in zip
	if ((*filenameWithoutDir) != '\0')
	{
		//we have a file, not directory
		if (unzOpenCurrentFile(zipFile) != UNZ_OK)
		{
			exc = CFException(CFException::PKZIP_EXTRACT_FAILED,
				__LINE__, __FILE__, "", "");
			throw exc;
		}
		//set the complete filename together
		string outFilename = extractDir + "/" + filenameWithoutDir;
		//and make it unique
		outFilename = UniqueFileNameGenerator(outFilename);
		//and open it for output
		fstream out(outFilename.c_str(), ios::out | ios::binary);
		if (!out)
		{
			exc = CFException(CFException::FILE_OPEN_FAILED,
				__LINE__, __FILE__, outFilename, "write from scratch");
			throw exc;
		}	

		do
		{
			err = unzReadCurrentFile(zipFile, buf, UNZIP_BUFLEN);
			if (err < 0)	
			{
				exc = CFException(CFException::PKZIP_EXTRACT_FAILED,
					__LINE__, __FILE__, "", "");
				throw exc;
			}
			if (err > 0)
				out.write(buf, err);
		}
		while (err > 0);

		if (!out)
		{
			exc = CFException(CFException::FILE_OPERATION_FAILED,
				__LINE__, __FILE__, outFilename, "writing");
			throw exc;
		}

        unzCloseCurrentFile(zipFile);
	}
	} //END TRY
	catch (...)
	{
		//always close the file in case of error
		unzCloseCurrentFile(zipFile);
		throw;
	}
}


void Unzip(const string& zipFilename, const string& extractDir) throw (CFException)
{
	unzFile zipFile = unzOpen(zipFilename.c_str());
	if (zipFile==NULL)
	{
		exc = CFException(CFException::FILE_OPEN_FAILED,
			__LINE__, __FILE__, zipFilename, "reading");
		throw exc;
	}

	try {
	unz_global_info globalInfo;
	if (unzGetGlobalInfo(zipFile, &globalInfo) != UNZ_OK)
	{
		exc = CFException(CFException::PKZIP_EXTRACT_FAILED,
			__LINE__, __FILE__, zipFilename, "");
		throw exc;
	}

	for (int i = 0; i < globalInfo.number_entry; i++)
	{
		try {
		//potentially add exception info which was not avail
        ZipExtractCurrentFile(zipFile, extractDir);
		}
		catch (CFException& actExc)
		{
			if (actExc.GetExceptionId() == CFException::PKZIP_EXTRACT_FAILED)
			{
				//set the filename here - it was not known in ZipExtractCurrentFile
				actExc.SetParam1(zipFilename);
			}
			throw actExc;
		}

		if ((i+1) < globalInfo.number_entry)
		{
			if (unzGoToNextFile(zipFile) != UNZ_OK)
			{
				exc = CFException(CFException::PKZIP_EXTRACT_FAILED,
					__LINE__, __FILE__, zipFilename, "");
				throw exc;
			}
		}
	}
	} //END TRY
	catch (...)
	{
		//close zip file in case of error after has been opened
		unzClose(zipFile);
		throw;
	}
	unzClose(zipFile);
}
