//////////////////////////////////////////////////////////////////////
//TargetD64 - C64 archive related conversion tool and emulator frontend
//////////////////////////////////////////////////////////////////////
//Copyright (C) 1998, 1999  Karlheinz Langguth klangguth@netscape.net
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
//COMPILE SWITCHES
// _MSC_VER
// indicates MS compiler
// _DEBUG
// for debug version which activates ASSERT and TRACE
// TD64_MODIFIED
// marks changes in foreign sources to fit into TargetD64
// must be set everywhere because also used for header files

#ifdef _MSC_VER
#pragma warning(disable:4786) //identifier truncation warning
#endif

#include <string>
#include <vector>
#include <iostream>
#include <strstream>
#include <fstream>

using namespace std;

#include "Exception.h"
#include "Image.h"
#include "DiskImage.h"
#include "Archive.h"
#include "MultiPart.h"
#include "C64Archive.h"
#include "HostArchive.h"
#include "ZlibAdapter.h"
#include "Tracing.h"

#include "lharc.h" //LHA interface


void CFHostArchive::ProcessFilesInArchive(void) throw (CFException)
{
	TRBEGIN("CFHostArchive::ProcessFilesInArchive");

	//descent into resolvable file that have been extracted
	//and process them as well
	CFArchive::ProcessFilesInArchive();

	TRreturn;
}


//-------------------------------------------------------------------
//CFLhaArchive
//-------------------------------------------------------------------


bool CFLhaArchive::CheckFileIsOfClass(const string& filename)
	throw (CFException)
{
	string extension, invalid;

	::SplitPathname(filename, invalid, invalid, extension);
	if ((extension == ".lha") || (extension == ".LHA")
			|| (extension == ".lzh") || (extension == ".LZH"))
	{
		return true;	
	}
	return false;
}


void CFLhaArchive::Extract(void) throw (CFException)
{
	TRBEGIN("CFLhaArchive::Extract");
	//if already extracted or mother of all archives do no extraction
	if ((m_state == EXTRACTED) || (m_state == ROOT))//already done
		TRreturn;

	//extract this archive into a new directory to build hirarchy
	m_workdir = NameAndCreateTmpDir();
	
	//extract the archive with builtin extractor
	char *errorText;
	if (Unlha(const_cast<char *>(m_pathname.c_str())
		, const_cast<char *>(m_workdir.c_str())
		, &errorText) != 0)
	{
		exc = CFException(CFException::LHA_EXTRACT_FAILED,
			__LINE__, __FILE__, m_pathname
			, (errorText != NULL) ? errorText : "Not available");
		throw exc;
	}
	TransitionTo(EXTRACTED);
	TRreturn;
}


//-------------------------------------------------------------------
//CFZipArchive
//-------------------------------------------------------------------


bool CFZipArchive::CheckFileIsOfClass(const string& filename)
	throw (CFException)
{
	string extension, invalid;

	::SplitPathname(filename, invalid, invalid, extension);
	if ((extension == ".zip") || (extension == ".ZIP"))
	{
		return true;	
	}
	return false;
}


void CFZipArchive::Extract(void) throw (CFException)
{
	TRBEGIN("CFZipArchive::Extract");
	//if already extracted or mother of all archives do no extraction
	if ((m_state == EXTRACTED) || (m_state == ROOT))//already done
		TRreturn;

	//extract this archive into a new directory to build hirarchy
	m_workdir = NameAndCreateTmpDir();
	
	//extract the archive with builtin extractor
	Unzip(m_pathname, m_workdir);
	TransitionTo(EXTRACTED);
	TRreturn;
}


//-------------------------------------------------------------------
//CFGzipArchive
//-------------------------------------------------------------------


bool CFGzipArchive::CheckFileIsOfClass(const string& filename)
	throw (CFException)
{
	string extension, invalid;

	::SplitPathname(filename, invalid, invalid, extension);
	if ((extension == ".gz") || (extension == ".GZ"))
	{
		return true;	
	}
	return false;
}


void CFGzipArchive::Extract(void) throw (CFException)
{
	TRBEGIN("CFGzipArchive::Extract");
	//if already extracted or mother of all archives do no extraction
	if ((m_state == EXTRACTED) || (m_state == ROOT))//already done
		TRreturn;

	//extract this archive into a new directory to build hirarchy
	m_workdir = NameAndCreateTmpDir();
	
	//extract the archive with builtin extractor
	Gunzip(m_pathname, m_workdir);
	TransitionTo(EXTRACTED);
	TRreturn;
}


//-------------------------------------------------------------------
//CFGenericArchive
//-------------------------------------------------------------------


vector<CFGenericArchive> CFGenericArchive::ms_genericArchives;


bool CFGenericArchive::CheckFileIsOfClass(const string& filename)
	throw (CFException)
{
	string genericExtensionLower = m_genericExtension;
	string genericExtensionUpper = m_genericExtension;
	for (string::size_type i = 0; i < genericExtensionLower.size(); i++)
	{
		genericExtensionLower[i] = (char)tolower(genericExtensionLower[i]);
		genericExtensionUpper[i] = (char)toupper(genericExtensionUpper[i]);
	}

	string extension, invalid;

	::SplitPathname(filename, invalid, invalid, extension);

	//check for lower case version, upper case version and
	//user specified version of extension
	if ((extension == genericExtensionLower)
		|| (extension == genericExtensionUpper)
		|| (extension == m_genericExtension))
	{
		return true;	
	}
	return false;
}


void CFGenericArchive::Extract(void) throw (CFException)
{
	TRBEGIN("CFGenericArchive::Extract");
	//if already extracted or mother of all archives do no extraction
	if ((m_state == EXTRACTED) || (m_state == ROOT))//already done
		TRreturn;

	ASSERT(!m_extractCommand.empty())
			
	//extract this archive into a new directory to build hirarchy
	m_workdir = NameAndCreateTmpDir();
	
	//build the extract command
	//replace placeholders by real pathes
	//%s1 is working dir and %s2 is pathname of archive
	//%s3 is the filename (without dir and extension) of the
	//archive this is used for C1514 extraction to produce
	//the name of a D64 image
	m_extractCommand = ReplacePlaceHoldersByStrings(m_extractCommand,
		m_workdir, m_pathname, m_filename);
	//execute archive extraction
	TRACE(<< "Execute command: " << m_extractCommand);
	if (SystemRedirectOutput(m_extractCommand.c_str()) != 0)
	{
		exc = CFException(CFException::SYSTEM_CALL_FAILED,
			__LINE__, __FILE__, m_extractCommand);
		throw exc;
	}
	TransitionTo(EXTRACTED);
	TRreturn;
}


