//////////////////////////////////////////////////////////////////////
//TargetD64 - C64 archive related conversion tool and emulator frontend
//////////////////////////////////////////////////////////////////////
//Copyright (C) 1998, 1999  Karlheinz Langguth klangguth@netscape.net
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//////////////////////////////////////////////////////////////////////

#ifndef _TAPE_IMAGE_HEADER
#define _TAPE_IMAGE_HEADER

class CFTapeImage
{
private:
	//type of tape entry encoded due to specification
	//only supported is NORMAL
	typedef enum { FREE = 0, NORMAL = 1, WITH_HEADER = 2, SNAPSHOT = 3
		,BLOCK = 4, STREAM = 5 } tEntry;
#ifdef _MSC_VER
#pragma pack(push, l1)
#pragma pack(1)
#endif
	//physical layout of a tape record
	typedef struct
	{
		char description[32]; //DOS tape description + EOF
		unsigned char versionHigh;
		unsigned char versionLow; //should be $0200 but is often $0100
		unsigned char nMaxDirEntriesLow;
		unsigned char nMaxDirEntriesHigh; //maxuimum number of entries in image
		unsigned char nUsedDirEntriesLow; 
		unsigned char nUsedDirEntriesHigh;//number of used entries
		unsigned char unused[2];
		char userDescription[24]; //user description displayed in emu
	} tTapeRecord;

	//physical layout of a file record
	typedef struct
	{
		unsigned char entryType; //entry type of file (see tEntry)
		//should be 1541 file type (e.g. $82 for PRG) - is often just 1
		unsigned char c64DiskFileType;
		unsigned char startAddressLow;
		unsigned char startAddressHigh; //start address in c64 memory
		unsigned char endAddressLow;
		unsigned char endAddressHigh; //end address + 1 in c64 memory
		unsigned char unused1[2];
		unsigned char offsetLowest;
		unsigned char offsetLow;
		unsigned char offsetHigh;
		unsigned char offsetHighest; //offset of file start in tape image
		unsigned char unused2[4];
		char c64Filename[16]; //name of file (padded with $20)
	} tFileRecord;
#ifdef _MSC_VER
#pragma pack(pop, l1)
#endif

public:
	CFTapeImage(void);

public:
	//attach a tape image from filesystem
	//consistency checks, parse directory
	void Attach(const string& filename) throw (CFException);
	//extract all files (except unextractable like STREAM) on tape into
	//directory P1
	//store host filename and CBM filename info into m_fileRecords
	//P1 I: directory to extract to
	void ExtractAllFiles(const string& dirname) throw (CFException);
	//R: vector with info about extracted files (needed for transfer into D64)
	inline const vector<CFExtractedFileInfo>& GetVectorOfExtractedFiles(void) const
		{ return m_extractedFiles; }
private:
	//R: number of used entries in tape
	inline unsigned int GetEntryCount(void) const
	{ return m_tapeRecord.nUsedDirEntriesLow + 256 * m_tapeRecord.nUsedDirEntriesHigh; }
	//R: maximum number of entries in tape
	inline unsigned int GetMaxEntryCount(void) const
	{ return m_tapeRecord.nMaxDirEntriesLow + 256 * m_tapeRecord.nMaxDirEntriesHigh; }
	//R: offset of a file specified by P1 from start of tape file
	//P1 I: file record specifies file to get offset from 
	inline unsigned int GetOffset(const tFileRecord& fileRecord) const
	{ return fileRecord.offsetLowest + fileRecord.offsetLow * 256
		+ fileRecord.offsetHigh * 256 * 256
		+ fileRecord.offsetHighest * 256 * 256 * 256; }
	//extract a file from tape image into host filesystem
	//use P4 as filename if set else derive filename from filename in tape
	//make filename in host filesystem always unique
	//P1 I: the file record entry describing the file
	//P2 I: directory to extract to
	//P3 O: additional file info from file record
	//P3 I: host filesystem filename ("" means not set)
	void ExtractFile(const tFileRecord& entry
		, const string& dirname
		, CFExtractedFileInfo& extractInfo
		, const string& filename = "") const throw (CFException);

private:
	//fileinfo of all extracted files of this tape
	vector<CFExtractedFileInfo> m_extractedFiles;
	//filename of this tape
	string m_filename;
	//opened with the tape image file
	mutable ifstream m_inStream; //any operation affects the stream (even read)
	//tape record of tape image, read in Attach()
	tTapeRecord m_tapeRecord;
	//all file records in use, read in Attach()
	vector<tFileRecord> m_fileRecords;
};

#endif

